/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ClassLister.java
 * Copyright (C) 2007 University of Waikato, Hamilton, New Zealand
 */

package adams.core;

import java.util.Enumeration;
import java.util.Vector;
import java.util.regex.Pattern;

import adams.env.ClassListerBlacklistDefinition;
import adams.env.ClassListerDefinition;
import adams.env.Environment;

/**
 * Determines the classnames of superclasses that are to be displayed in
 * the GUI for instance.
 * <p/>
 * <b>IMPORTANT NOTE:</b> Due to <a href="http://geekexplains.blogspot.com/2008/07/what-is-reentrant-synchronization-in.html" target="_blank">reentrant threads</a>,
 * the <code>getSingleton()</code> method is not allowed to be called from
 * <code>static {...}</code> blocks in classes that are managed by the
 * ClassLister class (and therefore the ClassLocator class). Since the
 * ClassLocator class loads classes into the JVM, the <code>static {...}</code>
 * block of these classes gets executed and the ClassLister gets initialized
 * once again. In this case, the previous singleton will most likely get
 * overwritten.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class ClassLister {

  /** the name of the props file. */
  public final static String FILENAME = "ClassLister.props";

  /** the name of the props file. */
  public final static String BLACKLIST = "ClassLister.blacklist";

  /** the properties (superclass/packages). */
  protected Properties m_Packages;

  /** the properties (superclass/classes). */
  protected Properties m_Classes;

  /** the singleton. */
  protected static ClassLister m_Singleton;

  /**
   * Initializes the classlister.
   */
  private ClassLister() {
    initialize();
  }

  /**
   * loads the props file and interpretes it.
   */
  private void initialize() {
    Enumeration			enm;
    String			superclass;
    String[]			packages;
    Vector<String>		classes;
    Properties			blacklist;
    String[]			patterns;
    int				i;
    Pattern			p;

    try {
      m_Packages = Environment.getInstance().read(ClassListerDefinition.KEY);
      m_Classes  = new Properties();
      blacklist  = Environment.getInstance().read(ClassListerBlacklistDefinition.KEY);

      enm = m_Packages.propertyNames();
      while (enm.hasMoreElements()) {
	superclass = (String) enm.nextElement();
	packages   = m_Packages.getString(superclass).replaceAll(" ", "").split(",");
	classes    = ClassLocator.find(superclass, packages);
	// remove blacklisted classes
	if (blacklist.hasKey(superclass)) {
	  try {
	    patterns = blacklist.getString(superclass).replaceAll(" ", "").split(",");
	    for (String pattern: patterns) {
	      p = Pattern.compile(pattern);
	      i = 0;
	      while (i < classes.size()) {
		if (p.matcher(classes.get(i)).matches())
		  classes.remove(i);
		else
		  i++;
	      }
	    }
	  }
	  catch (Exception ex) {
	    System.err.println("Failed to blacklist classes for superclass '" +  superclass + "':");
	    ex.printStackTrace();
	  }
	}
	// create class list
	m_Classes.setProperty(superclass, Utils.flatten(classes, ","));
      }
    }
    catch (Exception e) {
      System.err.println("Failed to determine packages/classes:");
      e.printStackTrace();
      m_Packages = new Properties();
    }
  }

  /**
   * Returns all the classnames that were found for this superclass.
   *
   * @param superclass	the superclass to return the derived classes for
   * @return		the classnames of the derived classes
   */
  public String[] getClassnames(Class superclass) {
    return getClassnames(superclass.getName());
  }

  /**
   * Returns all the classnames that were found for this superclass.
   *
   * @param superclass	the superclass to return the derived classes for
   * @return		the classnames of the derived classes
   */
  public String[] getClassnames(String superclass) {
    String	classes;

    classes = m_Classes.getString(superclass);
    if ((classes == null) || (classes.length() == 0))
      return new String[0];
    else
      return classes.split(",");
  }

  /**
   * Returns the superclass-found_classes relation.
   *
   * @return		the properties object listing all found classes
   */
  public Properties getClasses() {
    return m_Classes;
  }

  /**
   * Returns all the packages that were found for this superclass.
   *
   * @param superclass	the superclass to return the packages for
   * @return		the packages
   */
  public String[] getPackages(Class superclass) {
    return getPackages(superclass.getName());
  }

  /**
   * Returns all the packages that were found for this superclass.
   *
   * @param superclass	the superclass to return the packages for
   * @return		the packages
   */
  public String[] getPackages(String superclass) {
    String	packages;

    packages = m_Packages.getString(superclass);
    if ((packages == null) || (packages.length() == 0))
      return new String[0];
    else
      return packages.split(",");
  }

  /**
   * Returns the superclass-packages relation.
   *
   * @return		the properties object listing the packages
   */
  public Properties getPackages() {
    return m_Packages;
  }

  /**
   * Only prints the generated props file with all the classnames, based on
   * the package names for the individual packages.
   *
   * @return		the props file with the classnames
   */
  public String toString() {
    return m_Classes.toString();
  }

  /**
   * Returns the singleton instance of the class lister.
   *
   * @return		the singleton
   */
  public static synchronized ClassLister getSingleton() {
    if (m_Singleton == null)
      m_Singleton = new ClassLister();

    return m_Singleton;
  }

  /**
   * Only for testing.
   *
   * @param args	ignored
   */
  public static void main(String[] args) {
    System.out.println(new ClassLister());
  }
}
