/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * ConditionalTrigger.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.control;

import adams.flow.condition.bool.AbstractBooleanCondition;
import adams.flow.condition.bool.Counting;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Executes the tee-actor whenever a token gets passed through. In contrast to the Tee actor, it doesn't feed the tee-actor with the current token.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: ConditionalTrigger
 * </pre>
 * 
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded 
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 * 
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 * 
 * <pre>-tee &lt;adams.flow.core.AbstractActor&gt; [-tee ...] (property: actors)
 * &nbsp;&nbsp;&nbsp;The actors to siphon-off the tokens to.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-condition &lt;adams.flow.condition.bool.AbstractBooleanCondition&gt; (property: condition)
 * &nbsp;&nbsp;&nbsp;The boolean condition to evaluate.
 * &nbsp;&nbsp;&nbsp;default: adams.flow.condition.bool.Counting
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4838 $
 */
public class ConditionalTrigger
  extends Trigger {

  /** for serialization. */
  private static final long serialVersionUID = 4568526647666412543L;
  
  /** the condition to use. */
  protected AbstractBooleanCondition m_Condition;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Executes the tee-actor whenever a token gets passed through. In "
      + "contrast to the Tee actor, it doesn't feed the tee-actor with the "
      + "current token.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "condition", "condition",
	    new Counting());
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    return m_Condition.getQuickInfo();
  }

  /**
   * Sets the minimum number of counts before activating.
   *
   * @param value	the minimum (-1 to disable)
   */
  public void setCondition(AbstractBooleanCondition value) {
    m_Condition = value;
    reset();
  }

  /**
   * Returns the minimum of counts before activating.
   *
   * @return		the minimum (-1 if disabled)
   */
  public AbstractBooleanCondition getCondition() {
    return m_Condition;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String conditionTipText() {
    return "The boolean condition to evaluate.";
  }

  /**
   * Processes the token.
   *
   * @param token	used for evaluating the condition
   * @return		an optional error message, null if everything OK
   */
  protected String processInput(Token token) {
    String	result;
    String	msg;

    result = null;
    
    try {
      if (m_Condition.evaluate(this, token))
	result = super.processInput(token);
    }
    catch (Exception e) {
      msg = "Failed to trigger: ";
      getSystemErr().println(msg);
      getSystemErr().printStackTrace(e);
      result = msg + e;
    }

    return result;
  }
}
