/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * GlobalActorScreenshot.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.control;

import java.io.File;
import java.util.HashSet;
import java.util.Hashtable;

import javax.swing.SwingUtilities;

import adams.core.VariableName;
import adams.core.Variables;
import adams.core.io.FileUtils;
import adams.core.io.PlaceholderDirectory;
import adams.core.io.PlaceholderFile;
import adams.flow.core.AbstractActor;
import adams.flow.core.ControlActor;
import adams.flow.core.GlobalActorHelper;
import adams.flow.core.GlobalActorReference;
import adams.flow.core.InputConsumer;
import adams.flow.core.OutputProducer;
import adams.flow.core.Token;
import adams.flow.core.Unknown;
import adams.flow.sink.ComponentSupplier;
import adams.gui.print.JComponentWriter;
import adams.gui.print.NullWriter;

/**
 <!-- globalinfo-start -->
 * Takes a screenshot of a global actor whenever a token passes through.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: GlobalActorScreenshot
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-global &lt;adams.flow.core.GlobalActorReference&gt; (property: globalName)
 * &nbsp;&nbsp;&nbsp;The name of the global actor to use.
 * &nbsp;&nbsp;&nbsp;default: unknown
 * </pre>
 *
 * <pre>-output-dir &lt;adams.core.io.PlaceholderDirectory&gt; (property: outputDir)
 * &nbsp;&nbsp;&nbsp;The output directory.
 * &nbsp;&nbsp;&nbsp;default: ${CWD}
 * </pre>
 *
 * <pre>-filename-prefix &lt;java.lang.String&gt; (property: filenamePrefix)
 * &nbsp;&nbsp;&nbsp;The prefix for the filename in case of auto-generation (no path, just name
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: screenshot
 * </pre>
 *
 * <pre>-filename-var &lt;adams.core.VariableName&gt; (property: filenameVariable)
 * &nbsp;&nbsp;&nbsp;The variable to use for the filename instead of the auto-generated one.
 * &nbsp;&nbsp;&nbsp;default: variable
 * </pre>
 *
 * <pre>-writer &lt;adams.gui.print.JComponentWriter&gt; (property: writer)
 * &nbsp;&nbsp;&nbsp;The writer to use for generating the graphics output.
 * &nbsp;&nbsp;&nbsp;default: adams.gui.print.NullWriter
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class GlobalActorScreenshot
  extends AbstractActor
  implements ControlActor, InputConsumer, OutputProducer {

  /** for serialization. */
  private static final long serialVersionUID = -7346814880631564292L;

  /** the key for storing the current counter in the backup. */
  public final static String BACKUP_COUNTER = "counter";

  /** the key for storing the input token in the backup. */
  public final static String BACKUP_INPUT = "input";

  /** the global name. */
  protected GlobalActorReference m_GlobalName;

  /** the input token. */
  protected Token m_InputToken;

  /** the output token. */
  protected Token m_OutputToken;

  /** the writer to use. */
  protected JComponentWriter m_Writer;

  /** the dir to write to. */
  protected PlaceholderDirectory m_OutputDir;

  /** the prefix for the auto-generated filename. */
  protected String m_FilenamePrefix;

  /** the variable to use as filename. */
  protected VariableName m_FilenameVariable;

  /** the helper class. */
  protected GlobalActorHelper m_Helper;

  /** the global actor. */
  protected AbstractActor m_GlobalActor;

  /** the counter for the screenshots. */
  protected int m_Counter;

  /** for storing any exceptions while trying to create a screenshot. */
  protected String m_ScreenshotResult;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Takes a screenshot of a global actor whenever a token passes through.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "global", "globalName",
	    new GlobalActorReference("unknown"));

    m_OptionManager.add(
	    "output-dir", "outputDir",
	    new PlaceholderDirectory("."));

    m_OptionManager.add(
	    "filename-prefix", "filenamePrefix",
	    "screenshot");

    m_OptionManager.add(
	    "filename-var", "filenameVariable",
	    new VariableName());

    m_OptionManager.add(
	    "writer", "writer",
	    new NullWriter());
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Helper = new GlobalActorHelper();
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_Counter     = 0;
    m_InputToken  = null;
    m_OutputToken = null;
  }

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_COUNTER);
    pruneBackup(BACKUP_INPUT);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    result.put(BACKUP_COUNTER, m_Counter);

    if (m_InputToken != null)
      result.put(BACKUP_INPUT, m_InputToken);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_COUNTER)) {
      m_Counter = (Integer) state.get(BACKUP_COUNTER);
      state.remove(BACKUP_COUNTER);
    }

    if (state.containsKey(BACKUP_INPUT)) {
      m_InputToken = (Token) state.get(BACKUP_INPUT);
      state.remove(BACKUP_INPUT);
    }

    super.restoreState(state);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;
    String	output;
    String	prefix;
    String	result;

    variable = getOptionManager().getVariableForProperty("filenameVariable");

    if (variable != null) {
      result = variable;
    }
    else if ((m_FilenameVariable != null) && (m_FilenameVariable.getValue().length() > 0)) {
      result = Variables.START + m_FilenameVariable + Variables.END;
    }
    else {
      variable = getOptionManager().getVariableForProperty("output");
      if (variable == null)
	output = m_OutputDir.toString();
      else
	output = variable;

      variable = getOptionManager().getVariableForProperty("filenamePrefix");
      if (variable == null)
	prefix = m_FilenamePrefix.toString();
      else
	prefix = variable;

      if (!(getWriter() instanceof NullWriter)) {
	result = output + File.separator + prefix + "XYZ";
        if (getWriter().getExtensions().length > 0)
          result += getWriter().getExtensions()[0];
      }
      else {
	result = output;
      }
    }

    return result;
  }

  /**
   * Sets the name of the global actor to use.
   *
   * @param value 	the global name
   */
  public void setGlobalName(GlobalActorReference value) {
    m_GlobalName = value;
    reset();
  }

  /**
   * Returns the name of the global actor in use.
   *
   * @return 		the global name
   */
  public GlobalActorReference getGlobalName() {
    return m_GlobalName;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String globalNameTipText() {
    return "The name of the global actor to use.";
  }

  /**
   * Sets the output directory.
   *
   * @param value 	the directory
   */
  public void setOutputDir(PlaceholderDirectory value) {
    m_OutputDir = value;
    reset();
  }

  /**
   * Returns the output directory.
   *
   * @return 		the directory
   */
  public PlaceholderDirectory getOutputDir() {
    return m_OutputDir;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputDirTipText() {
    return "The output directory.";
  }

  /**
   * Sets the prefix for the filename in case of auto-generation.
   *
   * @param value	the prefix (just name, no path)
   */
  public void setFilenamePrefix(String value) {
    m_FilenamePrefix = value;
    reset();
  }

  /**
   * Returns the prefix for the filename in case of auto-generation.
   *
   * @return		the panel provider in use
   */
  public String getFilenamePrefix() {
    return m_FilenamePrefix;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String filenamePrefixTipText() {
    return "The prefix for the filename in case of auto-generation (no path, just name).";
  }

  /**
   * Sets the variable to use for generating the filename instead of the
   * auto-generated one.
   *
   * @param value	the variable name (without the @{ and })
   */
  public void setFilenameVariable(VariableName value) {
    m_FilenameVariable = value;
    reset();
  }

  /**
   * Returns the variable to use for generating the filename instead of the
   * auto-generated one.
   *
   * @return		the panel provider in use
   */
  public VariableName getFilenameVariable() {
    return m_FilenameVariable;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String filenameVariableTipText() {
    return "The variable to use for the filename instead of the auto-generated one.";
  }

  /**
   * Sets the writer.
   *
   * @param value 	the writer
   */
  public void setWriter(JComponentWriter value) {
    m_Writer = value;
    reset();
  }

  /**
   * Returns the writer.
   *
   * @return 		the writer
   */
  public JComponentWriter getWriter() {
    return m_Writer;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String writerTipText() {
    return "The writer to use for generating the graphics output.";
  }

  /**
   * Tries to find the global actor referenced by its global name.
   *
   * @return		the global actor or null if not found
   */
  protected AbstractActor findGlobalActor() {
    return m_Helper.findGlobalActorRecursive(this, getGlobalName());
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String		result;
    HashSet<String>	variables;

    result = super.setUp();

    if (result == null) {
      m_GlobalActor = findGlobalActor();
      if (m_GlobalActor == null) {
        result = "Couldn't find global actor '" + getGlobalName() + "'!";
      }
      else {
	if (!(m_GlobalActor instanceof ComponentSupplier)) {
	  result = "Global actor '" + getGlobalName() + "' is not a " + ComponentSupplier.class.getName() + "!";
	}
	else {
	  variables = findVariables(m_GlobalActor);
	  m_DetectedVariables.addAll(variables);
	  if (m_DetectedVariables.size() > 0)
	    getVariables().addVariableChangeListener(this);
	}
      }
    }

    return result;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Unknown.class};
  }

  /**
   * Does nothing.
   *
   * @param token	the token to accept and process
   */
  public void input(Token token) {
    m_InputToken = token;
  }

  /**
   * Generates the filename for the output.
   *
   * @return		the file
   */
  protected PlaceholderFile generateFilename() {
    PlaceholderFile	result;

    m_Counter++;

    if (getVariables().has(m_FilenameVariable.getValue())) {
      result = new PlaceholderFile(
	  getVariables().get(m_FilenameVariable.getValue()));
    }
    else {
      result = new PlaceholderFile(
	  m_OutputDir.getAbsolutePath() + File.separator
	  + FileUtils.createFilename(m_FilenamePrefix + m_Counter + m_Writer.getExtensions()[0], "_"));
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String			result;
    final PlaceholderFile	filename;
    Runnable			run;

    m_ScreenshotResult = null;
    filename           = generateFilename();

    run = new Runnable() {
      public void run() {
	synchronized(m_GlobalActor) {
	  debug("Saving to: " + filename);
	  m_Writer.setComponent(((ComponentSupplier) m_GlobalActor).supplyComponent());
	  m_Writer.setFile(filename);
	  try {
	    m_Writer.toOutput();
	  }
	  catch (Exception e) {
	    String msg = "Failed to generate screenshot ('" + filename + "'): ";
	    getSystemErr().println(msg);
	    getSystemErr().printStackTrace(e);
	    m_ScreenshotResult = msg + e;
	  }
	  m_Writer.setComponent(null);
	  synchronized(m_Self) {
	    try {
	      m_Self.notifyAll();
	    }
	    catch (Exception e) {
	      e.printStackTrace();
	    }
	  }
	}
      }
    };

    synchronized(m_Self) {
      SwingUtilities.invokeLater(run);
      try {
	m_Self.wait();
      }
      catch (Exception e) {
	e.printStackTrace();
      }
    }

    result             = m_ScreenshotResult;
    m_ScreenshotResult = null;
    m_OutputToken      = m_InputToken;

    return result;
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.core.Unknown.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Unknown.class};
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    result        = m_OutputToken;
    m_OutputToken = null;

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   * <p/>
   * The method is not allowed allowed to return "true" before the
   * actor has been executed. For actors that return an infinite
   * number of tokens, the m_Executed flag can be returned.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return (m_OutputToken != null);
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void wrapUp() {
    m_InputToken  = null;
    m_OutputToken = null;

    super.wrapUp();
  }
}
