/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractActorProcessor.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.processor;

import adams.core.ClassLister;
import adams.core.ShallowCopySupporter;
import adams.core.option.AbstractOptionConsumer;
import adams.core.option.ArrayConsumer;
import adams.core.option.AbstractOptionHandler;
import adams.core.option.OptionUtils;
import adams.flow.core.AbstractActor;

/**
 * Abstract base class for schemes that process a flow/actor in some fashion.
 *
 * Derived classes only have to override the <code>processActor()</code>
 * method. The <code>reset()</code> method can be used to reset an
 * algorithms internal state, e.g., after setting options, which invalidate
 * the previously generated data.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public abstract class AbstractActorProcessor
  extends AbstractOptionHandler
  implements Comparable, ShallowCopySupporter<AbstractActorProcessor> {

  /** for serialization. */
  private static final long serialVersionUID = 3610605513320220903L;

  /**
   * Resets the filter.
   * Derived classes must call this method in set-methods of parameters to
   * assure the invalidation of previously generated data.
   */
  public void reset() {
    super.reset();
  }

  /**
   * Processes the actor.
   *
   * @param actor	the actor to process
   */
  public void process(AbstractActor actor) {
    checkData(actor);
    if (actor instanceof ModifyingProcessor)
      processActor(actor.shallowCopy());
    else
      processActor(actor);
  }

  /**
   * The default implementation only checks whether there is any actor set.
   *
   * @param actor	the actor to process
   */
  protected void checkData(AbstractActor actor) {
    if (actor == null)
      throw new IllegalStateException("No actor provided!");
  }

  /**
   * Performs the actual processing.
   *
   * @param actor	the actor to process (is a copy of original for
   * 			processors implementing ModifyingProcessor)
   * @see		ModifyingProcessor
   */
  protected abstract void processActor(AbstractActor actor);

  /**
   * Compares this object with the specified object for order.  Returns a
   * negative integer, zero, or a positive integer as this object is less
   * than, equal to, or greater than the specified object.
   * <p/>
   * Only compares the commandlines of the two objects.
   *
   * @param o 	the object to be compared.
   * @return  	a negative integer, zero, or a positive integer as this object
   *		is less than, equal to, or greater than the specified object.
   *
   * @throws ClassCastException 	if the specified object's type prevents it
   *         				from being compared to this object.
   */
  public int compareTo(Object o) {
    if (o == null)
      return 1;

    return OptionUtils.getCommandLine(this).compareTo(OptionUtils.getCommandLine(o));
  }

  /**
   * Returns whether the two objects are the same.
   * <p/>
   * Only compares the commandlines of the two objects.
   *
   * @param o	the object to be compared
   * @return	true if the object is the same as this one
   */
  public boolean equals(Object o) {
    return (compareTo(o) == 0);
  }

  /**
   * Returns a shallow copy of itself, i.e., based on the commandline options.
   *
   * @return		the shallow copy
   */
  public AbstractActorProcessor shallowCopy() {
    return shallowCopy(false);
  }

  /**
   * Returns a shallow copy of itself, i.e., based on the commandline options.
   *
   * @param expand	whether to expand variables to their current values
   * @return		the shallow copy
   */
  public AbstractActorProcessor shallowCopy(boolean expand) {
    return (AbstractActorProcessor) OptionUtils.shallowCopy(this, expand);
  }

  /**
   * Returns a list with classnames of processors.
   *
   * @return		the processor classnames
   */
  public static String[] getProcessors() {
    return ClassLister.getSingleton().getClassnames(AbstractActorProcessor.class);
  }

  /**
   * Instantiates the processor with the given options.
   *
   * @param classname	the classname of the processor to instantiate
   * @param options	the options for the processor
   * @return		the instantiated processor or null if an error occurred
   */
  public static AbstractActorProcessor forName(String classname, String[] options) {
    AbstractActorProcessor	result;

    try {
      result = (AbstractActorProcessor) OptionUtils.forName(AbstractActorProcessor.class, classname, options);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Instantiates the filter from the given commandline
   * (i.e., classname and optional options).
   *
   * @param cmdline	the classname (and optional options) of the
   * 			filter to instantiate
   * @return		the instantiated filter
   * 			or null if an error occurred
   */
  public static AbstractActorProcessor forCommandLine(String cmdline) {
    return (AbstractActorProcessor) AbstractOptionConsumer.fromString(ArrayConsumer.class, cmdline);
  }
}
