/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * CheckVariableUsage.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.processor;

import java.awt.Component;
import java.awt.Dimension;
import java.util.Collections;
import java.util.Set;
import java.util.Vector;

import adams.core.NamedCounter;
import adams.core.Utils;
import adams.core.VariableName;
import adams.core.option.AbstractArgumentOption;
import adams.core.option.BooleanOption;
import adams.core.option.ClassOption;
import adams.core.option.OptionTraverser;
import adams.flow.core.AbstractActor;
import adams.gui.core.TextEditorPanel;

/**
 <!-- globalinfo-start -->
 * Performs a 'soft' check whether variables in use are actually set somewhere in the flow.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4858 $
 */
public class CheckVariableUsage
  extends AbstractActorProcessor 
  implements GraphicalOutputProducingProcessor {

  /** for serialization. */
  private static final long serialVersionUID = 737084782888325641L;

  /** the usage counter. */
  protected NamedCounter m_UsageCount;

  /** the set counter. */
  protected NamedCounter m_SetCount;
  
  /** the warnings that were produced. */
  protected String m_Warnings;
  
  /**
   * Returns a string describing the object.
   *
   * @return 		a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Performs a 'soft' check whether variables in use are actually set "
	+ "somewhere in the flow.";
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();
    
    m_UsageCount = new NamedCounter();
    m_SetCount   = new NamedCounter();
    m_Warnings   = null;
  }
  
  /**
   * Performs the actual processing.
   *
   * @param actor	the actor to process
   */
  protected void processActor(AbstractActor actor) {
    Set<String>		used;
    Set<String>		set;
    Vector<String>	vars;
    
    m_UsageCount.clear();
    m_SetCount.clear();
    m_Warnings = null;
    
    actor.getOptionManager().traverse(new OptionTraverser() {
      protected void incrementSetCount(VariableName var) {
	m_SetCount.next(var.getValue());
      }
      public void handleClassOption(ClassOption option) {
	if (option.isVariableAttached())
	  m_UsageCount.next(option.getVariableName());
      }
      public void handleBooleanOption(BooleanOption option) {
	// ignored
      }
      public void handleArgumentOption(AbstractArgumentOption option) {
	Object obj = null;
	// variable attached?
	if (option.isVariableAttached()) {
	  m_UsageCount.next(option.getVariableName());
	}
	else {
	  // standalone
	  if ((option.getOptionHandler() instanceof adams.flow.standalone.SetVariable) && (option.getBaseClass() == VariableName.class))
	    obj = option.getCurrentValue();
	  // transformer
	  if ((option.getOptionHandler() instanceof adams.flow.transformer.SetVariable) && (option.getBaseClass() == VariableName.class))
	    obj = option.getCurrentValue();
	  // increment counters
	  if (obj != null)
	    incrementSetCount((VariableName) obj);
	}
      }
      public boolean canRecurse(Class cls) {
        return true;
      }
    });

    used = m_UsageCount.nameSet();
    set  = m_SetCount.nameSet();
    used.removeAll(set);
    if (used.size() > 0) {
      vars = new Vector<String>(used);
      Collections.sort(vars);
      m_Warnings = "The following variables were never set:\n" + Utils.flatten(vars, "\n");
    }
  }

  /**
   * Returns the warnings, if any, on variables that might never get set.
   * 
   * @return		the warnings
   */
  public String getWarnings() {
    return m_Warnings;
  }
  
  /**
   * Returns whether graphical output was generated.
   *
   * @return		true if graphical output was generated
   */
  public boolean hasGraphicalOutput() {
    return (m_Warnings != null);
  }

  /**
   * Returns the graphical output that was generated.
   *
   * @return		the graphical output
   */
  public Component getGraphicalOutput() {
    TextEditorPanel	result;
    
    result = new TextEditorPanel();
    result.setPreferredSize(new Dimension(400, 300));
    result.setEditable(false);
    if (m_Warnings != null)
      result.setContent(m_Warnings);
    
    return result;
  }
}
