/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * BaseDirectoryChooser.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.chooser;

import java.io.File;

import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileSystemView;

import adams.core.Placeholders;
import adams.core.io.PlaceholderFile;
import adams.env.Environment;

import com.l2fprod.common.swing.JDirectoryChooser;

/**
 * Extended version of the com.l2fprod.common.swing.JDirectoryChooser to
 * handle PlaceholderFile objects.
 *
 * @author FracPete (fracpete at waikat dot ac dot nz)
 */
public class BaseDirectoryChooser extends JDirectoryChooser {

  /** for serialization. */
  private static final long serialVersionUID = -7252242971482953986L;

  /**
   * Creates a JDirectoryChooser pointing to the user's home directory.
   */
  public BaseDirectoryChooser() {
    super();
    initialize();
  }

  /**
   * Creates a BaseDirectoryChooser using the given File as the path.
   *
   * @param currentDirectory	the directory to start in
   */
  public BaseDirectoryChooser(File currentDirectory) {
    super(currentDirectory.getAbsoluteFile());
    initialize();
  }

  /**
   * Creates a BaseDirectoryChooser using the given current directory and
   * FileSystemView.
   *
   * @param currentDirectory	the directory to start in
   * @param fsv			the view to use
   */
  public BaseDirectoryChooser(File currentDirectory, FileSystemView fsv) {
    super(currentDirectory.getAbsoluteFile(), fsv);
    initialize();
  }

  /**
   * Creates a BaseDirectoryChooser using the given FileSystemView.
   *
   * @param fsv			the view to use
   */
  public BaseDirectoryChooser(FileSystemView fsv) {
    super(fsv);
    initialize();
  }

  /**
   * Creates a BaseDirectoryChooser using the given path.
   *
   * @param currentDirectoryPath	the directory to start in
   */
  public BaseDirectoryChooser(String currentDirectoryPath) {
    super(new PlaceholderFile(currentDirectoryPath).getAbsolutePath());
    initialize();
  }

  /**
   * Creates a BaseDirectoryChooser using the given path and FileSystemView.
   *
   * @param currentDirectoryPath	the directory to start in
   * @param fsv				the view to use
   */
  public BaseDirectoryChooser(String currentDirectoryPath, FileSystemView fsv) {
    super(new PlaceholderFile(currentDirectoryPath).getAbsolutePath(), fsv);
    initialize();
  }

  /**
   * For initializing some stuff.
   * <p/>
   * Default implementation does nothing.
   */
  protected void initialize() {
  }

  /**
   * Does nothing.
   *
   * @param filter	ignored
   */
  public void addChoosableFileFilter(FileFilter filter) {
  }

  /**
   * Sets the selected file. If the file's parent directory is
   * not the current directory, changes the current directory
   * to be the file's parent directory.
   *
   * @beaninfo
   *   preferred: true
   *       bound: true
   *
   * @see #getSelectedFile
   *
   * @param file the selected file
   */
  public void setSelectedFile(File file) {
    File	selFile;

    selFile = null;

    if (file != null)
      selFile = new File(file.getAbsolutePath());

    super.setSelectedFile(selFile);
  }

  /**
   * Returns the selected file. This can be set either by the
   * programmer via <code>setFile</code> or by a user action, such as
   * either typing the filename into the UI or selecting the
   * file from a list in the UI.
   *
   * @return the selected file
   */
  public File getSelectedFile() {
    File	result;
    int		pos;

    result = super.getSelectedFile();
    if (result != null) {
      // Unfortunately, JFileChooser automatically adds the current directory
      // to the filename. In case we have a placeholder in the name, we just
      // remove the part of the string preceding the placeholder start
      if ((pos = result.getPath().lastIndexOf(Placeholders.PLACEHOLDER_START)) > -1)
	result = new PlaceholderFile(result.getPath().substring(pos));
      else
	result = new PlaceholderFile(result);
    }

    return result;
  }

  /**
   * Sets the list of selected files if the file chooser is
   * set to allow multiple selection.
   *
   * @param selectedFiles	the files to select initially
   * @beaninfo
   *       bound: true
   * description: The list of selected files if the chooser is in multiple selection mode.
   */
  public void setSelectedFiles(File[] selectedFiles) {
    File[]	files;
    int		i;

    files = null;
    if (selectedFiles != null) {
      files = new File[selectedFiles.length];
      for (i = 0; i < selectedFiles.length; i++)
	files[i] = new File(selectedFiles[i].getAbsolutePath());
    }

    super.setSelectedFiles(files);
  }

  /**
   * Returns a list of selected files if the file chooser is
   * set to allow multiple selection.
   *
   * @return the selected file
   */
  public File[] getSelectedFiles() {
    File[]	result;
    int		i;
    int		pos;

    result = super.getSelectedFiles();
    for (i = 0; i < result.length; i++) {
      // Unfortunately, JFileChooser automatically adds the current directory
      // to the filename. In case we have a placeholder in the name, we just
      // remove the part of the string preceding the placeholder start
      if ((pos = result[i].getPath().indexOf(Placeholders.PLACEHOLDER_START)) > -1)
	result[i] = new PlaceholderFile(result[i].getPath().substring(pos));
      else
	result[i] = new PlaceholderFile(result[i]);
    }

    return result;
  }

  /**
   * Returns the current directory.
   *
   * @return the current directory, as PlaceholderFile
   * @see #setCurrentDirectory
   */
  public File getCurrentDirectory() {
    File	current;

    current = super.getCurrentDirectory();
    if (current == null)
      return null;
    else
      return new PlaceholderFile(current);
  }

  /**
   * Sets the current directory. Passing in <code>null</code> sets the
   * file chooser to point to the user's default directory.
   * This default depends on the operating system. It is
   * typically the "My Documents" folder on Windows, and the user's
   * home directory on Unix.
   *
   * If the file passed in as <code>currentDirectory</code> is not a
   * directory, the parent of the file will be used as the currentDirectory.
   * If the parent is not traversable, then it will walk up the parent tree
   * until it finds a traversable directory, or hits the root of the
   * file system.
   *
   * @param dir the current directory to point to
   * @see #getCurrentDirectory
   */
  public void setCurrentDirectory(File dir) {
    if (dir == null)
      super.setCurrentDirectory(null);
    else
      super.setCurrentDirectory(new PlaceholderFile(dir).getAbsoluteFile());
  }

  /**
   * For testing only.
   *
   * @param args	ignored
   */
  public static void main(String[] args) {
    Environment.setEnvironmentClass(Environment.class);
    BaseDirectoryChooser chooser = new BaseDirectoryChooser();
    chooser.setCurrentDirectory(new PlaceholderFile("${TMP}"));
    if (chooser.showOpenDialog(null) == BaseDirectoryChooser.APPROVE_OPTION)
      System.out.println(chooser.getSelectedFile());
  }
}
