/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * ImagePanel.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.visualization.image;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;

import javax.imageio.ImageIO;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;

import adams.core.Properties;
import adams.core.StatusMessageHandler;
import adams.core.Utils;
import adams.core.io.PlaceholderFile;
import adams.gui.chooser.ImageFileChooser;
import adams.gui.core.BasePanel;
import adams.gui.core.BaseScrollPane;
import adams.gui.core.BaseSplitPane;
import adams.gui.core.BaseStatusBar;
import adams.gui.core.BaseTable;
import adams.gui.core.CustomPopupMenuProvider;
import adams.gui.core.GUIHelper;
import adams.gui.core.MouseUtils;
import adams.gui.core.PropertiesTableModel;
import adams.gui.core.UndoPanel;
import adams.gui.event.UndoEvent;
import adams.gui.print.PrintMouseListener;

/**
 * For displaying a single image.
 * <p/>
 * The scroll wheel allows zooming in/out. Mouse-wheel clicking sets scale
 * back to 100%.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4844 $
 */
public class ImagePanel
  extends UndoPanel
  implements StatusMessageHandler {

  /** for serialization. */
  private static final long serialVersionUID = -3102446345758890249L;

  /**
   * The panel used for painting.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4844 $
   */
  public static class PaintPanel
    extends BasePanel {

    /** for serialization. */
    private static final long serialVersionUID = 184259023085417961L;

    /** the owning image panel. */
    protected ImagePanel m_Owner;

    /** the scaling factor. */
    protected double m_Scale;

    /** the current image. */
    protected BufferedImage m_CurrentImage;

    /** the mouse listener. */
    protected PrintMouseListener m_PrintMouseListener;

    /** the image overlays. */
    protected HashSet<ImageOverlay> m_ImageOverlays;
    
    /** whether to use a custom popup menu provider. */
    protected CustomPopupMenuProvider m_CustomPopupMenuProvider;

    /**
     * Initializes the panel.
     *
     * @param owner	the image panel this paint panel belongs to
     */
    public PaintPanel(ImagePanel owner) {
      super();

      m_Owner = owner;
    }

    /**
     * Initializes members.
     */
    protected void initialize() {
      super.initialize();

      m_CurrentImage            = null;
      m_Scale                   = 1.0;
      m_ImageOverlays           = new HashSet<ImageOverlay>();
      m_CustomPopupMenuProvider = null;
    }

    /**
     * Initializes the widgets.
     */
    protected void initGUI() {
      super.initGUI();

      addMouseMotionListener(new MouseMotionAdapter() {
	public void mouseMoved(MouseEvent e) {
	  updateStatus(e.getPoint());
	  super.mouseMoved(e);
	}
      });

      addMouseWheelListener(new MouseWheelListener() {
        public void mouseWheelMoved(MouseWheelEvent e) {
          int rotation = e.getWheelRotation();
          double scale = getScale();
          if (rotation < 0)
            scale = scale * Math.pow(1.2, -rotation);
          else
            scale = scale / Math.pow(1.2, rotation);
          setScale(scale);
          updateStatus();
        }
      });

      addMouseListener(new MouseAdapter() {
	public void mouseClicked(MouseEvent e) {
	  if (MouseUtils.isMiddleClick(e)) {
	    setScale(1.0);
	    updateStatus();
	    e.consume();
	  }
	  else if (MouseUtils.isRightClick(e)) {
	    showPopup(e);
	    e.consume();
	  }
	  else {
	    super.mouseClicked(e);
	  }
	}
      });

      m_PrintMouseListener = new PrintMouseListener(this);
    }

    /**
     * Updates the status bar.
     */
    protected void updateStatus() {
      Point	pos;

      pos = getMousePosition();
      if (pos != null)
	updateStatus(pos.getLocation());
    }

    /**
     * Turns the mouse position into pixel location.
     * 
     * @param mousePos	the mouse position
     * @return		the pixel location
     */
    public Point mouseToPixelLocation(Point mousePos) {
      int	x;
      int	y;
      
      x = (int) (mousePos.getX() / m_Scale);
      y = (int) (mousePos.getY() / m_Scale);
      
      return new Point(x, y);
    }
    
    /**
     * Updates the status bar.
     *
     * @param pos	the mouse position
     */
    protected void updateStatus(Point pos) {
      Point	loc;

      if (getOwner() == null)
	return;

      loc = mouseToPixelLocation(pos);
      getOwner().showStatus(
  	    "X: " + (int) (loc.getX() + 1)
	  + "   "
	  + "Y: " + (int) (loc.getY() + 1)
	  + "   "
	  + "Zoom: " + Utils.doubleToString(getScale() * 100, 1) + "%");
    }

    /**
     * Displays a popup-menu. Either provided by the custom popup menu provider
     * or the default one.
     *
     * @param e		the event that triggered the popup
     * @see		#m_CustomPopupMenuProvider
     */
    protected void showPopup(MouseEvent e) {
      JPopupMenu	menu;
      JMenuItem		menuitem;

      menu = null;
      if (m_CustomPopupMenuProvider != null)
	menu = m_CustomPopupMenuProvider.getCustomPopupMenu(e);
	
      if (menu == null) {
	menu = new JPopupMenu();

	menuitem = new JMenuItem("Copy", GUIHelper.getIcon("copy.gif"));
	menuitem.setEnabled(getCurrentImage() != null);
	menuitem.addActionListener(new ActionListener() {
	  public void actionPerformed(ActionEvent e) {
	    GUIHelper.copyToClipboard(getCurrentImage());
	  }
	});
	menu.add(menuitem);

	menuitem = new JMenuItem("Export...", GUIHelper.getIcon("save.gif"));
	menuitem.setEnabled(getCurrentImage() != null);
	menuitem.addActionListener(new ActionListener() {
	  public void actionPerformed(ActionEvent e) {
	    m_PrintMouseListener.saveComponent();
	  }
	});
	menu.addSeparator();
	menu.add(menuitem);
      }

      menu.show(this, e.getX(), e.getY());
    }

    /**
     * Returns the owning panel.
     *
     * @return		the owner
     */
    public ImagePanel getOwner() {
      return m_Owner;
    }

    /**
     * Sets the image to display.
     *
     * @param value	the image to use
     */
    public void setCurrentImage(BufferedImage value) {
      m_CurrentImage = value;
      setScale(1.0);
      repaint();
    }

    /**
     * Returns the current image.
     *
     * @return		the image, can be null if not set
     */
    public BufferedImage getCurrentImage() {
      return m_CurrentImage;
    }
    
    /**
     * Sets the scaling factor (0-16).
     * TODO: allow "fit" via "-1.0"
     *
     * @param value	the scaling factor
     */
    public void setScale(double value) {
      int	width;
      int	height;

      if ((value > 0) && (value <= 16)) {
	m_Scale = value;
	if (m_CurrentImage != null) {
	  width  = (int) ((double) m_CurrentImage.getWidth() * m_Scale);
	  height = (int) ((double) m_CurrentImage.getHeight() * m_Scale);
	}
	else {
	  width  = 320;
	  height = 200;
	}
	setSize(new Dimension(width, height));
	setMinimumSize(new Dimension(width, height));
	setPreferredSize(new Dimension(width, height));
	getOwner().getScrollPane().getHorizontalScrollBar().setUnitIncrement(width / 25);
	getOwner().getScrollPane().getHorizontalScrollBar().setBlockIncrement(width / 10);
	getOwner().getScrollPane().getVerticalScrollBar().setUnitIncrement(height / 25);
	getOwner().getScrollPane().getVerticalScrollBar().setBlockIncrement(height / 10);
	update();
      }
    }
    
    /**
     * Updates the image.
     */
    protected void update() {
      getOwner().invalidate();
      getOwner().validate();
      repaint();
    }

    /**
     * Returns the scaling factor (0-16).
     *
     * @return		the scaling factor
     */
    public double getScale() {
      return m_Scale;
    }

    /**
     * Removes all image overlays.
     */
    public void clearImageOverlays() {
      m_ImageOverlays.clear();
      update();
    }

    /**
     * Adds the image overlay.
     *
     * @param io	the image overlay to add
     */
    public void addImageOverlay(ImageOverlay io) {
      m_ImageOverlays.add(io);
      update();
    }

    /**
     * Removes the image overlay.
     *
     * @param io	the image overlay to remove
     */
    public void removeImageOverlay(ImageOverlay io) {
      m_ImageOverlays.remove(io);
      update();
    }

    /**
     * Returns an iterator over all the image overlays.
     * 
     * @return		iterator on image overlays
     */
    public Iterator<ImageOverlay> imageOverlays() {
      return m_ImageOverlays.iterator();
    }
    
    /**
     * Sets the custom popup menu provider.
     * 
     * @param value	the provider, use null to remove
     */
    public void setCustomPopupMenuProvider(CustomPopupMenuProvider value) {
      m_CustomPopupMenuProvider = value;
    }
    
    /**
     * Returns the custom popup menu provider.
     * 
     * @return		the provider, null if none set
     */
    public CustomPopupMenuProvider getCustomPopupMenuProvider() {
      return m_CustomPopupMenuProvider;
    }

    /**
     * Paints the image or just a white background.
     *
     * @param g		the graphics context
     */
    public void paint(Graphics g) {
      ImageOverlay[]	overlays;

      g.setColor(getBackground());
      g.fillRect(0, 0, getWidth(), getHeight());

      if (m_CurrentImage != null) {
	((Graphics2D) g).scale(m_Scale, m_Scale);
        g.drawImage(m_CurrentImage, 0, 0, getOwner().getBackgroundColor(), null);

        // overlays
        overlays = m_ImageOverlays.toArray(new ImageOverlay[m_ImageOverlays.size()]);
        for (ImageOverlay overlay: overlays)
          overlay.paintOverlay(this, g);
      }
    }
  }

  /** the current filename. */
  protected PlaceholderFile m_CurrentFile;

  /** the panel to paint on. */
  protected PaintPanel m_PaintPanel;

  /** the JScrollPane that embeds the paint panel. */
  protected BaseScrollPane m_ScrollPane;

  /** the status bar label. */
  protected BaseStatusBar m_StatusBar;

  /** the panel with the properties. */
  protected BasePanel m_PanelProperties;

  /** the table model with the image properties. */
  protected PropertiesTableModel m_ModelProperties;

  /** the table with the image properties. */
  protected BaseTable m_TableProperties;

  /** for displaying image and properties. */
  protected BaseSplitPane m_SplitPane;

  /** the background color. */
  protected Color m_BackgroundColor;

  /** whether the image was modified. */
  protected boolean m_Modified;

  /**
   * Initializes the panel.
   */
  public ImagePanel() {
    super(Object.class, true);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_BackgroundColor = getBackground();
    m_Modified        = false;
  }

  /**
   * Initializes the GUI.
   */
  protected void initGUI() {
    JPanel	panel;

    super.initGUI();

    setLayout(new BorderLayout());

    m_SplitPane = new BaseSplitPane();
    m_SplitPane.setResizeWeight(1.0);
    add(m_SplitPane, BorderLayout.CENTER);

    m_PaintPanel = new PaintPanel(this);
    panel = new JPanel(new FlowLayout(FlowLayout.CENTER));
    panel.add(m_PaintPanel);
    m_ScrollPane = new BaseScrollPane(panel);
    m_SplitPane.setLeftComponent(m_ScrollPane);
    m_SplitPane.setLeftComponentHidden(false);

    m_PanelProperties = new BasePanel(new BorderLayout());
    m_SplitPane.setRightComponent(m_PanelProperties);
    m_SplitPane.setRightComponentHidden(true);

    m_ModelProperties = new PropertiesTableModel();
    m_TableProperties = new BaseTable(m_ModelProperties);
    m_TableProperties.setAutoResizeMode(BaseTable.AUTO_RESIZE_OFF);
    m_PanelProperties.add(new BaseScrollPane(m_TableProperties));

    m_StatusBar = new BaseStatusBar();
    add(m_StatusBar, BorderLayout.SOUTH);

    clear();
  }

  /**
   * Returns the current state of the panel.
   *
   * @return		the state
   * @see		#setState(Vector)
   */
  protected Vector getState() {
    Vector	result;

    result = new Vector();
    result.add(getCurrentFile());
    result.add(getCurrentImage());
    result.add(m_Modified);
    result.add(getScale());

    return result;
  }

  /**
   * Sets the state of the image panel.
   *
   * @param state	the state
   * @see		#getState()
   */
  protected void setState(Vector value) {
    setCurrentImage((BufferedImage) value.get(1));
    m_CurrentFile = (PlaceholderFile) value.get(0);
    m_Modified    = (Boolean) value.get(2);
    setScale((Double) value.get(3));
  }

  /**
   * Adds an undo point, if possible.
   *
   * @param statusMsg	the status message to display while adding the undo point
   * @param undoComment	the comment for the undo point
   */
  public void addUndoPoint(String statusMsg, String undoComment) {
    if (isUndoSupported() && getUndo().isEnabled()) {
      showStatus(statusMsg);
      getUndo().addUndo(getState(), undoComment);
      showStatus("");
    }
  }

  /**
   * Sets the underlying image. Removes the filename.
   *
   * @param value	the image to display
   */
  public void setCurrentImage(BufferedImage value) {
    m_CurrentFile = null;
    m_PaintPanel.setCurrentImage(value);
    updateProperties();
  }

  /**
   * Returns the underlying image.
   *
   * @return		the current image, can be null
   */
  public BufferedImage getCurrentImage() {
    return m_PaintPanel.getCurrentImage();
  }

  /**
   * Sets the filename of the current image.
   */
  public void setCurrentFile(File value) {
    if (getCurrentImage() != null) {
      if (value != null)
	m_CurrentFile = new PlaceholderFile(value);
      else
	m_CurrentFile = null;
      updateProperties();
    }
  }

  /**
   * Returns the current filename.
   *
   * @return		the current filename, can be null
   */
  public File getCurrentFile() {
    return m_CurrentFile;
  }

  /**
   * Sets the scaling factor (0-16).
   *
   * @param value	the scaling factor
   */
  public void setScale(double value) {
    //addUndoPoint("Saving undo data...", "Scaling with factor " + value);
    m_PaintPanel.setScale(value);
  }

  /**
   * Returns the scaling factor (0-16).
   *
   * @return		the scaling factor
   */
  public double getScale() {
    return m_PaintPanel.getScale();
  }

  /**
   * Sets the background color.
   *
   * @param value 	the color
   */
  public void setBackgroundColor(Color value) {
    m_BackgroundColor = value;
    repaint();
  }

  /**
   * Returns the background color.
   *
   * @return 		the color
   */
  public Color getBackgroundColor() {
    return m_BackgroundColor;
  }

  /**
   * Returns the JScrollPane that embeds the paint panel.
   *
   * @return		the JScrollPane
   */
  protected JScrollPane getScrollPane() {
    return m_ScrollPane;
  }

  /**
   * Removes the image.
   */
  public void clear() {
    addUndoPoint("Saving undo data...", "Removing image");
    m_CurrentFile = null;
    m_PaintPanel.setCurrentImage(null);
    updateProperties();
    showStatus("");
    repaint();
  }

  /**
   * Opens the file with the specified image reader.
   *
   * @param file	the file to open
   * @return		true if successfully read
   */
  public boolean load(File file) {
    boolean	result;

    addUndoPoint("Saving undo data...", "Loading file '" + file + "'");
    try {
      file = new File(file.getAbsolutePath());
      m_PaintPanel.setCurrentImage(ImageIO.read(file));
      m_CurrentFile = new PlaceholderFile(file);
      result        = true;
      updateProperties();
      repaint();
    }
    catch (Exception e) {
      e.printStackTrace();
      clear();
      result = false;
    }

    return result;
  }

  /**
   * Writes the current image to the given file.
   * Sets the modified flag to false if successfully saved.
   *
   * @param file	the file to write to
   * @return		true if successfully written, false if not or no image
   * @see		#isModified()
   */
  public boolean save(File file) {
    boolean	result;
    String	formatName;

    result = false;

    if (m_PaintPanel.getCurrentImage() != null) {
      try {
	file = new File(file.getAbsolutePath());
	formatName = ImageFileChooser.getWriterFormatName(file);
	if (formatName != null) {
	  ImageIO.write(m_PaintPanel.getCurrentImage(), formatName, file);
	  m_CurrentFile = new PlaceholderFile(file);
	}
	else {
	  System.err.println("Failed to find format name for '" + file + "'!");
	}
	result = true;
      }
      catch (Exception e) {
	e.printStackTrace();
	result = false;
      }
    }

    if (result)
      m_Modified = false;

    return result;
  }

  /**
   * Displays a message.
   *
   * @param msg		the message to display
   */
  public void showStatus(String msg) {
    m_StatusBar.showStatus(msg);
  }

  /**
   * Returns whether the image properties are currently displayed or not.
   *
   * @return		true if the properties are displayed
   */
  public boolean getShowProperties() {
    return !m_SplitPane.isRightComponentHidden();
  }

  /**
   * Sets the display status of the properties panel.
   *
   * @param value	if true then the properties get displayed
   */
  public void setShowProperties(boolean value) {
    m_SplitPane.setRightComponentHidden(!value);
  }

  /**
   * Sets the modified state.
   *
   * @param value	if true then the image gets flagged as modified
   */
  public void setModified(boolean value) {
    m_Modified = value;
  }

  /**
   * Returns whether the image was modified or not.
   *
   * @return		true if modified
   */
  public boolean isModified() {
    return m_Modified;
  }

  /**
   * Updates the properties of the image.
   */
  protected void updateProperties() {
    Properties		props;
    BufferedImage	image;

    props = new Properties();

    image = getCurrentImage();
    if (image != null) {
      if (m_CurrentFile != null)
	props.setProperty("File", "" + m_CurrentFile);
      props.setInteger("Width", image.getWidth());
      props.setInteger("Height", image.getHeight());
    }

    m_ModelProperties = new PropertiesTableModel(props);
    m_TableProperties.setModel(m_ModelProperties);
    m_TableProperties.setOptimalColumnWidth();
  }

  /**
   * An undo event, like add or remove, has occurred.
   *
   * @param e		the trigger event
   */
  public void undoOccurred(UndoEvent e) {
    // ignored
  }

  /**
   * Returns the actual panel that displays the image.
   *
   * @return		the panel
   */
  public PaintPanel getPaintPanel() {
    return m_PaintPanel;
  }

  /**
   * Removes all image overlays.
   */
  public void clearImageOverlays() {
    m_PaintPanel.clearImageOverlays();
  }

  /**
   * Adds the image overlay.
   *
   * @param io	the image overlay to add
   */
  public void addImageOverlay(ImageOverlay io) {
    m_PaintPanel.addImageOverlay(io);
  }

  /**
   * Removes the image overlay.
   *
   * @param io	the image overlay to remove
   */
  public void removeImageOverlay(ImageOverlay io) {
    m_PaintPanel.removeImageOverlay(io);
  }

  /**
   * Returns an iterator over all the image overlays.
   * 
   * @return		iterator on image overlays
   */
  public Iterator<ImageOverlay> imageOverlays() {
    return m_PaintPanel.imageOverlays();
  }

  /**
   * Sets the custom popup menu provider.
   * 
   * @param value	the provider, use null to remove
   */
  public void setCustomPopupMenuProvider(CustomPopupMenuProvider value) {
    m_PaintPanel.setCustomPopupMenuProvider(value);
  }
  
  /**
   * Returns the custom popup menu provider.
   * 
   * @return		the provider, null if none set
   */
  public CustomPopupMenuProvider getCustomPopupMenuProvider() {
    return m_PaintPanel.getCustomPopupMenuProvider();
  }

  /**
   * Turns the mouse position into pixel location.
   * 
   * @param mousePos	the mouse position
   * @return		the pixel location
   */
  public Point mouseToPixelLocation(Point mousePos) {
    return m_PaintPanel.mouseToPixelLocation(mousePos);
  }
}
