/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * BooleanExpression.java
 * Copyright (C) 2008-2012 University of Waikato, Hamilton, New Zealand
 */

package adams.parser;

import java.io.ByteArrayInputStream;
import java.util.HashMap;
import java.util.Vector;

import java_cup.runtime.DefaultSymbolFactory;
import java_cup.runtime.SymbolFactory;
import adams.data.report.AbstractField;
import adams.data.report.Report;
import adams.parser.booleanexpression.Parser;
import adams.parser.booleanexpression.Scanner;

/**
 <!-- globalinfo-start -->
 * Evaluates boolean expressions.<br/>
 * <br/>
 * It uses the following grammar:<br/>
 * <br/>
 *  expr_list ::= expr_list expr_part | expr_part ;<br/>
 *  expr_part ::= boolexpr ;<br/>
 * <br/>
 *  boolexpr ::=    ( boolexpr )<br/>
 *                | true<br/>
 *                | false<br/>
 *                | bool ( VARIABLE (must represent a boolean string: true|false) )<br/>
 *                | numexpr &lt; numexpr<br/>
 *                | numexpr &lt;= numexpr<br/>
 *                | numexpr &gt; numexpr<br/>
 *                | numexpr &gt;= numexpr<br/>
 *                | numexpr = numexpr<br/>
 *                | numexpr != numexpr<br/>
 *                | strexpr &lt; strexpr<br/>
 *                | strexpr &lt;= strexpr<br/>
 *                | strexpr &gt; strexpr<br/>
 *                | strexpr &gt;= strexpr<br/>
 *                | strexpr = strexpr<br/>
 *                | strexpr != strexpr<br/>
 *                | matches ( strexpr , regexp )<br/>
 *                | ! boolexpr<br/>
 *                | boolexpr &amp; boolexpr<br/>
 *                | boolexpr | boolexpr<br/>
 *                ;<br/>
 * <br/>
 *  numexpr   ::=   ( numexpr )<br/>
 *                | NUMBER<br/>
 *                | -numexpr<br/>
 *                | constexpr<br/>
 *                | opexpr<br/>
 *                | funcexpr<br/>
 *                ;<br/>
 * <br/>
 *  strexpr   ::=   substr ( strexpr , start )<br/>
 *                | substr ( strexpr , start , end )<br/>
 *                | lowercase ( strexpr )<br/>
 *                | uppercase ( strexpr )<br/>
 *                | string<br/>
 *                | str ( VARIABLE (must represent a string) )<br/>
 *                ;<br/>
 * <br/>
 *  constexpr ::=   PI<br/>
 *                | E<br/>
 *                ;<br/>
 * <br/>
 *  opexpr    ::=   numexpr + numexpr<br/>
 *                | numexpr - numexpr<br/>
 *                | numexpr * numexpr<br/>
 *                | numexpr &#47; numexpr<br/>
 *                | numexpr ^ numexpr (power of)<br/>
 *                | numexpr % numexpr (modulo)<br/>
 *                ;<br/>
 * <br/>
 *  funcexpr ::=    abs ( numexpr )<br/>
 *                | sqrt ( numexpr )<br/>
 *                | log ( numexpr )<br/>
 *                | exp ( numexpr )<br/>
 *                | sin ( numexpr )<br/>
 *                | cos ( numexpr )<br/>
 *                | tan ( numexpr )<br/>
 *                | rint ( numexpr )<br/>
 *                | floor ( numexpr )<br/>
 *                | pow ( numexpr , numexpr )<br/>
 *                | ceil ( numexpr )<br/>
 *                | ifelse ( boolexpr , numexpr (if true) , numexpr (if false) )<br/>
 *                | ifmissing ( variable , numexpr (default value if variable is missing) )<br/>
 *                | num ( VARIABLE (must represent a number) )<br/>
 *                | length ( str )<br/>
 *                ;<br/>
 * <br/>
 * Notes:<br/>
 * - Variables are either all upper case letters (e.g., "ABC") or any character   apart from "]" enclosed by "[" and "]" (e.g., "[Hello World]").<br/>
 * - 'start' and 'end' for function 'substr' are indices that start at 1.<br/>
 * - Index 'end' for function 'substr' is excluded (like Java's 'String.substring(int,int)' method)<br/>
 * - Line comments start with '#'<br/>
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-env &lt;java.lang.String&gt; (property: environment)
 * &nbsp;&nbsp;&nbsp;The class to use for determining the environment.
 * &nbsp;&nbsp;&nbsp;default: adams.env.Environment
 * </pre>
 * 
 * <pre>-expression &lt;java.lang.String&gt; (property: expression)
 * &nbsp;&nbsp;&nbsp;The boolean expression to evaluate (must evaluate to a boolean).
 * &nbsp;&nbsp;&nbsp;default: true
 * </pre>
 * 
 * <pre>-symbol &lt;adams.core.base.BaseString&gt; [-symbol ...] (property: symbols)
 * &nbsp;&nbsp;&nbsp;The symbols to initialize the parser with, key-value pairs: name=value.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4839 $
 */
public class BooleanExpression
  extends AbstractSymbolEvaluator<Boolean> {

  /** for serialization. */
  private static final long serialVersionUID = -5923987640355752595L;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Evaluates boolean expressions.\n\n"
      + "It uses the following grammar:\n\n"
      + getGrammar();
  }

  /**
   * Returns a string representation of the grammar.
   *
   * @return		the grammar, null if not available
   */
  public String getGrammar() {
    return
        " expr_list ::= expr_list expr_part | expr_part ;\n"
      + " expr_part ::= boolexpr ;\n"
      + "\n"
      + " boolexpr ::=    ( boolexpr )\n"
      + "               | true\n"
      + "               | false\n"
      + "               | bool ( VARIABLE (must represent a boolean string: true|false) )\n"
      + "               | numexpr < numexpr\n"
      + "               | numexpr <= numexpr\n"
      + "               | numexpr > numexpr\n"
      + "               | numexpr >= numexpr\n"
      + "               | numexpr = numexpr\n"
      + "               | numexpr != numexpr\n"
      + "               | strexpr < strexpr\n"
      + "               | strexpr <= strexpr\n"
      + "               | strexpr > strexpr\n"
      + "               | strexpr >= strexpr\n"
      + "               | strexpr = strexpr\n"
      + "               | strexpr != strexpr\n"
      + "               | matches ( strexpr , regexp )\n"
      + "               | ! boolexpr\n"
      + "               | boolexpr & boolexpr\n"
      + "               | boolexpr | boolexpr\n"
      + "               ;\n"
      + "\n"
      + " numexpr   ::=   ( numexpr )\n"
      + "               | NUMBER\n"
      + "               | -numexpr\n"
      + "               | constexpr\n"
      + "               | opexpr\n"
      + "               | funcexpr\n"
      + "               ;\n"
      + "\n"
      + " strexpr   ::=   substr ( strexpr , start )\n"
      + "               | substr ( strexpr , start , end )\n"
      + "               | lowercase ( strexpr )\n"
      + "               | uppercase ( strexpr )\n"
      + "               | string\n"
      + "               | str ( VARIABLE (must represent a string) )\n"
      + "               ;\n"
      + "\n"
      + " constexpr ::=   PI\n"
      + "               | E\n"
      + "               ;\n"
      + "\n"
      + " opexpr    ::=   numexpr + numexpr\n"
      + "               | numexpr - numexpr\n"
      + "               | numexpr * numexpr\n"
      + "               | numexpr / numexpr\n"
      + "               | numexpr ^ numexpr (power of)\n"
      + "               | numexpr % numexpr (modulo)\n"
      + "               ;\n"
      + "\n"
      + " funcexpr ::=    abs ( numexpr )\n"
      + "               | sqrt ( numexpr )\n"
      + "               | log ( numexpr )\n"
      + "               | exp ( numexpr )\n"
      + "               | sin ( numexpr )\n"
      + "               | cos ( numexpr )\n"
      + "               | tan ( numexpr )\n"
      + "               | rint ( numexpr )\n"
      + "               | floor ( numexpr )\n"
      + "               | pow ( numexpr , numexpr )\n"
      + "               | ceil ( numexpr )\n"
      + "               | ifelse ( boolexpr , numexpr (if true) , numexpr (if false) )\n"
      + "               | ifmissing ( variable , numexpr (default value if variable is missing) )\n"
      + "               | num ( VARIABLE (must represent a number) )\n"
      + "               | length ( str )\n"
      + "               ;\n"
      + "\n"
      + "Notes:\n"
      + "- Variables are either all upper case letters (e.g., \"ABC\") or any character "
      + "  apart from \"]\" enclosed by \"[\" and \"]\" (e.g., \"[Hello World]\").\n"
      + "- 'start' and 'end' for function 'substr' are indices that start at 1.\n"
      + "- Index 'end' for function 'substr' is excluded (like Java's 'String.substring(int,int)' method)\n"
      + "- Line comments start with '#'\n"
      ;
  }

  /**
   * Returns the default expression to use.
   *
   * @return		the default expression
   */
  protected String getDefaultExpression() {
    return "true";
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String expressionTipText() {
    return "The boolean expression to evaluate (must evaluate to a boolean).";
  }

  /**
   * Initializes the symbol.
   *
   * @param name	the name of the symbol
   * @param value	the string representation of the symbol
   * @return		the object representation of the symbol
   */
  protected Object initializeSymbol(String name, String value) {
    Double	result;

    try {
      result = new Double(value);
    }
    catch (Exception e) {
      result = null;
      getSystemErr().println("Failed to parse the value of symbol '" + name + "': " + value);
      getSystemErr().printStackTrace(e);
    }

    return result;
  }

  /**
   * Performs the actual evaluation.
   *
   * @param symbols	the symbols to use
   * @return		the evaluation, or null in case of error
   * @throws Exception	if evaluation fails
   */
  protected Boolean doEvaluate(HashMap symbols) throws Exception {
    return evaluate(m_Expression, symbols);
  }

  /**
   * Parses and evaluates the given expression.
   * Returns the result of the boolean expression, based on the given
   * values of the symbols.
   *
   * @param expr	the expression to evaluate
   * @param symbols	the symbol/value mapping
   * @return		the evaluated result
   * @throws Exception	if something goes wrong
   */
  public static boolean evaluate(String expr, HashMap symbols) throws Exception {
    SymbolFactory 		sf;
    ByteArrayInputStream 	parserInput;
    Parser 			parser;

    if (expr.equals("true"))
      return true;
    else if (expr.equals("false"))
      return false;

    sf          = new DefaultSymbolFactory();
    parserInput = new ByteArrayInputStream(expr.getBytes());
    parser      = new Parser(new Scanner(parserInput, sf), sf);
    parser.setSymbols(symbols);
    parser.parse();

    return parser.getResult();
  }

  /**
   * Parses and evaluates the given expression.
   * Returns the result of the mathematical expression, based on the given
   * values of the symbols.
   *
   * @param expr	the expression to evaluate
   * @param report	the report
   * @return		the evaluated result
   * @throws Exception	if something goes wrong
   */
  public static boolean evaluate(String expr, Report report) throws Exception {
    HashMap			values;
    Vector<AbstractField>	fields;
    
    // transfer values
    values = new HashMap();
    fields = report.getFields();
    for (AbstractField field: fields) {
      switch (field.getDataType()) {
	case NUMERIC:
	  values.put(field.toString(), report.getDoubleValue(field));
	  break;
	case BOOLEAN:
	  values.put(field.toString(), report.getBooleanValue(field));
	  break;
	default:
	  values.put(field.toString(), "" + report.getValue(field));
      }
    }

    return evaluate(expr, values);
  }

  /**
   * Runs the evaluator from command-line.
   *
   * @param args	the command-line options, use "-help" to list them
   */
  public static void main(String[] args) {
    runEvaluator(BooleanExpression.class, args);
  }
}
