/*
 * FileUtils.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.core.io;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import java.util.Vector;

import adams.core.management.OS;

/**
 * Utility class for I/O related actions.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3846 $
 */
public class FileUtils {

  /** valid characters for filenames. */
  public final static String FILENAME_CHARS = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_-.,()";

  /** the maximum length for an extension. */
  public final static int MAX_EXTENSION_LENGTH = 5;

  /**
   * Returns the content of the given file, null in case of an error.
   *
   * @param file	the file to load
   * @return		the content/lines of the file
   */
  public static Vector<String> loadFromFile(File file) {
    Vector<String>	result;
    BufferedReader	reader;
    String		line;

    result = new Vector<String>();

    try {
      reader = new BufferedReader(new FileReader(file.getAbsolutePath()));
      while ((line = reader.readLine()) != null)
        result.add(line);
      reader.close();
    }
    catch (Exception e) {
      result = null;
      e.printStackTrace();
    }

    return result;
  }

  /**
   * Loads the binary file.
   *
   * @param file	the file to load
   * @return		the binary content, null in case of an error
   */
  public static byte[] loadFromBinaryFile(File file) {
    byte[]		result;
    Vector<Byte>	content;
    int			read;
    byte[]		buffer;
    BufferedInputStream	stream;
    int			i;

    content = new Vector<Byte>();

    stream = null;
    try {
      buffer = new byte[1024];
      stream = new BufferedInputStream(new FileInputStream(file.getAbsoluteFile()));
      while ((read = stream.read(buffer)) != -1) {
	for (i = 0; i < read; i++)
	  content.add(buffer[i]);
      }
    }
    catch (Exception e) {
      System.err.println("Problem loading binary file '" + file + "':");
      e.printStackTrace();
      content = null;
    }
    if (stream != null) {
      try {
	stream.close();
      }
      catch (Exception e) {
	// ignored
      }
    }

    if (content != null) {
      result = new byte[content.size()];
      for (i = 0; i < result.length; i++)
	result[i] = content.get(i);
    }
    else {
      result = null;
    }

    return result;
  }

  /**
   * Saves the content to the given file.
   *
   * @param content	the content to save
   * @param file	the file to save the content to
   * @return		true if successfully saved
   */
  public static boolean saveToFile(String[] content, File file) {
    Vector<String>	lines;
    int			i;

    lines = new Vector<String>();
    for (i = 0; i < content.length; i++)
      lines.add(content[i]);

    return FileUtils.saveToFile(lines, file);
  }

  /**
   * Saves the content to the given file.
   *
   * @param content	the content to save
   * @param file	the file to save the content to
   * @return		true if successfully saved
   */
  public static boolean saveToFile(List<String> content, File file) {
    boolean		result;
    BufferedWriter	writer;
    int			i;

    result = true;

    try {
      writer = new BufferedWriter(new FileWriter(new File(file.getAbsolutePath())));
      for (i = 0; i < content.size(); i++) {
        writer.write(content.get(i));
        writer.newLine();
      }
      writer.flush();
      writer.close();
    }
    catch (Exception e) {
      result = false;
      e.printStackTrace();
    }

    return result;
  }

  /**
   * Writes the given object to the specified file. The object is always
   * appended.
   *
   * @param filename	the file to write to
   * @param obj		the object to write
   * @return		true if writing was successful
   */
  public static boolean writeToFile(String filename, Object obj) {
    return FileUtils.writeToFile(filename, obj, true);
  }

  /**
   * Writes the given object to the specified file. The message is either
   * appended or replaces the current content of the file.
   *
   * @param filename	the file to write to
   * @param obj		the object to write
   * @param append	whether to append the message or not
   * @return		true if writing was successful
   */
  public static boolean writeToFile(String filename, Object obj, boolean append) {
    boolean		result;
    BufferedWriter	writer;

    try {
      writer = new BufferedWriter(new FileWriter(filename, append));
      writer.write("" + obj);
      writer.newLine();
      writer.flush();
      writer.close();
      result = true;
    }
    catch (Exception e) {
      result = false;
    }

    return result;
  }

  /**
   * Copies or moves files and directories (recursively).
   * If targetLocation does not exist, it will be created.
   * <p/>
   * Original code from <a href="http://www.java-tips.org/java-se-tips/java.io/how-to-copy-a-directory-from-one-location-to-another-loc.html" target="_blank">Java-Tips.org</a>.
   *
   * @param sourceLocation	the source file/dir
   * @param targetLocation	the target file/dir
   * @param move		if true then the source files/dirs get deleted
   * 				as soon as copying finished
   * @throws IOException	if copying/moving fails
   */
  public static void copyOrMove(File sourceLocation, File targetLocation, boolean move) throws IOException {
    String[] 		children;
    int 		i;
    InputStream 	in;
    OutputStream 	out;
    byte[] 		buf;
    int 		len;

    if (sourceLocation.isDirectory()) {
      if (!targetLocation.exists())
        targetLocation.mkdir();

      children = sourceLocation.list();
      for (i = 0; i < children.length; i++) {
        copyOrMove(
            new File(sourceLocation.getAbsoluteFile(), children[i]),
            new File(targetLocation.getAbsoluteFile(), children[i]),
            move);
      }

      if (move)
        sourceLocation.delete();
    }
    else {
      in = new FileInputStream(sourceLocation.getAbsoluteFile());
      // do we need to append the filename?
      if (targetLocation.isDirectory())
        out = new FileOutputStream(targetLocation.getAbsolutePath() + File.separator + sourceLocation.getName());
      else
        out = new FileOutputStream(targetLocation.getAbsoluteFile());

      // Copy the content from instream to outstream
      buf = new byte[1024];
      while ((len = in.read(buf)) > 0)
        out.write(buf, 0, len);

      in.close();
      out.close();

      if (move)
        sourceLocation.delete();
    }
  }

  /**
   * Copies the file/directory (recursively).
   *
   * @param sourceLocation	the source file/dir
   * @param targetLocation	the target file/dir
   * @throws IOException	if copying fails
   */
  public static void copy(File sourceLocation, File targetLocation) throws IOException {
    copyOrMove(sourceLocation, targetLocation, false);
  }

  /**
   * Moves the file/directory (recursively).
   *
   * @param sourceLocation	the source file/dir
   * @param targetLocation	the target file/dir
   * @throws IOException	if moving fails
   */
  public static void move(File sourceLocation, File targetLocation) throws IOException {
    copyOrMove(sourceLocation, targetLocation, true);
  }

  /**
   * Deletes the specified file. If the file represents a directory, then this
   * will get deleted recursively.
   *
   * @param file	the file/dir to delete
   * @return		true if successfully deleted
   */
  public static boolean delete(String file) {
    return delete(new PlaceholderFile(file));
  }

  /**
   * Deletes the specified file. If the file represents a directory, then this
   * will get deleted recursively.
   *
   * @param file	the file/dir to delete
   * @return		true if successfully deleted
   */
  public static boolean delete(File file) {
    boolean	result;
    File[]	files;

    result = true;

    if (file.isDirectory()) {
      files = file.listFiles();
      for (File f: files) {
	if (f.getName().equals(".") || f.getName().equals(".."))
	  continue;
	delete(f);
      }
    }

    result = file.delete();

    return result;
  }

  /**
   * Replaces all characters that would create problems on a filesystem.
   * The string is to be expected a filename without a path.
   *
   * @param s		the string to process
   * @param replace	the character to replace "invalid" characters with,
   * 			use empty string to strip "invalid" characters instead
   * 			of replacing them.
   * @return		the processed string
   * @see		#FILENAME_CHARS
   */
  public static String createFilename(String s, String replace) {
    StringBuffer	result;
    int			i;

    result = new StringBuffer();

    for (i = 0; i < s.length(); i++) {
      if (FILENAME_CHARS.indexOf(s.charAt(i)) == -1) {
	result.append(replace);
      }
      else {
	result.append(s.charAt(i));
      }
    }

    return result.toString();
  }

  /**
   * Creates a partial filename for the given file, based on how many parent
   * directories should be included. Examples:
   * <pre>
   * createPartialFilename(new File("/home/some/where/file.txt"), -1)
   *   = /home/some/where/file.txt
   * createPartialFilename(new File("/home/some/where/file.txt"), 0)
   *   = file.txt
   * createPartialFilename(new File("/home/some/where/file.txt"), 1)
   *   = where/file.txt
   * createPartialFilename(new File("/home/some/where/file.txt"), 2)
   *   = some/where/file.txt
   * </pre>
   *
   * @param file		the file to create the partial filename for
   * @param numParentDirs	the number of parent directories to include in
   * 				the partial name, -1 returns the absolute
   * 				filename
   * @return			the generated filename
   */
  public static String createPartialFilename(File file, int numParentDirs) {
    String	result;
    File	parent;
    int		i;

    if (numParentDirs == -1) {
      result = file.getAbsolutePath();
    }
    else {
      result = file.getName();
      parent = file;
      for (i = 0; (i < numParentDirs) && (parent.getParentFile() != null); i++) {
        parent = parent.getParentFile();
        result = parent.getName() + File.separator + result;
      }
    }

    return result;
  }

  /**
   * Checks whether the directory is empty.
   *
   * @param dir		the directory to check
   * @return		true if empty
   */
  public static boolean isDirEmpty(File dir) {
    return isDirEmpty(dir, null);
  }

  /**
   * Checks whether the directory is empty.
   *
   * @param dir		the directory to check
   * @return		true if empty
   */
  public static boolean isDirEmpty(String dir) {
    return isDirEmpty(dir, null);
  }

  /**
   * Checks whether the directory is empty.
   *
   * @param dir		the directory to check
   * @param regExp	a regular expression to look for, use null to ignore
   * @return		true if empty
   */
  public static boolean isDirEmpty(File dir, String regExp) {
    return isDirEmpty(dir.getAbsolutePath(), regExp);
  }

  /**
   * Checks whether the directory is empty.
   *
   * @param dir		the directory to check
   * @param regExp	a regular expression to look for, use null to ignore
   * @return		true if empty
   */
  public static boolean isDirEmpty(String dir, String regExp) {
    boolean	result;
    File	file;
    String[]	files;
    int		i;

    result = true;

    file  = new File(dir);
    files = file.list();
    for (i = 0; i < files.length; i++) {
      // skip . and ..
      if (files[i].equals(".") || files[i].equals(".."))
	continue;

      // only look for matching filenames only
      if ((regExp != null) && (!files[i].matches(regExp)))
	continue;

      // found at least 1 file!
      result = false;
      break;
    }

    return result;
  }

  /**
   * Adjusts the extension according to the platform. For Windows it
   * automatically adds ".exe" it neither ".com" nor ".exe" extension present.
   * For other platforms it removes ".exe" and ".com".
   *
   * @param executable	the executable (full path or just filename) to process
   * @return		the processed executable
   */
  public static String fixExecutable(String executable)  {
    String	result;

    result = executable;

    if (OS.isWindows()) {
      if (!result.endsWith(".exe") || !result.endsWith(".com"))
	result += ".exe";
    }
    else {
      if (result.endsWith(".exe") || result.endsWith(".com"))
	result = result.substring(0, result.length() - 4);
    }

    return result;
  }

  /**
   * Surrounds the executable with double quotes if a blank is in the path.
   *
   * @param executable	the executable (full path, no parameters)
   * @return		the processed executable
   */
  public static String quoteExecutable(String executable) {
    String	result;

    result = executable;
    if (result.indexOf(' ') > -1)
      result = "\"" + result + "\"";

    return result;
  }

  /**
   * Returns the extension of the file, if any.
   *
   * @param file	the file to get the extension from
   * @return		the extension, null if none available
   */
  public static String getExtension(File file) {
    return getExtension(file.getAbsolutePath());
  }

  /**
   * Returns the extension of the file, if any.
   *
   * @param filename	the file to get the extension from
   * @return		the extension, null if none available
   */
  public static String getExtension(String filename) {
    String[]	result;

    result = getExtensions(filename);

    if (result != null)
      return result[0];
    else
      return null;
  }

  /**
   * Returns the extensions of the file, if any.
   * Returns ".txt.gz" and ".gz", for instance, for file "hello_world.txt.gz".
   * The longer extension always comes first.
   *
   * @param file	the file to get the extensions from
   * @return		the extensions, null if none available
   */
  public static String[] getExtensions(File file) {
    return getExtensions(file.getAbsolutePath());
  }

  /**
   * Returns the extensions of the file, if any.
   * Returns ".txt.gz" and ".gz", for instance, for file "hello_world.txt.gz".
   * The longer extension always comes first.
   *
   * @param filename	the file to get the extensions from
   * @return		the extensions, null if none available
   * @see		#MAX_EXTENSION_LENGTH
   */
  public static String[] getExtensions(String filename) {
    Vector<String>	result;
    int			pos;
    int			posNext;

    if (filename.indexOf('.') == -1)
      return null;

    result = new Vector<String>();
    pos    = filename.lastIndexOf('.');
    result.add(filename.substring(pos + 1));

    posNext = filename.lastIndexOf('.', pos - 1);
    if (pos - posNext <= MAX_EXTENSION_LENGTH)
      result.add(0, filename.substring(posNext + 1));

    return result.toArray(new String[result.size()]);
  }

  /**
   * Returns the temporary directory.
   *
   * @return	the temp directory
   */
  public static File getTempDirectory() {
    return new File(System.getProperty("java.io.tmpdir"));
  }
}
