/**
 * Launcher.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core.management;

import java.util.Arrays;
import java.util.Vector;

import adams.core.ClassLocator;
import adams.core.option.OptionUtils;

/**
 * Launches a new JVM process with the specified memory
 * (<code>-memory &lt;amount&gt;</code>) and main class
 * (<code>-main &lt;classname&gt;</code>).
 * <p/>
 * All other command-line arguments are passed on to the new process.
 * <p/>
 * In addition to parameters from the commandline, additional parameters can
 * be defined in the <code>ADAMS_OPTS</code> environment variable.
 * Note: No checks are performed if the same parameter is defined in this
 * env variable and on the commandline.
 * <p/>
 * When run from commandline, the method <code>addShutdownHook()</code> gets
 * called automatically before <code>execute()</code>, which adds a hook
 * thread to the runtime, killing the launched process, e.g., when
 * using <code>ctrl+c</code>.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4139 $
 * @see #ENV_ADAMS_OPTS
 */
public class Launcher {

  /** the return code for restarting the main class. */
  public final static int CODE_RESTART = 100;

  /** the return code for restarting the main class with 50% more heap. */
  public final static int CODE_RESTART_MORE_HEAP = 101;

  /** the environment variable with additional options. */
  public final static String ENV_ADAMS_OPTS = "ADAMS_OPTS";

  /** the amount of memory to use for the process. */
  protected String m_Memory;

  /** the main class to launch. */
  protected String m_MainClass;

  /** optional JVM options. */
  protected Vector<String> m_JVMOptions;

  /** the arguments for the process. */
  protected String[] m_Arguments;

  /** the runtime object in use. */
  protected Runtime m_Runtime;

  /** the process that got launched. */
  protected Process m_Process;

  /**
   * Initializes the launcher.
   */
  public Launcher() {
    super();

    m_Memory     = "";
    m_MainClass  = "";
    m_JVMOptions = new Vector<String>();
    m_Arguments  = new String[0];
    m_Runtime    = Runtime.getRuntime();
  }

  /**
   * Sets the amount of memory for the process.
   *
   * @param value	the amount
   * @return		null if valid amount, otherwise error message
   */
  public String setMemory(String value) {
    String	result;

    result = null;
    value  = value.trim().toLowerCase();
    if (value.endsWith("k") || value.endsWith("m") || value.endsWith("g")) {
      try {
	Integer.parseInt(value.substring(0, value.length() - 1));
	m_Memory = value;
      }
      catch (Exception e) {
	result = "Failed to parse '" + value + "': " + e;
      }
    }
    else {
      result = "Memory amount must end with one of the following quantifiers: k, m, g";
    }

    return result;
  }

  /**
   * Increase the heap by 50%.
   */
  protected void increaseHeap() {
    String	suffix;
    String	amount;
    int		i;
    double	factor;

    // split up memory string
    suffix = "";
    for (i = m_Memory.length() - 1; i >= 0; i--) {
      if ((m_Memory.charAt(i) >= '0') && (m_Memory.charAt(i) <= '9'))
	break;
      else
	suffix = m_Memory.charAt(i) + suffix;
    }
    amount = m_Memory.substring(0, m_Memory.length() - suffix.length());
    factor = 1.5;
    if (suffix.toLowerCase().equals("g")) {
      suffix = "m";
      factor = 1500;
    }

    m_Memory = (long) (Long.parseLong(amount) * factor) + suffix;
  }

  /**
   * Sets the main class of the process.
   *
   * @param value	the class name
   * @return		null if valid class, otherwise error message
   */
  public String setMainClass(String value) {
    String	result;

    result = null;

    try {
      Class.forName(value);
      m_MainClass = value;
    }
    catch (Exception e) {
      result = "Class not found: " + value;
    }

    return result;
  }

  /**
   * Adds the JVM option.
   *
   * @param value	the option
   */
  public void addJVMOption(String value) {
    m_JVMOptions.add(value);
  }

  /**
   * Sets the arguments for the process.
   *
   * @param value	the arguments
   */
  public String setArguments(String[] value) {
    String	result;

    result = null;

    try {
      m_Arguments = OptionUtils.splitOptions(OptionUtils.joinOptions(value));
    }
    catch (Exception e) {
      result = "Failed to parse the arguments: " + e;
    }

    return result;
  }

  /**
   * Adds a shutdown hook, to kill the launched process.
   *
   * @see 		#m_Process
   */
  public void addShutdownHook() {
    Thread	thread;

    thread = new Thread() {
      public void run() {
	if (m_Process != null)
	  m_Process.destroy();
      }
    };
    m_Runtime.addShutdownHook(thread);
  }

  /**
   * Launches the main class.
   */
  public void execute() {
    Vector<String>	cmd;
    int			retVal;
    OutputProcessStream	stdout;
    OutputProcessStream	stderr;
    boolean		enableRestart;

    enableRestart = false;
    try {
      if (ClassLocator.hasInterface(RestartableApplication.class, Class.forName(m_MainClass)))
	enableRestart = true;
    }
    catch (Exception e) {
      System.err.println("Failed to instantiate class '" + m_MainClass + "'!");
      e.printStackTrace();
    }

    cmd = new Vector<String>();
    cmd.add(Java.getJavaExecutable());
    cmd.add("-Xmx" + m_Memory);
    cmd.addAll(m_JVMOptions);
    cmd.add("-classpath");
    cmd.add(System.getProperty("java.class.path"));
    cmd.add(m_MainClass);
    if (enableRestart)
      cmd.add(RestartableApplication.FLAG_ENABLE_RESTART);
    cmd.addAll(Arrays.asList(m_Arguments));

    if (System.getenv(ENV_ADAMS_OPTS) != null) {
      try {
	cmd.addAll(Arrays.asList(OptionUtils.splitOptions(System.getenv(ENV_ADAMS_OPTS))));
      }
      catch (Exception e) {
	System.err.println("Error parsing environment variable '" + ENV_ADAMS_OPTS + "':");
	e.printStackTrace();
      }
    }

    try {
      m_Process = m_Runtime.exec(cmd.toArray(new String[cmd.size()]));
      stdout    = new OutputProcessStream(m_Process, true);
      stderr    = new OutputProcessStream(m_Process, false);
      new Thread(stdout).start();
      new Thread(stderr).start();

      retVal = m_Process.waitFor();

      stdout.stop();
      stderr.stop();

      if (retVal == CODE_RESTART) {
	execute();
	return;
      }
      else if (retVal == CODE_RESTART_MORE_HEAP) {
	increaseHeap();
	execute();
	return;
      }
    }
    catch (Exception e) {
      System.err.println("Exception occurred launching " + m_MainClass + ":");
      e.printStackTrace();
    }
  }

  /**
   * Returns the value of the option (if available). Also removes the option
   * from the list.
   *
   * @param args	the option list to parse
   * @param option	the option to look for (incl "-")
   * @return		the value, null if not found
   */
  protected static String getOption(Vector<String> args, String option) {
    String	result;
    int		i;

    result = OptionUtils.getOption(args, option);
    if (result != null) {
      i = 0;
      while (i < args.size()) {
	if (args.get(i).equals(option)) {
	  args.remove(i);
	  if (i < args.size())
	    args.remove(i);
	  break;
	}
	i++;
      }
    }

    return result;
  }

  /**
   * Configures the launcher.
   *
   * @param args	the commandline arguments to use for launching
   * @param launcher
   */
  protected static String configure(String[] args, Launcher launcher) {
    String		result;
    String		value;
    Vector<String>	options;

    result = null;

    options = new Vector<String>(Arrays.asList(args));

    // memory
    value = getOption(options, "-memory");
    if (value != null)
      result = launcher.setMemory(value);
    else
      result = "Missing option: -memory";

    // main class
    if (result == null) {
      value = getOption(options, "-main");
      if (value != null)
        result = launcher.setMainClass(value);
      else
        result = "Missing option: -main";
    }

    // JVM options
    if (result == null) {
      while ((value = getOption(options, "-jvm")) != null)
	launcher.addJVMOption(value);
    }

    if (result == null)
      result = launcher.setArguments(options.toArray(new String[options.size()]));

    return result;
  }

  /**
   * Executes the class from the command-line.
   *
   * @param args	the command-line arguments
   */
  public static void main(String[] args) throws Exception {
    Launcher 	launcher;
    String 	error;

    if (OptionUtils.helpRequested(args)) {
      System.out.println("Options:");
      System.out.println("-memory <amount>");
      System.out.println("\tSpecifies the maximum amount of memory to allocate for");
      System.out.println("\tthe heap in the JVM for the process that is being launch.");
      System.out.println("\tUse 'k' for kilobytes, 'm' for megabytes and 'g' for ");
      System.out.println("\tgigabytes. Examples: 1000m, 2g");
      System.out.println("-main <classname>");
      System.out.println("\tThe class to launch as main class in the new JVM process.");
      System.out.println("[-jvm <option>]");
      System.out.println("\tOptional arguments for the JVM.");
      System.out.println("\tExample: -jvm -javaagent:sizeofag.jar");
      System.out.println("-...");
      System.out.println("\tAny other option will get passed on to the main class.");
      return;
    }

    launcher = new Launcher();
    error    = configure(args, launcher);
    if (error == null) {
      launcher.addShutdownHook();
      launcher.execute();
    }
    else {
      System.err.println("Failed to execute launcher:");
      System.err.println(error);
      System.exit(1);
    }
  }
}
