/*
 * AbstractDirector.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;


import adams.core.CleanUpHandler;
import adams.core.ConsoleObject;
import adams.core.DebugOutputHandler;
import adams.core.Pausable;
import adams.core.Stoppable;
import adams.core.Variables;

/**
 * Manages the execution of actors.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3136 $
 */
public abstract class AbstractDirector
  extends ConsoleObject
  implements DebugOutputHandler, CleanUpHandler, Stoppable, Pausable {

  /** for serialization. */
  private static final long serialVersionUID = -1634725837304059804L;

  /** the control actor to execute. */
  protected AbstractControlActor m_ControlActor;

  /** the debugging level (0 = off, >0 = on). */
  protected int m_DebugLevel;

  /** whether execution was stopped. */
  protected boolean m_Stopped;

  /** whether execution is in the process of being stopped. */
  protected boolean m_Stopping;

  /** whether the director has been paused. */
  protected boolean m_Paused;

  /**
   * Initializes the item.
   */
  public AbstractDirector() {
    super();

    initialize();
  }

  /**
   * Initializes the members.
   * <p/>
   * Default implementation does nothing
   */
  protected void initialize() {
  }

  /**
   * Updates the prefix of the print objects.
   */
  public void updatePrefix() {
    if (m_ControlActor != null) {
      getSystemOut().setPrefix(m_ControlActor.getFullName() + "." + getClass().getSimpleName() + "/" + hashCode());
      getSystemErr().setPrefix(m_ControlActor.getFullName() + "." + getClass().getSimpleName() + "/" + hashCode());
      getDebugging().setPrefix(m_ControlActor.getFullName() + "." + getClass().getSimpleName() + "/" + hashCode());
    }
  }

  /**
   * Sets the debugging level (0 = off).
   *
   * @param value 	>0 if debugging output should be printed
   */
  public void setDebugLevel(int value) {
    m_DebugLevel = value;
    getDebugging().setEnabled(value > 0);
  }

  /**
   * Returns the debugging level (0 = turned off).
   *
   * @return 		true if debugging output is on
   */
  public int getDebugLevel() {
    return m_DebugLevel;
  }

  /**
   * Returns true if debugging output is turned on (any level).
   *
   * @return		true if debugging output is turned on
   */
  protected boolean isDebugOn() {
    return (m_DebugLevel > 0);
  }

  /**
   * Processes the debugging message.
   *
   * @param msg		the debugging message to process
   */
  public void debug(String msg) {
    debug(msg, 1);
  }

  /**
   * Processes the debugging message.
   *
   * @param level	the debugging level
   * @param msg		the debugging message to process
   */
  public void debug(String msg, int level) {
    if (level <= m_DebugLevel)
      getDebugging().println(msg);
  }

  /**
   * Sets the control actor to execute.
   *
   * @param value 	the control actor
   */
  public void setControlActor(AbstractControlActor value) {
    m_ControlActor = value;
    updatePrefix();
  }

  /**
   * Returns the control actor to execute.
   *
   * @return 		the control actor
   */
  public AbstractControlActor getControlActor() {
    return m_ControlActor;
  }

  /**
   * Return the Variables instance used by the control actor.
   *
   * @return		the instance in use
   */
  protected Variables getVariables() {
    return m_ControlActor.getVariables();
  }

  /**
   * Executes the group of actors.
   *
   * @return		null if everything went smooth
   */
  public abstract String execute();

  /**
   * Pauses the execution.
   */
  public void pauseExecution() {
    m_Paused = true;
  }

  /**
   * The pause loop.
   */
  protected void pause() {
    // paused?
    while (isPaused() && !isStopped() && !isStopping()) {
      try {
	synchronized(this) {
	  wait(500);
	}
      }
      catch (Exception e) {
	// ignored
      }
    }
  }

  /**
   * Returns whether the object is currently paused.
   *
   * @return		true if object is paused
   */
  public boolean isPaused() {
    return m_Paused;
  }

  /**
   * Resumes the execution.
   */
  public void resumeExecution() {
    if (isPaused()) {
      m_Paused = false;
      try {
	synchronized(this) {
	  notifyAll();
	}
      }
      catch (Exception e) {
	// ignored
      }
    }
  }

  /**
   * Stops the execution.
   */
  public void stopExecution() {
    int		i;

    m_Stopping = true;

    // resume first?
    if (isPaused())
      resumeExecution();

    if (isDebugOn())
      debug("stop called");

    for (i = m_ControlActor.size() - 1; i >= 0; i--) {
      if (!m_ControlActor.get(i).getSkip())
	m_ControlActor.get(i).stopExecution();
    }

    m_Stopped  = true;
    m_Stopping = false;
  }

  /**
   * Returns whether the execution was stopped.
   *
   * @return		true if execution was stopped
   */
  public boolean isStopped() {
    return m_Stopped;
  }

  /**
   * Returns whether the execution is being stopped.
   *
   * @return		true if the stop of the execution has been initiated
   */
  public boolean isStopping() {
    return m_Stopping;
  }

  /**
   * Checks whether the director has finished. Default implementation
   * always returns true.
   *
   * @return		true
   */
  public boolean isFinished() {
    return true;
  }

  /**
   * Cleans up data structures, frees up memory.
   */
  public void cleanUp() {
  }

  /**
   * Returns a string representation of the director.
   *
   * @return		a string representation
   */
  public String toString() {
    return getClass().getName() + "/" + hashCode() + ": " + ((m_ControlActor == null) ? "--" : m_ControlActor.getFullName());
  }
}
