/*
 * Count.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;

import java.util.Hashtable;

import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Counts the number of tokens that pass through and tees off the current count every n-th token.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Count
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-progress (property: showProgress)
 * &nbsp;&nbsp;&nbsp;If set to true, progress information will be output to stdout ('.').
 * </pre>
 *
 * <pre>-stop-on-errors (property: stopOnErrors)
 * &nbsp;&nbsp;&nbsp;If set to true, errors (like exceptions) will stop the flow execution; otherwise
 * &nbsp;&nbsp;&nbsp;it is attempted to continue.
 * </pre>
 *
 * <pre>-tee &lt;adams.flow.core.AbstractActor [options]&gt; (property: teeActor)
 * &nbsp;&nbsp;&nbsp;The actor to siphon-off the tokens to.
 * &nbsp;&nbsp;&nbsp;default: adams.flow.sink.Null
 * </pre>
 *
 * <pre>-min &lt;int&gt; (property: minimum)
 * &nbsp;&nbsp;&nbsp;The minimum number of tokens to count before activating (-1 to disable).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -1
 * </pre>
 *
 * <pre>-max &lt;int&gt; (property: maximum)
 * &nbsp;&nbsp;&nbsp;The maximum number of tokens to count before de-activating (-1 to disable
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -1
 * </pre>
 *
 * <pre>-interval &lt;int&gt; (property: interval)
 * &nbsp;&nbsp;&nbsp;The number of tokens (or multiples) to count before teeing-off the current
 * &nbsp;&nbsp;&nbsp;count.
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -1
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2391 $
 */
public class Count
  extends Tee {

  /** for serialization. */
  private static final long serialVersionUID = -3408173173526076280L;

  /** the key for storing the current counter in the backup. */
  public final static String BACKUP_CURRENT = "current";

  /** the minimum number of tokens to process before activating. */
  protected int m_Minimum;

  /** the maximum number of tokens to process before de-activating. */
  protected int m_Maximum;

  /** the number of tokens after which to tee-off token. */
  protected int m_Interval;

  /** the current count. */
  protected int m_Current;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Counts the number of tokens that pass through and tees off the current count every n-th token.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "min", "minimum",
	    -1, -1, null);

    m_OptionManager.add(
	    "max", "maximum",
	    -1, -1, null);

    m_OptionManager.add(
	    "interval", "interval",
	    -1, -1, null);
  }

  /**
   * Sets the minimum number of counts before activating.
   *
   * @param value	the minimum (-1 to disable)
   */
  public void setMinimum(int value) {
    if (value < -1)
      value = -1;

    m_Minimum = value;
    reset();
  }

  /**
   * Returns the minimum of counts before activating.
   *
   * @return		the minimum (-1 if disabled)
   */
  public int getMinimum() {
    return m_Minimum;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String minimumTipText() {
    return "The minimum number of tokens to count before activating (-1 to disable).";
  }

  /**
   * Sets the maximum number of counts before de-activating.
   *
   * @param value	the maximum (-1 to disable)
   */
  public void setMaximum(int value) {
    if (value < -1)
      value = -1;

    m_Maximum = value;
    reset();
  }

  /**
   * Returns the maximum of counts before de-activating.
   *
   * @return		the maximum (-1 if disabled)
   */
  public int getMaximum() {
    return m_Maximum;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String maximumTipText() {
    return "The maximum number of tokens to count before de-activating (-1 to disable).";
  }

  /**
   * Sets the number of tokens after which to tee off the current count.
   *
   * @param value	the number
   */
  public void setInterval(int value) {
    if (value < -1)
      value = -1;

    m_Interval = value;
    reset();
  }

  /**
   * Returns the number of tokens after which to tee off the current count.
   *
   * @return		the number
   */
  public int getInterval() {
    return m_Interval;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String intervalTipText() {
    return "The number of tokens (or multiples) to count before teeing-off the current count.";
  }

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_CURRENT);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    result.put(BACKUP_CURRENT, m_Current);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_CURRENT)) {
      m_Current = (Integer) state.get(BACKUP_CURRENT);
      state.remove(BACKUP_CURRENT);
    }

    super.restoreState(state);
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_Current = 0;
  }

  /**
   * Checks whether we can process the token.
   *
   * @return		true if token can be processed
   */
  protected boolean canFire() {
    boolean	result;

    result = true;
    if (result && (m_Minimum > -1) && (m_Current < m_Minimum))
      result = false;
    if (result && (m_Maximum > -1) && (m_Current > m_Maximum))
      result = false;
    if (result && (m_Interval > -1) && (m_Current % m_Interval != 0))
      result = false;

    return result;
  }

  /**
   * The method that accepts the input token and then processes it.
   *
   * @param token	the token to accept and process
   */
  public void input(Token token) {
    m_Current++;

    super.input(token);
  }

  /**
   * Returns whether the token can be processed in the tee actor.
   *
   * @param token	the token to process
   * @return		true if token can be processed
   */
  protected boolean canProcessInput(Token token) {
    return (super.canProcessInput(token) && canFire());
  }

  /**
   * Creates the token to tee-off.
   *
   * @param token	the input token
   * @return		the token to tee-off
   */
  protected Token createTeeToken(Token token) {
    return new Token(new Integer(m_Current));
  }
}
