/*
 *    AbstractScriptActor.java
 *    Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.flow.core;

import adams.core.io.PlaceholderFile;

/**
 * Abstract ancestor for actors that execute external scripts.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2638 $
 */
public abstract class AbstractScriptActor
  extends AbstractActor {

  /** for serialization. */
  private static final long serialVersionUID = -8187233244973711251L;

  /** the Groovy module. */
  protected PlaceholderFile m_ScriptFile;

  /** the options for the Groovy module. */
  protected String m_ScriptOptions;

  /** the loaded script object. */
  protected transient Object m_ScriptObject;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "script", "scriptFile",
	    new PlaceholderFile("."));

    m_OptionManager.add(
	    "options", "scriptOptions",
	    "");
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("scriptFile");

    if (variable != null)
      return variable;
    else
      return new String(m_ScriptFile + " " + m_ScriptOptions).trim();
  }

  /**
   * Sets the Groovy module.
   *
   * @param value 	the Groovy module
   */
  public void setScriptFile(PlaceholderFile value) {
    m_ScriptFile = value;
    reset();
  }

  /**
   * Gets the Groovy module.
   *
   * @return 		the Groovy module
   */
  public PlaceholderFile getScriptFile() {
    return m_ScriptFile;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   * 			displaying in the explorer/experimenter gui
   */
  public String scriptFileTipText() {
    return "The script file to load and execute.";
  }

  /**
   * Sets the script options.
   *
   * @param value 	the options
   */
  public void setScriptOptions(String value) {
    m_ScriptOptions = value;
    reset();
  }

  /**
   * Gets the script options.
   *
   * @return 		the options
   */
  public String getScriptOptions() {
    return m_ScriptOptions;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   * 			displaying in the explorer/experimenter gui
   */
  public String scriptOptionsTipText() {
    return "The options for the script.";
  }

  /**
   * Loads the scripts object and sets its options.
   *
   * @return		null if OK, otherwise the error message
   */
  protected abstract String loadScriptObject();

  /**
   * Checks the script object.
   *
   * @return		null if OK, otherwise the error message
   */
  protected abstract String checkScriptObject();

  /**
   * Tries to initialize the scripts object, sets its options and performs
   * some checks.
   *
   * @return		null if OK, otherwise the error message
   */
  protected String initScriptObject() {
    String	result;

    result = loadScriptObject();
    if (result == null)
      result = checkScriptObject();

    return result;
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null)
      result = initScriptObject();

    return result;
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void wrapUp() {
    super.wrapUp();

    m_ScriptObject = null;
  }
}
