/*
 * SequencePlotter.java
 * Copyright (C) 2009-2012 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.sink;

import adams.data.sequence.XYSequence;
import adams.data.sequence.XYSequencePoint;
import adams.data.sequence.XYSequencePointComparator.Comparison;
import adams.flow.container.SequencePlotterContainer;
import adams.flow.core.Token;
import adams.flow.sink.sequenceplotter.MarkerPaintlet;
import adams.flow.sink.sequenceplotter.NoMarkers;
import adams.flow.sink.sequenceplotter.SequencePlotterPanel;
import adams.gui.core.BasePanel;
import adams.gui.visualization.core.AbstractColorProvider;
import adams.gui.visualization.core.AxisPanelOptions;
import adams.gui.visualization.core.DefaultColorProvider;
import adams.gui.visualization.core.axis.Type;
import adams.gui.visualization.core.plot.Axis;
import adams.gui.visualization.sequence.AbstractXYSequencePaintlet;
import adams.gui.visualization.sequence.XYSequenceContainer;
import adams.gui.visualization.sequence.XYSequenceContainerManager;
import adams.gui.visualization.sequence.XYSequenceStickPaintlet;

/**
 <!-- globalinfo-start -->
 * Actor that plots sequences over time.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.container.SequencePlotterContainer<br/>
 * <p/>
 * Container information:<br/>
 * - adams.flow.container.SequencePlotterContainer: PlotName, X, Y, IsMarker
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SequencePlotter
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-width &lt;int&gt; (property: width)
 * &nbsp;&nbsp;&nbsp;The width of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 800
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 * <pre>-height &lt;int&gt; (property: height)
 * &nbsp;&nbsp;&nbsp;The height of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 350
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 * <pre>-x &lt;int&gt; (property: x)
 * &nbsp;&nbsp;&nbsp;The X position of the dialog (&gt;=0: absolute, -1: left, -2: center, -3: right
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 *
 * <pre>-y &lt;int&gt; (property: y)
 * &nbsp;&nbsp;&nbsp;The Y position of the dialog (&gt;=0: absolute, -1: top, -2: center, -3: bottom
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 *
 * <pre>-writer &lt;adams.gui.print.JComponentWriter&gt; (property: writer)
 * &nbsp;&nbsp;&nbsp;The writer to use for generating the graphics output.
 * &nbsp;&nbsp;&nbsp;default: adams.gui.print.NullWriter
 * </pre>
 *
 * <pre>-comparison &lt;X|Y|X_AND_Y&gt; (property: comparisonType)
 * &nbsp;&nbsp;&nbsp;The type of comparison to use for the data points of the sequence.
 * &nbsp;&nbsp;&nbsp;default: X
 * </pre>
 *
 * <pre>-paintlet &lt;adams.gui.visualization.sequence.AbstractXYSequencePaintlet&gt; (property: paintlet)
 * &nbsp;&nbsp;&nbsp;The paintlet to use for painting the data.
 * &nbsp;&nbsp;&nbsp;default: adams.gui.visualization.sequence.XYSequenceStickPaintlet
 * </pre>
 *
 * <pre>-marker-paintlet &lt;adams.flow.sink.sequenceplotter.MarkerPaintlet&gt; (property: markerPaintlet)
 * &nbsp;&nbsp;&nbsp;The marker paintlet to use for painting marker overlays.
 * &nbsp;&nbsp;&nbsp;default: adams.flow.sink.sequenceplotter.NoMarkers
 * </pre>
 *
 * <pre>-color-provider &lt;adams.gui.visualization.core.AbstractColorProvider&gt; (property: colorProvider)
 * &nbsp;&nbsp;&nbsp;The color provider in use for generating the colors for the various plots.
 * &nbsp;&nbsp;&nbsp;default: adams.gui.visualization.core.DefaultColorProvider
 * </pre>
 *
 * <pre>-title &lt;java.lang.String&gt; (property: title)
 * &nbsp;&nbsp;&nbsp;The title for the border around the plot.
 * &nbsp;&nbsp;&nbsp;default: X-Y Sequences
 * </pre>
 *
 * <pre>-axis-x &lt;adams.gui.visualization.core.AxisPanelOptions&gt; (property: axisX)
 * &nbsp;&nbsp;&nbsp;The setup for the X axis.
 * &nbsp;&nbsp;&nbsp;default: adams.gui.visualization.core.AxisPanelOptions -label x -width 40 -custom-format 0.000
 * </pre>
 *
 * <pre>-axis-y &lt;adams.gui.visualization.core.AxisPanelOptions&gt; (property: axisY)
 * &nbsp;&nbsp;&nbsp;The setup for the Y axis.
 * &nbsp;&nbsp;&nbsp;default: adams.gui.visualization.core.AxisPanelOptions -label y -num-ticks 10 -width 40 -custom-format 0.0
 * </pre>
 *
 * <pre>-limit &lt;int&gt; (property: limit)
 * &nbsp;&nbsp;&nbsp;The limit on data points in the sequences, does not apply to markers; -1
 * &nbsp;&nbsp;&nbsp;is unlimited.
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -1
 * </pre>
 *
 * <pre>-update-interval &lt;int&gt; (property: updateInterval)
 * &nbsp;&nbsp;&nbsp;Specifies the number of tokens after which the display is being updated.
 * &nbsp;&nbsp;&nbsp;default: 1
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4220 $
 */
public class SequencePlotter
  extends AbstractGraphicalDisplay {

  /** for serialization. */
  private static final long serialVersionUID = 3238389451500168650L;

  /** the comparison to use for the X/Y points. */
  protected Comparison m_ComparisonType;

  /** the paintlet to use for painting the XY data. */
  protected AbstractXYSequencePaintlet m_Paintlet;

  /** the paintlet to use for painting markers. */
  protected MarkerPaintlet m_MarkerPaintlet;

  /** the color provider to use. */
  protected AbstractColorProvider m_ColorProvider;

  /** the title. */
  protected String m_Title;

  /** the options for the X axis. */
  protected AxisPanelOptions m_AxisX;

  /** the options for the Y axis. */
  protected AxisPanelOptions m_AxisY;

  /** the window of data points to display per sequence. */
  protected int m_Limit;

  /** the interval of tokens processed after which to update the display. */
  protected int m_UpdateInterval;

  /** the number of tokens received. */
  protected int m_NumTokensAccepted;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Actor that plots sequences over time.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "comparison", "comparisonType",
	     Comparison.X);

    m_OptionManager.add(
	    "paintlet", "paintlet",
	    new XYSequenceStickPaintlet());

    m_OptionManager.add(
	    "marker-paintlet", "markerPaintlet",
	    new NoMarkers());

    m_OptionManager.add(
	    "color-provider", "colorProvider",
	    new DefaultColorProvider());

    m_OptionManager.add(
	    "title", "title",
	    "X-Y Sequences");

    m_OptionManager.add(
	    "axis-x", "axisX",
	    getDefaultAxisX());

    m_OptionManager.add(
	    "axis-y", "axisY",
	    getDefaultAxisY());

    m_OptionManager.add(
	    "limit", "limit",
	    -1, -1, null);

    m_OptionManager.add(
	    "update-interval", "updateInterval",
	    1, 1, null);
  }

  /**
   * Resets the actor.
   */
  protected void reset() {
    super.reset();

    m_NumTokensAccepted = 0;
  }

  /**
   * Returns the default width for the dialog.
   *
   * @return		the default width
   */
  protected int getDefaultWidth() {
    return 800;
  }

  /**
   * Returns the default height for the dialog.
   *
   * @return		the default height
   */
  protected int getDefaultHeight() {
    return 350;
  }

  /**
   * Sets the type of comparison to use for the X/Y points.
   *
   * @param value	the type of comparison to use
   */
  public void setComparisonType(Comparison value) {
    m_ComparisonType = value;
    reset();
  }

  /**
   * Returns the type of comparison currently in use for the X/Y points.
   *
   * @return		the type of comparison
   */
  public Comparison getComparisonType() {
    return m_ComparisonType;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String comparisonTypeTipText() {
    return "The type of comparison to use for the data points of the sequence.";
  }

  /**
   * Sets the paintlet to use.
   *
   * @param value 	the paintlet
   */
  public void setPaintlet(AbstractXYSequencePaintlet value) {
    m_Paintlet = value;
    reset();
  }

  /**
   * Returns the paintlet to use.
   *
   * @return 		the paintlet
   */
  public AbstractXYSequencePaintlet getPaintlet() {
    return m_Paintlet;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String paintletTipText() {
    return "The paintlet to use for painting the data.";
  }

  /**
   * Sets the marker paintlet to use.
   *
   * @param value 	the marker paintlet
   */
  public void setMarkerPaintlet(MarkerPaintlet value) {
    m_MarkerPaintlet = value;
    reset();
  }

  /**
   * Returns the marker paintlet to use.
   *
   * @return 		the marker paintlet
   */
  public MarkerPaintlet getMarkerPaintlet() {
    return m_MarkerPaintlet;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String markerPaintletTipText() {
    return "The marker paintlet to use for painting marker overlays.";
  }

  /**
   * Sets the color provider to use.
   *
   * @param value 	the color provider
   */
  public void setColorProvider(AbstractColorProvider value) {
    m_ColorProvider = value;
    reset();
  }

  /**
   * Returns the color provider in use.
   *
   * @return 		the color provider
   */
  public AbstractColorProvider getColorProvider() {
    return m_ColorProvider;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String colorProviderTipText() {
    return "The color provider in use for generating the colors for the various plots.";
  }

  /**
   * Returns the setup for the X axis.
   *
   * @return 		the setup
   */
  protected AxisPanelOptions getDefaultAxisX() {
    AxisPanelOptions	result;

    result = new AxisPanelOptions();
    result.setType(Type.ABSOLUTE);
    result.setLabel("x");
    result.setCustomFormat("0.000");
    result.setShowGridLines(true);
    result.setShowOnlyMajorGridLines(false);
    result.setLengthTicks(4);
    result.setNumTicks(20);
    result.setWidth(40);
    result.setTopMargin(0.0);
    result.setBottomMargin(0.0);

    return result;
  }

  /**
   * Returns the setup for the Y axis.
   *
   * @return 		the setup
   */
  protected AxisPanelOptions getDefaultAxisY() {
    AxisPanelOptions	result;

    result = new AxisPanelOptions();
    result.setType(Type.ABSOLUTE);
    result.setLabel("y");
    result.setCustomFormat("0.0");
    result.setShowGridLines(true);
    result.setShowOnlyMajorGridLines(false);
    result.setLengthTicks(4);
    result.setNumTicks(10);
    result.setWidth(40);
    result.setTopMargin(0.0);
    result.setBottomMargin(0.0);

    return result;
  }

  /**
   * Sets the setup for the X axis.
   *
   * @param value 	the setup
   */
  public void setAxisX(AxisPanelOptions value) {
    m_AxisX = value;
    reset();
  }

  /**
   * Returns the setup for the X axis.
   *
   * @return 		the setup
   */
  public AxisPanelOptions getAxisX() {
    return m_AxisX;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String axisXTipText() {
    return "The setup for the X axis.";
  }

  /**
   * Sets the setup for the Y axis.
   *
   * @param value 	the setup
   */
  public void setAxisY(AxisPanelOptions value) {
    m_AxisY = value;
    reset();
  }

  /**
   * Returns the setup for the Y axis.
   *
   * @return 		the setup
   */
  public AxisPanelOptions getAxisY() {
    return m_AxisY;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String axisYTipText() {
    return "The setup for the Y axis.";
  }

  /**
   * Sets the title for border around the plot.
   *
   * @param value 	the title
   */
  public void setTitle(String value) {
    m_Title = value;
    reset();
  }

  /**
   * Returns the title for border around the plot.
   *
   * @return 		the title
   */
  public String getTitle() {
    return m_Title;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String titleTipText() {
    return "The title for the border around the plot.";
  }

  /**
   * Sets the limit on the number of data points per sequence.
   *
   * @param value 	the limit
   */
  public void setLimit(int value) {
    if (value >= -1) {
      m_Limit = value;
      reset();
    }
    else {
      getSystemErr().println(
	  "Limit must be >= -1, provided: " + value);
    }
  }

  /**
   * Returns the limit on the number of data points per sequence.
   *
   * @return 		the limit
   */
  public int getLimit() {
    return m_Limit;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String limitTipText() {
    return "The limit on data points in the sequences, does not apply to markers; -1 is unlimited.";
  }

  /**
   * Sets the number of tokens after which the display is being updated.
   *
   * @param value 	the number of tokens
   */
  public void setUpdateInterval(int value) {
    if (value >= 1) {
      m_UpdateInterval = value;
      reset();
    }
    else {
      getSystemErr().println(
	  "Update interval must be >= 1, provided: " + value);
    }
  }

  /**
   * Returns the number of tokens after which the display is being updated.
   *
   * @return 		the number of tokens
   */
  public int getUpdateInterval() {
    return m_UpdateInterval;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String updateIntervalTipText() {
    return "Specifies the number of tokens after which the display is being updated.";
  }

  /**
   * Clears the content of the panel.
   */
  public void clearPanel() {
    ((SequencePlotterPanel) m_Panel).getContainerManager().clear();
    ((SequencePlotterPanel) m_Panel).getMarkerContainerManager().clear();
  }

  /**
   * Creates the panel to display in the dialog.
   *
   * @return		the panel
   */
  protected BasePanel newPanel() {
    SequencePlotterPanel	result;

    result = new SequencePlotterPanel(getTitle());
    result.setPaintlet(getPaintlet());
    result.setMarkerPaintlet(getMarkerPaintlet());
    m_AxisX.configure(result.getPlot(), Axis.BOTTOM);
    m_AxisY.configure(result.getPlot(), Axis.LEFT);
    result.setColorProvider(m_ColorProvider);

    return result;
  }

  /**
   * Creates a title for the dialog. Default implementation only returns
   * the full name of the actor.
   *
   * @return		the title of the dialog
   */
  protected String createTitle() {
    return "Sequence plot - " + getFullName();
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.container.SequencePlotterContainer.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{SequencePlotterContainer.class};
  }

  /**
   * Trims the sequences.
   *
   * @param manager	the sequence manager
   * @param plotName	the current plot name
   */
  protected void trimSequences(XYSequenceContainerManager manager, String plotName) {
    XYSequence		seq;

    seq = manager.get(manager.indexOf(plotName)).getData();
    synchronized(seq) {
      while (seq.size() > m_Limit)
	seq.toList().remove(0);
    }
  }

  /**
   * Displays the token (the panel and dialog have already been created at
   * this stage).
   *
   * @param token	the token to display
   */
  protected void display(Token token) {
    XYSequenceContainer		cont;
    XYSequenceContainerManager	manager;
    XYSequence			seq;
    XYSequencePoint		point;
    SequencePlotterContainer	plotCont;
    String			plotName;
    Double			x;
    Double			y;
    boolean			isMarker;

    m_NumTokensAccepted++;

    plotCont = (SequencePlotterContainer) token.getPayload();
    plotName = (String) plotCont.getValue(SequencePlotterContainer.VALUE_PLOTNAME);
    x        = (Double) plotCont.getValue(SequencePlotterContainer.VALUE_X);
    y        = (Double) plotCont.getValue(SequencePlotterContainer.VALUE_Y);
    isMarker = (Boolean) plotCont.getValue(SequencePlotterContainer.VALUE_ISMARKER);

    if (isMarker)
      manager = ((SequencePlotterPanel) m_Panel).getMarkerContainerManager();
    else
      manager = ((SequencePlotterPanel) m_Panel).getContainerManager();
    manager.startUpdate();

    // find or create new plot
    if (manager.indexOf(plotName) == -1) {
      seq  = new XYSequence();
      seq.setComparison(m_ComparisonType);
      seq.setID(plotName);
      cont = manager.newContainer(seq);
      manager.add(cont);
    }
    else {
      cont = manager.get(manager.indexOf(plotName));
      seq  = cont.getData();
    }

    // create and add new point
    if (x == null) {
      if (seq.size() > 0)
	x = ((XYSequencePoint) seq.toList().get(seq.toList().size() - 1)).getX().doubleValue() + 1;
      else
	x = 1.0;
    }
    point = new XYSequencePoint("" + seq.size(), x, y);
    seq.add(point);

    // limit size of sequence?
    if (!isMarker && (m_Limit > -1))
      trimSequences(manager, plotName);

    manager.finishUpdate((m_NumTokensAccepted % m_UpdateInterval) == 0);
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void wrapUp() {
    if (m_UpdateInterval > 1) {
      ((SequencePlotterPanel) m_Panel).getMarkerContainerManager().finishUpdate();
      ((SequencePlotterPanel) m_Panel).getContainerManager().finishUpdate();
    }

    super.wrapUp();
  }
}
