/*
 * AbstractDataContainerFileImport.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.io.File;
import java.util.Hashtable;
import java.util.Vector;

import adams.core.io.PlaceholderFile;
import adams.data.container.DataContainer;
import adams.data.io.input.AbstractDataContainerReader;
import adams.db.AbstractDatabaseConnection;
import adams.db.DataProvider;
import adams.flow.core.Token;
import adams.flow.provenance.ActorType;
import adams.flow.provenance.Provenance;
import adams.flow.provenance.ProvenanceContainer;
import adams.flow.provenance.ProvenanceInformation;
import adams.flow.provenance.ProvenanceSupporter;

/**
 * Abstract ancestor for actors that import data containers.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3507 $
 * @param <T> the type of data that is imported
 */
public abstract class AbstractDataContainerFileImport<T extends DataContainer>
  extends AbstractDataProcessor
  implements ProvenanceSupporter {

  /** for serialization. */
  private static final long serialVersionUID = -3449734957975707303L;

  /** the key for storing the current ids in the backup. */
  public final static String BACKUP_IDS = "ids";

  /** the key for storing the current containers in the backup. */
  public final static String BACKUP_CONTAINERS = "containers";

  /** the reader to use for processing the containers. */
  protected AbstractDataContainerReader<T> m_Reader;

  /** the IDs of the containers that have been imported. */
  protected Vector<Integer> m_IDs;

  /** the containers that have been read. */
  protected Vector<T> m_Containers;

  /** whether to import the containers into the database. */
  protected boolean m_Import;

  /** whether to forward the containers instead of the IDs. */
  protected boolean m_Forward;

  /** the database connection. */
  protected adams.db.AbstractDatabaseConnection m_DatabaseConnection;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "reader", "reader",
	    getDefaultReader());

    m_OptionManager.add(
	    "forward", "forward",
	    false);

    m_OptionManager.add(
	    "import", "import",
	    true);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_DatabaseConnection = getDefaultDatabaseConnection();
  }

  /**
   * Returns the default database connection.
   *
   * @return 		the default database connection
   */
  protected abstract AbstractDatabaseConnection getDefaultDatabaseConnection();

  /**
   * Returns the default reader for loading the data.
   *
   * @return		the default reader
   */
  protected abstract AbstractDataContainerReader<T> getDefaultReader();

  /**
   * Sets the reader to use.
   *
   * @param value	the reader
   */
  public void setReader(AbstractDataContainerReader value) {
    m_Reader = value;
    reset();
  }

  /**
   * Returns the reader in use.
   *
   * @return		the reader
   */
  public AbstractDataContainerReader getReader() {
    return m_Reader;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String readerTipText() {
    return "The reader to use for importing the containers.";
  }

  /**
   * Sets whether to forward the containers instead of the IDs.
   *
   * @param value	if true then the containers are forwarded
   */
  public void setForward(boolean value) {
    m_Forward = value;
    reset();
  }

  /**
   * Returns whether to forward the containers or the IDs.
   *
   * @return		true if the containers are forwarded
   */
  public boolean getForward() {
    return m_Forward;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String forwardTipText() {
    return
        "If set to true then the containers are forwarded instead of the IDs.";
  }

  /**
   * Sets whether to import the containers into the database.
   *
   * @param value	if true then the containers are imported
   */
  public void setImport(boolean value) {
    m_Import = value;
    reset();
  }

  /**
   * Returns whether to import the containers into the database.
   *
   * @return		true if the containers are imported
   */
  public boolean getImport() {
    return m_Import;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String importTipText() {
    return
        "If set to true then the containers are imported into the database.";
  }

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_IDS);
    pruneBackup(BACKUP_CONTAINERS);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    result.put(BACKUP_IDS, m_IDs);
    result.put(BACKUP_CONTAINERS, m_Containers);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_IDS)) {
      m_IDs = (Vector<Integer>) state.get(BACKUP_IDS);
      state.remove(BACKUP_IDS);
    }

    if (state.containsKey(BACKUP_CONTAINERS)) {
      m_Containers = (Vector<T>) state.get(BACKUP_CONTAINERS);
      state.remove(BACKUP_CONTAINERS);
    }

    super.restoreState(state);
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_IDs        = new Vector<Integer>();
    m_Containers = new Vector<T>();
  }

  /**
   * Returns the report provider to use for writing the reports to the database.
   *
   * @return		the provider to use
   */
  protected abstract DataProvider<T> getDataProvider();

  /**
   * Determines the database connection in the flow.
   *
   * @return		the database connection to use
   */
  protected abstract adams.db.AbstractDatabaseConnection getDatabaseConnection();

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null)
      m_DatabaseConnection = getDatabaseConnection();

    return result;
  }

  /**
   * Processes the given data.
   *
   * @param file	the file/dir to process
   * @return		true if everything went alright
   */
  protected boolean processData(File file) {
    boolean		result;
    Vector<T> 		conts;
    int			i;
    int			id;
    DataProvider<T>	provider;

    result = false;

    // setup reader
    m_Reader.setInput(new PlaceholderFile(file));
    if (isDebugOn())
      debug("Attempting to load '" + file + "'");

    // read data
    try {
      conts = m_Reader.read();
      if (isDebugOn())
	debug(conts.size() + " containers read");
      m_Reader.cleanUp();
    }
    catch (Exception e) {
	m_ProcessError = "Error reading '" + file + "': " + e;
	return result;
    }

    // import data into database
    if (conts.size() > 0) {
      result   = true;
      provider = null;
      if (m_Import)
	provider = getDataProvider();
      for (i = 0; i < conts.size(); i++) {
	try {
	  if (m_Import) {
	    id = provider.add(conts.get(i));
	    if (isDebugOn())
	      debug("Container (" + conts.get(i) + ") imported: " + id);
	    if (m_Forward)
	      m_Containers.add(conts.get(i));
	    else
	      m_IDs.add(id);
	  }
	  else if (m_Forward) {
	    m_Containers.add(conts.get(i));
	  }
	}
	catch (Exception e) {
	  result = false;
	  getSystemErr().println("Error importing container (" + conts.get(i) + "): " + e);
	}
      }
    }

    return result;
  }

  /**
   * Returns the data container class in use.
   *
   * @return		the class
   */
  protected abstract Class getDataContainerClass();

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.Integer.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    if (m_Forward)
      return new Class[]{getDataContainerClass()};
    else if (m_Import)
      return new Class[]{Integer.class};
    else
      return new Class[]{};
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    result = null;

    if (m_Forward) {
      result = new Token(m_Containers.get(0));
      m_Containers.remove(0);
      updateProvenance(result);
    }
    else if (m_Import) {
      result = new Token(m_IDs.get(0));
      m_IDs.remove(0);
    }

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    if (m_Forward)
      return (m_Containers.size() > 0);
    else if (m_Import)
      return (m_IDs.size() > 0);
    else
      return false;
  }

  /**
   * Updates the provenance information in the provided container.
   *
   * @param cont	the provenance container to update
   */
  public void updateProvenance(ProvenanceContainer cont) {
    if (Provenance.getSingleton().isEnabled())
      cont.addProvenance(new ProvenanceInformation(ActorType.DATAGENERATOR, this, ((Token) cont).getPayload().getClass()));
  }

  /**
   * Cleans up after the execution has finished. Graphical output is left
   * untouched.
   */
  public void wrapUp() {
    m_DatabaseConnection = null;

    super.wrapUp();
  }
}
