/*
 * MakePlotContainer.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import adams.flow.container.SequencePlotterContainer;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Creates a named container for the SequencePlotter actor.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Double<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Double[]<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Integer<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Integer[]<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.container.SequencePlotterContainer<br/>
 * <p/>
 * Container information:<br/>
 * - adams.flow.container.SequencePlotterContainer: PlotName, X, Y, IsMarker
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: MakePlotContainer
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-plot-name &lt;java.lang.String&gt; (property: plotName)
 * &nbsp;&nbsp;&nbsp;The name for the plot.
 * &nbsp;&nbsp;&nbsp;default: Plot
 * </pre>
 *
 * <pre>-is-marker (property: isMarker)
 * &nbsp;&nbsp;&nbsp;If enabled, a container for a marker is created.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3932 $
 */
public class MakePlotContainer
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = 2488434486963278287L;

  /** the name of the plot. */
  protected String m_PlotName;

  /** whether the plot container is for a marker. */
  protected boolean m_IsMarker;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Creates a named container for the SequencePlotter actor.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "plot-name", "plotName",
	    "Plot");

    m_OptionManager.add(
	    "is-marker", "isMarker",
	    false);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("plotName");
    if (variable != null)
      result = variable;
    else if ((m_PlotName != null) && (m_PlotName.length() > 0))
      result = m_PlotName;
    else
      result = "-unnamed-";

    if (m_IsMarker)
      result += " (marker)";

    return result;
  }

  /**
   * Sets the plot name.
   *
   * @param value	the name
   */
  public void setPlotName(String value) {
    m_PlotName = value;
    reset();
  }

  /**
   * Returns the current plot name.
   *
   * @return		the name
   */
  public String getPlotName() {
    return m_PlotName;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String plotNameTipText() {
    return "The name for the plot.";
  }

  /**
   * Sets the whether it is a marker.
   *
   * @param value	true if it is a marker
   */
  public void setIsMarker(boolean value) {
    m_IsMarker = value;
    reset();
  }

  /**
   * Returns whether it is a marker.
   *
   * @return		true if it is a marker
   */
  public boolean getIsMarker() {
    return m_IsMarker;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String isMarkerTipText() {
    return "If enabled, a container for a marker is created.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.Double.class, java.lang.Double[].class, java.lang.Integer.class, java.lang.Integer[].class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Double.class, Double[].class, Integer.class, Integer[].class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String			result;
    SequencePlotterContainer	cont;

    result = null;

    cont = null;
    if (m_InputToken.getPayload() instanceof Double) {
      cont = new SequencePlotterContainer(
	  m_PlotName,
	  (Double) m_InputToken.getPayload(),
	  m_IsMarker);
    }
    else if (m_InputToken.getPayload() instanceof Double[]) {
      if (((Double[]) m_InputToken.getPayload()).length == 2) {
	cont = new SequencePlotterContainer(
	    m_PlotName,
	    ((Double[]) m_InputToken.getPayload())[0],
	    ((Double[]) m_InputToken.getPayload())[1],
	    m_IsMarker);
      }
      else {
	result =
	    "Double array must have length 2 (provided: "
	    + ((Double[]) m_InputToken.getPayload()).length + ")!";
      }
    }
    else if (m_InputToken.getPayload() instanceof Integer) {
      cont = new SequencePlotterContainer(
	  m_PlotName,
	  ((Integer) m_InputToken.getPayload()).doubleValue(),
	  m_IsMarker);
    }
    else if (m_InputToken.getPayload() instanceof Integer[]) {
      if (((Integer[]) m_InputToken.getPayload()).length == 2) {
	cont = new SequencePlotterContainer(
	    m_PlotName,
	    ((Integer[]) m_InputToken.getPayload())[0].doubleValue(),
	    ((Integer[]) m_InputToken.getPayload())[1].doubleValue(),
	    m_IsMarker);
      }
      else {
	result =
	    "Double array must have length 2 (provided: "
	    + ((Double[]) m_InputToken.getPayload()).length + ")!";
      }
    }

    if (cont != null)
      m_OutputToken = new Token(cont);

    return result;
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.container.SequencePlotterContainer.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{SequencePlotterContainer.class};
  }
}
