/**
 * AbstractFlowScriptlet.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.scripting;

import java.util.Vector;

import adams.core.Utils;
import adams.core.option.OptionUtils;
import adams.data.container.DataContainer;
import adams.flow.control.SubProcess;
import adams.flow.core.ActorUtils;
import adams.flow.core.Token;
import adams.gui.visualization.container.ContainerManager;
import adams.gui.visualization.container.VisibilityContainerManager;

/**
 * Ancestor for scriptlets that run flows.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3020 $
 */
public abstract class AbstractFlowScriptlet
  extends AbstractDataContainerUpdatingScriptlet {

  /** for serialization. */
  private static final long serialVersionUID = -2467186252442407554L;

  /**
   * Returns a one-line listing of the options of the action.
   *
   * @return		the options or null if none
   */
  protected String getOptionsDescription() {
    return "<filename>";
  }

  /**
   * Processes the options.
   *
   * @param options	additional/optional options for the action
   * @param overlay	whether to overlay the data
   * @return		null if no error, otherwise error message
   * @throws Exception 	if something goes wrong
   */
  public String process(String options, boolean overlay) throws Exception {
    SubProcess			actor;
    SubProcess			runActor;
    String			result;
    String[]			list;
    int				i;
    ContainerManager		manager;
    Vector<DataContainer>	runInput;
    Vector<DataContainer>	runOutput;
    Class[]			flowClasses;
    Vector<String>		errors;

    manager = getDataContainerPanel().getContainerManager();

    if (((VisibilityContainerManager) manager).countVisible() == 0)
      return "No visible chromatograms!";

    // instantiate flow
    list    = OptionUtils.splitOptions(options);
    if (list.length != 1)
      return "Needs a single filename as parameter!";
    errors = new Vector<String>();
    actor  = (SubProcess) ActorUtils.read(list[0], errors);
    if (!errors.isEmpty())
      return "Failed to load actor from '" + list[0] + "':\n" + Utils.flatten(errors, "\n");
    if (actor == null)
      return "Could not instantiate actor from '" + list[0] + "'!";

    // does flow accept chromatograms?
    flowClasses = actor.accepts();
    result      = "Flow '" + list[0] + "' does not accept single chromatograms!";
    for (i = 0; i < flowClasses.length; i++) {
      if (flowClasses[i] == getOwner().getRequiredFlowClass()) {
	result = null;
	break;
      }
    }
    if (result != null)
      return result;

    // does flow generate chromatograms?
    flowClasses = actor.generates();
    result      = "Flow '" + list[0] + "' does not generate single chromatograms!";
    for (i = 0; i < flowClasses.length; i++) {
      if (flowClasses[i] == getOwner().getRequiredFlowClass()) {
	result = null;
	break;
      }
    }
    if (result != null)
      return result;

    // get data that is to be run through flow
    runInput = new Vector<DataContainer>();
    for (i = 0; i < manager.count(); i++) {
      if (((VisibilityContainerManager) manager).isVisible(i))
	runInput.add((DataContainer) manager.get(i).getPayload());
    }

    // process data
    runOutput = new Vector<DataContainer>();
    for (i = 0; i < runInput.size(); i++) {
      showStatus("Passing data through flow " + (i+1) + "/" + ((VisibilityContainerManager) manager).countVisible());

      runActor = (SubProcess) actor.shallowCopy(true);
      runActor.setUp();
      runActor.input(new Token(runInput.get(i)));
      result = runActor.execute();
      if (result == null)
	runOutput.add((DataContainer) runActor.output().getPayload());
      runActor.wrapUp();
      runActor.destroy();

      if (result != null)
	break;
    }
    actor.destroy();
    showStatus("");
    if (result != null)
      return result;

    // undo
    addUndoPoint("Saving undo data...", "Flow: " + list[0]);

    // update data
    updateDataContainers(runOutput, overlay);

    return null;
  }
}
