/**
 * FileContentBrowserPanel.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.tools;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.Arrays;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListModel;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import adams.core.CleanUpHandler;
import adams.core.Properties;
import adams.core.io.FileUtils;
import adams.core.io.PlaceholderFile;
import adams.env.Environment;
import adams.env.PreviewBrowserPanelDefinition;
import adams.gui.chooser.DirectoryChooserPanel;
import adams.gui.core.BasePanel;
import adams.gui.core.BaseScrollPane;
import adams.gui.core.BaseSplitPane;
import adams.gui.core.SearchPanel;
import adams.gui.core.SearchableBaseList;
import adams.gui.core.SearchPanel.LayoutType;
import adams.gui.event.SearchEvent;
import adams.gui.event.SearchListener;
import adams.gui.tools.previewbrowser.AbstractArchiveHandler;
import adams.gui.tools.previewbrowser.AbstractContentHandler;
import adams.gui.tools.previewbrowser.CreatingPreviewPanel;
import adams.gui.tools.previewbrowser.NoPreviewAvailablePanel;

/**
 * Allows the browsing of files and previewing the content.
 * Also works for archives.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3888 $
 */
public class PreviewBrowserPanel
  extends BasePanel {

  /** for serialization. */
  private static final long serialVersionUID = 5284765971012530865L;

  /** the name of the props file. */
  public final static String FILENAME = "PreviewBrowser.props";

  /** the prefix for the preferred content handler keys in the props file. */
  public final static String PREFIX_PREFERRED_CONTENT_HANDLER = "PreferredContentHandler-";

  /** the prefix for the preferred archive handler keys in the props file. */
  public final static String PREFIX_PREFERRED_ARCHIVE_HANDLER = "PreferredArchiveHandler-";

  /** the properties. */
  protected static Properties m_Properties;

  /** the split pane in use. */
  protected BaseSplitPane m_SplitPane;

  /** for selecting the directory to browse. */
  protected DirectoryChooserPanel m_PanelDir;

  /** the panel for browsing. */
  protected BasePanel m_PanelBrowsing;

  /** the panel with the local files. */
  protected BasePanel m_PanelLocalFiles;

  /** the list with the local files. */
  protected SearchableBaseList m_ListLocalFiles;

  /** the model for the local files. */
  protected DefaultListModel m_ModelLocalFiles;

  /** the search panel for the local files. */
  protected SearchPanel m_SearchLocalFiles;

  /** the panel with the archive files. */
  protected BasePanel m_PanelArchiveFiles;

  /** the list with the archive files. */
  protected SearchableBaseList m_ListArchiveFiles;

  /** the model for the archive files. */
  protected DefaultListModel m_ModelArchiveFiles;

  /** the search panel for the archive files. */
  protected SearchPanel m_SearchArchiveFiles;

  /** the panel for the content display. */
  protected BasePanel m_PanelContent;

  /** the panel for the view. */
  protected BasePanel m_PanelView;

  /** the panel with the content handlers. */
  protected BasePanel m_PanelContentHandlers;

  /** the combobox with the content handlers (if more than one available). */
  protected JComboBox m_ComboBoxContentHandlers;

  /** the model of the combobox. */
  protected DefaultComboBoxModel m_ModelContentHandlers;

  /** whether to ignore selections of the content handler combobox temporarily. */
  protected boolean m_IgnoreContentHandlerChanges;

  /** the panel with the archive handlers. */
  protected BasePanel m_PanelArchiveHandlers;

  /** the combobox with the archive handlers (if more than one available). */
  protected JComboBox m_ComboBoxArchiveHandlers;

  /** the model of the combobox. */
  protected DefaultComboBoxModel m_ModelArchiveHandlers;

  /** whether to ignore selections of the archive handler combobox temporarily. */
  protected boolean m_IgnoreArchiveHandlerChanges;

  /** the current archive handler. */
  protected AbstractArchiveHandler m_ArchiveHandler;

  /** the currently selected file. */
  protected File m_CurrentFile;

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_ArchiveHandler              = null;
    m_IgnoreContentHandlerChanges = false;
    m_IgnoreArchiveHandlerChanges = false;
    m_CurrentFile                 = null;
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    Properties		props;
    JPanel		panel;

    super.initGUI();

    props = getProperties();

    setLayout(new BorderLayout());

    m_SplitPane = new BaseSplitPane(BaseSplitPane.HORIZONTAL_SPLIT);
    m_SplitPane.setOneTouchExpandable(true);
    m_SplitPane.setDividerLocation(props.getInteger("DividerLocation", 200));
    add(m_SplitPane, BorderLayout.CENTER);

    // browsing
    m_PanelBrowsing = new BasePanel(new GridLayout(2, 1));
    m_SplitPane.setLeftComponent(m_PanelBrowsing);

    m_PanelLocalFiles = new BasePanel(new BorderLayout(0, 5));
    m_PanelBrowsing.add(m_PanelLocalFiles);
    m_PanelLocalFiles.setBorder(BorderFactory.createTitledBorder("Files"));

    m_PanelDir = new DirectoryChooserPanel(props.getString("InitialDir", "%h"));
    m_PanelDir.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
	refreshLocalFiles();
      }
    });
    m_PanelLocalFiles.add(m_PanelDir, BorderLayout.NORTH);

    m_ModelLocalFiles = new DefaultListModel();
    m_ListLocalFiles  = new SearchableBaseList(m_ModelLocalFiles);
    m_ListLocalFiles.addListSelectionListener(new ListSelectionListener() {
      public void valueChanged(ListSelectionEvent e) {
	displayLocalFile();
      }
    });
    m_PanelLocalFiles.add(new BaseScrollPane(m_ListLocalFiles), BorderLayout.CENTER);

    m_SearchLocalFiles = new SearchPanel(LayoutType.VERTICAL, false);
    m_SearchLocalFiles.addSearchListener(new SearchListener() {
      public void searchInitiated(SearchEvent e) {
	m_ListLocalFiles.search(e.getParameters().getSearchString(), e.getParameters().isRegExp());
      }
    });
    panel = new JPanel(new FlowLayout(FlowLayout.LEFT));
    panel.add(m_SearchLocalFiles);
    m_PanelLocalFiles.add(panel, BorderLayout.SOUTH);

    m_PanelArchiveFiles = new BasePanel(new BorderLayout());
    m_PanelBrowsing.add(m_PanelArchiveFiles);
    m_PanelArchiveFiles.setVisible(false);
    m_PanelArchiveFiles.setBorder(BorderFactory.createTitledBorder("Archive"));

    m_ModelArchiveFiles = new DefaultListModel();
    m_ListArchiveFiles  = new SearchableBaseList(m_ModelArchiveFiles);
    m_ListArchiveFiles.addListSelectionListener(new ListSelectionListener() {
      public void valueChanged(ListSelectionEvent e) {
	displayArchiveContent();
      }
    });
    m_PanelArchiveFiles.add(new BaseScrollPane(m_ListArchiveFiles), BorderLayout.CENTER);

    m_SearchArchiveFiles = new SearchPanel(LayoutType.VERTICAL, false);
    m_SearchArchiveFiles.addSearchListener(new SearchListener() {
      public void searchInitiated(SearchEvent e) {
	m_ListArchiveFiles.search(e.getParameters().getSearchString(), e.getParameters().isRegExp());
      }
    });
    panel = new JPanel(new FlowLayout(FlowLayout.LEFT));
    panel.add(m_SearchArchiveFiles);
    m_PanelArchiveFiles.add(panel, BorderLayout.SOUTH);

    m_PanelArchiveHandlers = new BasePanel(new FlowLayout(FlowLayout.LEFT));
    m_PanelArchiveHandlers.setVisible(false);
    m_PanelArchiveFiles.add(m_PanelArchiveHandlers, BorderLayout.SOUTH);

    m_ModelArchiveHandlers    = new DefaultComboBoxModel();
    m_ComboBoxArchiveHandlers = new JComboBox(m_ModelArchiveHandlers);
    m_ComboBoxArchiveHandlers.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	if (m_IgnoreArchiveHandlerChanges)
	  return;
        updatePreferredArchiveHandler();
        displayArchiveContent();
      }
    });
    m_PanelArchiveHandlers.add(new JLabel("Preferred handler"));
    m_PanelArchiveHandlers.add(m_ComboBoxArchiveHandlers);

    // content
    m_PanelContent = new BasePanel(new BorderLayout());
    m_SplitPane.setRightComponent(m_PanelContent);

    m_PanelView = new BasePanel(new BorderLayout());
    m_PanelContent.add(m_PanelView, BorderLayout.CENTER);

    m_PanelContentHandlers = new BasePanel(new FlowLayout(FlowLayout.LEFT));
    m_PanelContentHandlers.setVisible(false);
    m_PanelContent.add(m_PanelContentHandlers, BorderLayout.SOUTH);

    m_ModelContentHandlers    = new DefaultComboBoxModel();
    m_ComboBoxContentHandlers = new JComboBox(m_ModelContentHandlers);
    m_ComboBoxContentHandlers.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	if (m_IgnoreContentHandlerChanges)
	  return;
        updatePreferredContentHandler();
        if (m_CurrentFile != null)
          displayLocalContent(m_CurrentFile);
      }
    });
    m_PanelContentHandlers.add(new JLabel("Preferred handler"));
    m_PanelContentHandlers.add(m_ComboBoxContentHandlers);
  }

  /**
   * Finishes the initialization.
   */
  protected void finishInit() {
    super.finishInit();

    refreshLocalFiles();
  }

  /**
   * Refreshes the local file list.
   */
  protected synchronized void refreshLocalFiles() {
    File	dir;
    File[]	files;

    dir   = m_PanelDir.getCurrent();
    files = dir.listFiles();
    Arrays.sort(files);
    m_ModelLocalFiles.clear();
    for (File file: files) {
      if (file.isDirectory())
	continue;
      m_ModelLocalFiles.addElement(file.getName());
    }
    m_ListLocalFiles.search(m_ListLocalFiles.getSeachString(), m_ListLocalFiles.isRegExpSearch());

    m_ModelArchiveFiles.clear();
    m_PanelArchiveFiles.setVisible(false);
  }

  /**
   * Displays the given view in the content panel.
   *
   * @param panel	the view to display
   */
  protected void displayView(final JPanel panel) {
    Runnable	runnable;

    runnable = new Runnable() {
      public void run() {
	if ((m_PanelView.getComponentCount() > 0) && (m_PanelView.getComponent(0) instanceof CleanUpHandler))
	  ((CleanUpHandler) m_PanelView.getComponent(0)).cleanUp();
	m_PanelView.removeAll();
	m_PanelView.add(panel, BorderLayout.CENTER);
	m_SplitPane.validate();
      }
    };
    SwingUtilities.invokeLater(runnable);
  }

  /**
   * Displays "Creating view...".
   */
  protected void displayCreatingView() {
    displayView(new CreatingPreviewPanel());
  }

  /**
   * Displays a local file.
   */
  protected synchronized void displayLocalFile() {
    PlaceholderFile	localFile;
    Vector<Class>	handlers;
    String[]		files;

    if (m_ListLocalFiles.getSelectedIndex() < 0)
      return;

    localFile = new PlaceholderFile(m_PanelDir.getCurrent().getAbsolutePath() + File.separator + m_ListLocalFiles.getSelectedValue());

    if (AbstractArchiveHandler.hasHandler(localFile)) {
      m_PanelArchiveFiles.setVisible(true);
      handlers = AbstractArchiveHandler.getHandlersForFile(localFile);
      // update combobox
      m_IgnoreArchiveHandlerChanges = true;
      m_ModelArchiveHandlers.removeAllElements();
      for (Class cls: handlers)
	m_ModelArchiveHandlers.addElement(cls.getName());
      m_PanelArchiveHandlers.setVisible(m_ModelArchiveHandlers.getSize() > 1);
      AbstractArchiveHandler preferred = getPreferredArchiveHandler(localFile);
      // set preferred one
      m_ComboBoxArchiveHandlers.setSelectedIndex(0);
      if (preferred != null) {
	for (int i = 0; i < handlers.size(); i++) {
	  if (preferred.getClass() == handlers.get(i)) {
	    m_ComboBoxArchiveHandlers.setSelectedIndex(i);
	    break;
	  }
	}
      }
      m_IgnoreArchiveHandlerChanges = false;
      try {
	Class cls = Class.forName((String) m_ComboBoxArchiveHandlers.getSelectedItem());
	m_ArchiveHandler = (AbstractArchiveHandler) cls.newInstance();
      }
      catch (Exception e) {
	m_ArchiveHandler = null;
	System.err.println("Failed to obtain archive handler for '" + localFile + "':");
	e.printStackTrace();
      }
    }
    else {
      m_ArchiveHandler = null;
    }

    m_ModelArchiveFiles.clear();
    m_PanelArchiveFiles.setVisible((m_ArchiveHandler != null));
    m_PanelView.removeAll();
    if (m_ArchiveHandler == null) {
      displayLocalContent(localFile);
    }
    else {
      m_CurrentFile = null;
      m_ArchiveHandler.setArchive(localFile);
      files = m_ArchiveHandler.getFiles();
      for (String file: files)
	m_ModelArchiveFiles.addElement(file);
    }
    m_ListArchiveFiles.search(m_ListArchiveFiles.getSeachString(), m_ListArchiveFiles.isRegExpSearch());
  }

  /**
   * Displays the content from the specified file.
   *
   * @param localFile	the file to display
   */
  protected void displayLocalContent(final File localFile) {
    SwingWorker 	worker;

    m_CurrentFile = localFile;

    // notify user
    displayCreatingView();

    worker = new SwingWorker() {
      JPanel contentPanel;
      protected Object doInBackground() throws Exception {
	contentPanel = new NoPreviewAvailablePanel();
	if (AbstractContentHandler.hasHandler(localFile)) {
	  Vector<Class> handlers = AbstractContentHandler.getHandlersForFile(localFile);
	  // update combobox
	  m_IgnoreContentHandlerChanges = true;
	  m_ModelContentHandlers.removeAllElements();
	  for (Class cls: handlers)
	    m_ModelContentHandlers.addElement(cls.getName());
	  m_PanelContentHandlers.setVisible(m_ModelContentHandlers.getSize() > 1);
	  AbstractContentHandler preferred = getPreferredContentHandler(localFile);
	  // set preferred one
	  m_ComboBoxContentHandlers.setSelectedIndex(0);
	  if (preferred != null) {
	    for (int i = 0; i < handlers.size(); i++) {
	      if (preferred.getClass() == handlers.get(i)) {
		m_ComboBoxContentHandlers.setSelectedIndex(i);
		break;
	      }
	    }
	  }
	  m_IgnoreContentHandlerChanges = false;
	  // get preferred handler
	  AbstractContentHandler contentHandler;
	  try {
	    Class cls = Class.forName((String) m_ComboBoxContentHandlers.getSelectedItem());
	    contentHandler = (AbstractContentHandler) cls.newInstance();
	    contentPanel   = contentHandler.getPreview(localFile);
	  }
	  catch (Exception e) {
	    contentHandler = null;
	    System.err.println("Failed to obtain content handler for '" + localFile + "':");
	    e.printStackTrace();
	  }
	}
        return null;
      }
      protected void done() {
	displayView(contentPanel);
        super.done();
      }
    };
    worker.execute();
  }

  /**
   * Displays an archive file.
   */
  protected synchronized void displayArchiveContent() {
    String	selFile;
    File	tmpFile;
    JPanel	contentPanel;

    if (m_ListArchiveFiles.getSelectedIndex() < 0)
      return;

    // notify user
    displayCreatingView();

    contentPanel = new NoPreviewAvailablePanel();
    selFile      = (String) m_ListArchiveFiles.getSelectedValue();
    try {
      tmpFile = File.createTempFile("adams-fcb-", "." + FileUtils.getExtension(selFile), FileUtils.getTempDirectory());
      if (m_ArchiveHandler.extract(selFile, tmpFile)) {
	displayLocalContent(tmpFile);
	return;
      }
      else {
	System.err.println("Failed to extract file '" + selFile + "'!");
      }
    }
    catch (Exception e) {
      System.err.println("Failed to extract file '" + selFile + "':");
      e.printStackTrace();
    }
    displayView(contentPanel);
  }

  /**
   * Returns the preferrned content handler.
   *
   * @param file	the file to get the preferred handler for
   * @return		the preferred handler
   */
  protected AbstractContentHandler getPreferredContentHandler(File file) {
    AbstractContentHandler	result;
    Properties			props;
    String			ext;
    String			handler;

    result = null;

    ext = FileUtils.getExtension(file);
    if (ext == null)
      return result;
    ext = ext.toLowerCase();

    props = getProperties();
    if (props.hasKey(PREFIX_PREFERRED_CONTENT_HANDLER + ext)) {
      handler = props.getProperty(PREFIX_PREFERRED_CONTENT_HANDLER + ext);
      try {
	result = (AbstractContentHandler) Class.forName(handler).newInstance();
      }
      catch (Exception e) {
	System.err.println("Failed to instantiate handler: " + handler);
	e.printStackTrace();
      }
    }

    return result;
  }

  /**
   * Updates the preferred handler.
   */
  protected void updatePreferredContentHandler() {
    String	ext;
    String	handler;
    Properties	props;
    String	filename;

    if (m_CurrentFile == null)
      return;

    ext = FileUtils.getExtension(m_CurrentFile);
    if (ext == null)
      return;
    ext = ext.toLowerCase();

    if (m_ComboBoxContentHandlers.getSelectedIndex() < 0)
      handler = (String) m_ComboBoxContentHandlers.getItemAt(0);
    else
      handler = (String) m_ComboBoxContentHandlers.getSelectedItem();

    // update props
    props = getProperties();
    props.setProperty(PREFIX_PREFERRED_CONTENT_HANDLER + ext, handler);
    filename = Environment.getInstance().getHome() + File.separator + FILENAME;
    if (!props.save(filename))
      System.err.println("Failed to save properties to '" + filename + "'!");
  }

  /**
   * Returns the preferrned archive handler.
   *
   * @param file	the file to get the preferred handler for
   * @return		the preferred handler
   */
  protected AbstractArchiveHandler getPreferredArchiveHandler(File file) {
    AbstractArchiveHandler	result;
    Properties			props;
    String			ext;
    String			handler;

    result = null;

    ext = FileUtils.getExtension(file);
    if (ext == null)
      return result;
    ext = ext.toLowerCase();

    props = getProperties();
    if (props.hasKey(PREFIX_PREFERRED_ARCHIVE_HANDLER + ext)) {
      handler = props.getProperty(PREFIX_PREFERRED_ARCHIVE_HANDLER + ext);
      try {
	result = (AbstractArchiveHandler) Class.forName(handler).newInstance();
      }
      catch (Exception e) {
	System.err.println("Failed to instantiate handler: " + handler);
	e.printStackTrace();
      }
    }

    return result;
  }

  /**
   * Updates the preferred handler.
   */
  protected void updatePreferredArchiveHandler() {
    String	ext;
    String	handler;
    Properties	props;
    String	filename;

    if (m_CurrentFile == null)
      return;

    ext = FileUtils.getExtension(m_CurrentFile);
    if (ext == null)
      return;
    ext = ext.toLowerCase();

    if (m_ComboBoxArchiveHandlers.getSelectedIndex() < 0)
      handler = (String) m_ComboBoxArchiveHandlers.getItemAt(0);
    else
      handler = (String) m_ComboBoxArchiveHandlers.getSelectedItem();

    // update props
    props = getProperties();
    props.setProperty(PREFIX_PREFERRED_ARCHIVE_HANDLER + ext, handler);
    filename = Environment.getInstance().getCustomPropertiesFilename(PreviewBrowserPanelDefinition.KEY);
    if (!Environment.getInstance().write(PreviewBrowserPanelDefinition.KEY, props))
      System.err.println("Failed to save properties to '" + filename + "'!");
  }

  /**
   * Returns the properties that define the editor.
   *
   * @return		the properties
   */
  public static synchronized Properties getProperties() {
    if (m_Properties == null)
      m_Properties = Environment.getInstance().read(PreviewBrowserPanelDefinition.KEY);

    return m_Properties;
  }
}
