/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Scanner.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.parser.basedatetime;

import adams.core.DateFormat;
import adams.core.base.BaseDateTime;

import java_cup.runtime.SymbolFactory;
import java.io.*;
import java.util.*;

/**
 * A scanner for date/time expressions.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision$
 */
%%
%cup
%public
%class Scanner
%{
  // Author: FracPete (fracpete at waikato dot ac dot nz)
  // Version: $Revision$
  protected SymbolFactory sf;

  protected static DateFormat m_Format;
  static {
    m_Format = new DateFormat(BaseDateTime.FORMAT);
  }

  public Scanner(InputStream r, SymbolFactory sf){
    this(r);
    this.sf = sf;
  }
%}
%eofval{
    return sf.newSymbol("EOF",sym.EOF);
%eofval}

%%
// types
"SECOND" { return sf.newSymbol("Second", sym.TYPE, new Amount(Calendar.SECOND,    1)); }
"MINUTE" { return sf.newSymbol("Minute", sym.TYPE, new Amount(Calendar.MINUTE,    1)); }
"HOUR"   { return sf.newSymbol("Hour",   sym.TYPE, new Amount(Calendar.HOUR,      1)); }
"DAY"    { return sf.newSymbol("Day" ,   sym.TYPE, new Amount(Calendar.HOUR,     24)); }
"WEEK"   { return sf.newSymbol("Week",   sym.TYPE, new Amount(Calendar.HOUR,   24*7)); }
"MONTH"  { return sf.newSymbol("Month",  sym.TYPE, new Amount(Calendar.MONTH,     1)); }
"YEAR"   { return sf.newSymbol("Year",   sym.TYPE, new Amount(Calendar.YEAR,      1)); }

// numbers
[+][0-9][0-9]* { return sf.newSymbol("Number", sym.NUMBER, new Integer(yytext().substring(1))); }
[-][0-9][0-9]* { return sf.newSymbol("Number", sym.NUMBER, new Integer(yytext())); }

// date
[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9][ ][0-9][0-9]:[0-9][0-9]:[0-9][0-9] { return sf.newSymbol("Date", sym.DATE_ACTUAL, m_Format.parse(yytext())); }
"-INF" { return sf.newSymbol("-INF", sym.DATE_ACTUAL, m_Format.parse(BaseDateTime.INF_PAST_DATE)); }
"+INF" { return sf.newSymbol("+INF", sym.DATE_ACTUAL, m_Format.parse(BaseDateTime.INF_FUTURE_DATE)); }
"NOW"  { return sf.newSymbol("Now",  sym.DATE_ACTUAL, new Date()); }
"START" { return sf.newSymbol("Start", sym.DATE_START,  m_Format.parse(BaseDateTime.INF_PAST_DATE)); }
"END"   { return sf.newSymbol("End",   sym.DATE_END,    m_Format.parse(BaseDateTime.INF_FUTURE_DATE)); }

// whitespaces
[ \r\n\t\f] { /* ignore white space. */ }

// catch all
. { System.err.println("Illegal character: " + yytext()); }
