/**
 * CsvSpreadSheetWriter.java
 * Copyright (C) 2010-2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;

import java.io.Writer;
import java.util.Enumeration;

import adams.core.Utils;
import adams.core.io.SpreadSheet.Cell;
import adams.core.io.SpreadSheet.Row;

/**
 <!-- globalinfo-start -->
 * Writes CSV files.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-missing &lt;java.lang.String&gt; (property: missingValue)
 * &nbsp;&nbsp;&nbsp;The placeholder for missing values.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-number-format &lt;java.lang.String&gt; (property: numberFormat)
 * &nbsp;&nbsp;&nbsp;The format for the numbers (see java.text.DecimalFormat), use empty string
 * &nbsp;&nbsp;&nbsp;for default 'double' output.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3896 $
 */
public class CsvSpreadSheetWriter
  extends AbstractFormattedSpreadSheetWriter {

  /** for serialization. */
  private static final long serialVersionUID = -3549185519778801930L;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Writes CSV files.";
  }

  /**
   * Returns a string describing the format (used in the file chooser).
   *
   * @return 			a description suitable for displaying in the
   * 				file chooser
   */
  public String getFormatDescription() {
    return "CSV (comma-separated values)";
  }

  /**
   * Returns the extension(s) of the format.
   *
   * @return 			the extension (without the dot!)
   */
  public String[] getFormatExtensions() {
    return new String[]{"csv"};
  }

  /**
   * Returns whether to write to an OutputStream rather than a Writer.
   *
   * @return		true if to write to an OutputStream
   */
  protected boolean getUseOutputStream() {
    return false;
  }

  /**
   * Performs the actual writing. The caller must ensure that the writer gets
   * closed.
   *
   * @param content	the spreadsheet to write
   * @param writer	the writer to write the spreadsheet to
   * @return		true if successfully written
   */
  protected boolean doWrite(SpreadSheet content, Writer writer) {
    boolean			result;
    Enumeration<String>		rowKeys;
    Enumeration<String>		cellKeys;
    boolean			first;
    Row				row;
    Cell			cell;
    int				i;
    String			newline;

    result = true;

    try {
      newline = System.getProperty("line.separator");

      // comments?
      for (i = 0; i < content.getComments().size(); i++)
	writer.write(SpreadSheet.COMMENT + " " + content.getComments().get(i) + newline);

      // write header
      cellKeys = content.getHeaderRow().cellKeys();
      first    = true;
      while (cellKeys.hasMoreElements()) {
	cell = content.getHeaderRow().getCell(cellKeys.nextElement());

	if (!first)
	  writer.write(",");
	if (cell.isMissing())
	  writer.write(Utils.doubleQuote(m_MissingValue));
	else
	  writer.write(Utils.doubleQuote(cell.getContent()));

	first = false;
      }
      writer.write(newline);

      // write data rows
      rowKeys = content.rowKeys();
      while (rowKeys.hasMoreElements()) {
	row      = content.getRow(rowKeys.nextElement());
	cellKeys = content.getHeaderRow().cellKeys();
	first    = true;
	while (cellKeys.hasMoreElements()) {
	  cell = row.getCell(cellKeys.nextElement());

	  if (!first)
	    writer.write(",");
	  if ((cell != null) && (cell.getContent() != null) && !cell.isMissing()) {
	    if (cell.isNumeric())
	      writer.write(format(cell.toDouble()));
	    else
	      writer.write(Utils.doubleQuote(cell.getContent()));
	  }
	  else {
	    writer.write(Utils.doubleQuote(m_MissingValue));
	  }

	  first = false;
	}
	writer.write(newline);
      }
    }
    catch (Exception e) {
      result = false;
      e.printStackTrace();
    }

    return result;
  }
}
