/**
 * FileLog.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.util.Date;

import adams.core.DateFormat;

/**
 * Simple file logger that dumps the logging information in two separate
 * files, one for the default output and one for the error output.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 529 $
 */
public class FileLog
  extends Log {

  /** for serialization. */
  private static final long serialVersionUID = -1037691508623147214L;

  /** the suffix for the file for the default output. */
  public final static String SUFFIX_FILEOUT = "Out";;

  /** the suffix for the file for the Error output. */
  public final static String SUFFIX_FILEERR = "Err";;

  /** the suffix for whether to use timestamps. */
  public final static String SUFFIX_USETIMESTAMPS = "UseTimestamps";;

  /** the suffix for the timestamp format. */
  public final static String SUFFIX_TIMESTAMPFORMAT = "TimestampFormat";;

  /** the filename for the default output. */
  protected String m_FilenameOut;

  /** the filename for the error output. */
  protected String m_FilenameErr;

  /** whether to use timestamps. */
  protected boolean m_UseTimestamps;

  /** whether a newline has just been written to the file previously. */
  protected boolean m_NewLineWritten;

  /** for formatting the timestamps. */
  protected DateFormat m_TimestampFormat;

  /**
   * Initializes the logger.
   */
  protected void initialize() {
    super.initialize();

    m_FilenameOut     = getProperties().getString(getPropertyKey(SUFFIX_FILEOUT), "%p/log.out");
    m_FilenameErr     = getProperties().getString(getPropertyKey(SUFFIX_FILEERR), "%p/log.err");
    m_UseTimestamps   = getProperties().getBoolean(getPropertyKey(SUFFIX_USETIMESTAMPS), false);
    m_TimestampFormat = new DateFormat(getProperties().getString(getPropertyKey(SUFFIX_TIMESTAMPFORMAT), "yyyy-MM-dd HH:mm:ss"));

    FileUtils.delete(m_FilenameOut);
    FileUtils.delete(m_FilenameErr);

    m_NewLineWritten = true;
  }

  /**
   * Logs the message.
   *
   * @param msg		the message to log
   * @param out		whether to log default output (= true) or error output (= false)
   * @param newline	whether to add a newline or not
   */
  protected synchronized void doLog(String msg, boolean out, boolean newline) {
    BufferedWriter	writer;

    writer = null;
    try {
      // open file
      if (out)
	writer = new BufferedWriter(new FileWriter(m_FilenameOut, true));
      else
	writer = new BufferedWriter(new FileWriter(m_FilenameErr, true));

      // timestamp?
      if ((newline || (!newline && m_NewLineWritten)) && m_UseTimestamps)
	writer.write(m_TimestampFormat.format(new Date()) + "\t");

      // actual message
      writer.write(msg);

      // new line?
      m_NewLineWritten = newline;
      if (newline)
	writer.newLine();

      writer.flush();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    finally {
      // close file
      if (writer != null) {
	try {
	  writer.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
    }
  }
}
