/*
 * RunFit.java
 * Copyright (C) 2008 University of Waikato, Hamilton, New Zealand
 */

package adams.data.fit;

import adams.core.Utils;
import adams.core.option.ArrayConsumer;
import adams.core.option.OptionHandler;
import adams.core.option.OptionManager;
import adams.core.option.OptionUtils;
import adams.env.Environment;

/**
 * Runs a fitting algorithm from commandline.
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-x &lt;java.lang.String&gt; (property: x)
 *         The x values (comma-separated list).
 *         default: 1,2,3,4,5,6,7,8,9,10
 * </pre>
 *
 * <pre>-y &lt;java.lang.String&gt; (property: y)
 *         The y values (comma-separated list).
 *         default: 1,4,9,16,25,36,49,64,81,100
 * </pre>
 *
 * <pre>-sigma-x &lt;java.lang.String&gt; (property: sigmaX)
 *         The sigma-x values (comma-separated list).
 *         default:
 * </pre>
 *
 * <pre>-sigma-y &lt;java.lang.String&gt; (property: sigmaY)
 *         The sigma-y values (comma-separated list).
 *         default:
 * </pre>
 *
 * <pre>-num-points &lt;int&gt; (property: numPoints)
 *         The number of points, ie, basis functions, to use.
 *         default: 3
 * </pre>
 *
 * <pre>-guesses &lt;java.lang.String&gt; (property: initialGuesses)
 *         The values of the initial guesses (comma-separated list).
 *         default:
 * </pre>
 *
 * <pre>-fit &lt;adams.core.fit.Fit [options]&gt; (property: fit)
 *         The fitting algorithm to use.
 *         default: adams.core.fit.LinearLeastSquares -function adams.core.fit.Polynomial -iterations 30
 * </pre>
 *
 * Default options for adams.core.fit.LinearLeastSquares (-fit/fit):
 *
 * <pre>-D (property: debug)
 *         If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-function &lt;adams.core.fit.LinearFunction [options]&gt; (property: function)
 *         The function to use for calculating the y values based on the x input.
 *         default: adams.core.fit.Polynomial
 * </pre>
 *
 * <pre>-iterations &lt;int&gt; (property: numIterations)
 *         The number of iterations to perform in singular value decomposition.
 *         default: 30
 * </pre>
 *
 * <pre>Default options for adams.core.fit.Polynomial (-function/function):
 * </pre>
 *
 * <pre>-D (property: debug)
 *         If set to true, function may output additional info to the console.
 * </pre>
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3136 $
 */
public class RunFit
  implements OptionHandler {

  /** for managing the available options. */
  protected OptionManager m_OptionManager;

  /** the x values. */
  protected double[] m_X;

  /** the y values. */
  protected double[] m_Y;

  /** the std. deviations for x. */
  protected double[] m_SigmaX;

  /** the std. deviations for y. */
  protected double[] m_SigmaY;

  /** the number of points (i.e., basis functions). */
  protected int m_NumPoints;

  /** the initial parameter guesses. */
  protected double[] m_InitialGuesses;

  /** the peak-aligner algorithm to run. */
  protected Fit m_Fit;

  /**
   * default constructor.
   */
  public RunFit() {
    super();
    initialize();
    defineOptions();
    getOptionManager().setDefaults();
  }

  /**
   * initializes member variables.
   */
  protected void initialize() {
    m_X      = new double[0];
    m_Y      = new double[0];
    m_SigmaX = null;
    m_SigmaY = null;
  }

  /**
   * Returns a new instance of the option manager.
   *
   * @return		the manager to use
   */
  protected OptionManager newOptionManager() {
    return new OptionManager(this);
  }

  /**
   * Adds options to the internal list of options. Derived classes must
   * override this method to add additional options.
   */
  public void defineOptions() {
    m_OptionManager = newOptionManager();

    m_OptionManager.add(
	    "x", "x",
	    "1,2,3,4,5,6,7,8,9,10");

    m_OptionManager.add(
	    "y", "y",
	    "1,4,9,16,25,36,49,64,81,100");

    m_OptionManager.add(
	    "sigma-x", "sigmaX",
	    "");

    m_OptionManager.add(
	    "sigma-y", "sigmaY",
	    "");

    m_OptionManager.add(
	    "num-points", "numPoints",
	    3);

    m_OptionManager.add(
	    "guesses", "initialGuesses",
	    "");

    m_OptionManager.add(
	    "fit", "fit",
	    new LinearLeastSquares());
  }

  /**
   * Returns the option manager.
   *
   * @return		the manager
   */
  public OptionManager getOptionManager() {
    if (m_OptionManager == null)
      defineOptions();

    return m_OptionManager;
  }

  /**
   * Cleans up the options.
   */
  public void cleanUpOptions() {
    if (m_OptionManager != null) {
      m_OptionManager.cleanUp();
      m_OptionManager = null;
    }
  }

  /**
   * Frees up memory in a "destructive" non-reversible way.
   * <p/>
   * Cleans up the options.
   *
   * @see	#cleanUpOptions()
   */
  public void destroy() {
    cleanUpOptions();
  }

  /**
   * Turns the (comma-separated) string into a double array, or null if
   * string is of length 0.
   *
   * @param s		the string to convert
   * @return		the double values
   */
  protected double[] str2doubleArray(String s) {
    double[]	result;
    String[]	parts;
    int		i;

    result = null;

    if ((s != null) && (s.length() > 0)) {
      parts  = s.replaceAll(" ", "").split(",");
      result = new double[parts.length];
      for (i = 0; i < parts.length; i++)
	result[i] = Double.parseDouble(parts[i]);
    }

    return result;
  }

  /**
   * Turns the double array into a comma-separated string.
   *
   * @param d		the double array, can be null
   * @return		the string
   */
  protected String doubleArray2str(double[] d) {
    String	result;
    int		i;

    result = "";

    if (d != null) {
      for (i = 0; i < d.length; i++) {
	if (i > 0)
	  result += ",";
	result += d[i];
      }
    }

    return result;
  }

  /**
   * Sets the x values.
   *
   * @param value 	the comma-separated values
   */
  public void setX(String value) {
    if (value.length() != 0)
      m_X = str2doubleArray(value);
    else
      System.err.println("x values cannot be empty!");
  }

  /**
   * Returns the x values as string.
   *
   * @return 		the comma-separated values
   */
  public String getX() {
    return doubleArray2str(m_X);
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String xTipText() {
    return "The x values (comma-separated list).";
  }

  /**
   * Sets the y values.
   *
   * @param value 	the comma-separated values
   */
  public void setY(String value) {
    if (value.length() != 0)
      m_Y = str2doubleArray(value);
    else
      System.err.println("y values cannot be empty!");
  }

  /**
   * Returns the y values as string.
   *
   * @return 		the comma-separated values
   */
  public String getY() {
    return doubleArray2str(m_Y);
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String yTipText() {
    return "The y values (comma-separated list).";
  }

  /**
   * Sets the sigma-x values.
   *
   * @param value 	the comma-separated values
   */
  public void setSigmaX(String value) {
    m_SigmaX = str2doubleArray(value);
  }

  /**
   * Returns the sigma-x values as string.
   *
   * @return 		the comma-separated values
   */
  public String getSigmaX() {
    return doubleArray2str(m_SigmaX);
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String sigmaXTipText() {
    return "The sigma-x values (comma-separated list).";
  }

  /**
   * Sets the sigma-y values.
   *
   * @param value 	the comma-separated values
   */
  public void setSigmaY(String value) {
    m_SigmaY = str2doubleArray(value);
  }

  /**
   * Returns the sigma-y values as string.
   *
   * @return 		the comma-separated values
   */
  public String getSigmaY() {
    return doubleArray2str(m_SigmaY);
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String sigmaYTipText() {
    return "The sigma-y values (comma-separated list).";
  }

  /**
   * Sets the number of points, i.e., basis functions, to use.
   *
   * @param value 	the number
   */
  public void setNumPoints(int value) {
    m_NumPoints = value;
  }

  /**
   * Returns the number of points, i.e., basis functions, in use.
   *
   * @return 		the number
   */
  public int getNumPoints() {
    return m_NumPoints;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String numPointsTipText() {
    return "The number of points, ie, basis functions, to use.";
  }

  /**
   * Sets the values of the initial guesses.
   *
   * @param value 	the comma-separated values
   */
  public void setInitialGuesses(String value) {
    m_InitialGuesses = str2doubleArray(value);
  }

  /**
   * Returns the values of the initial guesses as string.
   *
   * @return 		the comma-separated values
   */
  public String getInitialGuesses() {
    return doubleArray2str(m_InitialGuesses);
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String initialGuessesTipText() {
    return "The values of the initial guesses (comma-separated list).";
  }

  /**
   * Sets the fitting algorithm to run.
   *
   * @param value 	the algorithm
   */
  public void setFit(Fit value) {
    m_Fit = value;
  }

  /**
   * Returns the fitting algorithm being used.
   *
   * @return 		the algorithm
   */
  public Fit getFit() {
    return m_Fit;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String fitTipText() {
    return "The fitting algorithm to use.";
  }

  /**
   * Runs the fitting algorithm and prints the result to stdout.
   *
   * @throws Exception	if something goes wrong
   */
  public void run() throws Exception {
    double[]	parameters;
    double[]	initial;
    int		i;
    boolean	success;

    initial = null;
    if (m_InitialGuesses != null)
      initial = m_InitialGuesses.clone();
    if (initial == null) {
      parameters = new double[m_NumPoints];
    }
    else {
      if (initial.length != m_NumPoints)
	throw new IllegalArgumentException("Number of initial guesses differ from number of points!");
      parameters = initial.clone();
    }

    // a few sanity checks
    if (m_X.length != m_Y.length)
      throw new IllegalArgumentException("x and y have different length!");
    if ((m_SigmaX != null) && (m_SigmaX.length != m_X.length))
      throw new IllegalArgumentException("sigma-x and x have different length!");
    if ((m_SigmaY != null) && (m_SigmaY.length != m_X.length))
      throw new IllegalArgumentException("sigma-x and x have different length!");

    // can we guess the initial parameters?
    if ((m_InitialGuesses == null) && (m_Fit.canGuess())) {
      initial = m_Fit.guess(m_X, m_Y);
      parameters = initial.clone();
    }

    success = m_Fit.fitClean(parameters, m_X, m_Y, m_SigmaX, m_SigmaY);
    System.out.println(m_Fit.getDescription());
    System.out.println(m_Fit.getDescription().replaceAll(".", "="));
    System.out.println();
    System.out.println("Algorithm..: " + OptionUtils.getCommandLine(m_Fit));
    System.out.println("x..........: " + getX());
    System.out.println("y..........: " + getY());
    System.out.println("sigma-x....: " + ((getSigmaX().length() == 0) ? "-none-" : getSigmaX()));
    System.out.println("sigma-y....: " + ((getSigmaY().length() == 0) ? "-none-" : getSigmaY()));
    System.out.println("# of points: " + getNumPoints());
    System.out.println("guesses....: " + ((initial == null) ? "-none-" : Utils.arrayToString(initial)));
    System.out.println();
    if (!success) {
      System.out.println("Fitting wasn't successful!");
      System.out.println();
    }
    System.out.println("Coefficients");
    for (i = 0; i < m_NumPoints; i++)
      System.out.println(parameters[i]);
  }

  /**
   * Runs the tool from commandline.
   *
   * @param args	the commandline arguments, use -help to display all
   */
  public static void main(String[] args) {
    RunFit 	runFit;

    Environment.setEnvironmentClass(Environment.class);

    runFit = new RunFit();

    try {
      if (OptionUtils.helpRequested(args)) {
	System.out.println("Help requested...\n");
	System.out.println(OptionUtils.list(runFit));
      }
      else {
	ArrayConsumer.setOptions(runFit, args);
	runFit.run();
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }
}
