/*
 * ActorHelper.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.core;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Array;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Vector;

import javax.swing.JFrame;

import adams.core.ClassLocator;
import adams.core.Utils;
import adams.core.option.AbstractArgumentOption;
import adams.core.option.AbstractOption;
import adams.core.option.NestedConsumer;
import adams.core.option.NestedProducer;
import adams.core.option.OptionHandler;
import adams.env.Environment;
import adams.flow.processor.AbstractActorProcessor;
import adams.flow.processor.FixDeprecatedCommandlineTransformers;
import adams.flow.processor.FlattenStructure;
import adams.flow.processor.MultiProcessor;
import adams.flow.processor.RemoveDisabledActors;
import adams.flow.processor.RemoveUnusedGlobalActors;
import adams.flow.sink.InstantiatableSink;
import adams.flow.source.InstantiatableSource;
import adams.flow.standalone.InstantiatableStandalone;
import adams.flow.transformer.GlobalTransformer;
import adams.flow.transformer.InstantiatableTransformer;
import adams.gui.application.AbstractApplicationFrame;
import adams.gui.core.GUIHelper;

/**
 * Helper class for actors.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4216 $
 */
public class ActorUtils {

  /**
   * Enumerates all children of the given actor (depth-first search).
   *
   * @param actor	the actor to obtain all children from
   * @param children	for collecting the children
   */
  protected static void enumerate(AbstractActor actor, Vector<AbstractActor> children) {
    int			i;
    ActorHandler	handler;

    if (actor instanceof ActorHandler) {
      handler = (ActorHandler) actor;
      for (i = 0; i < handler.size(); i++) {
	children.add(handler.get(i));
	enumerate(handler.get(i), children);
      }
    }
  }

  /**
   * Enumerates all children of the given actor (depth-first search).
   *
   * @param actor	the actor to obtain all children from
   * @return		all the children (if any)
   */
  public static Vector<AbstractActor> enumerate(AbstractActor actor) {
    Vector<AbstractActor>	result;

    result = new Vector<AbstractActor>();
    enumerate(actor, result);

    return result;
  }

  /**
   * Replaces any occurrence of the object to find with the replacement. Can
   * be done recursive. Performs the replacement also in arrays and nested
   * OptionHandlers.
   *
   * @param handler	the OptionHandler to process
   * @param find	the object to look for
   * @param replace	the replacement
   * @param recursive	if true then all children (if any) of the actor will
   * 			be processed as well
   * @return		the number of replacements
   */
  public static int replace(OptionHandler handler, Comparable find, Comparable replace, boolean recursive) {
    return replace(handler, find, replace, recursive, new HashSet<Class>());
  }

  /**
   * Replaces any occurrence of the object to find with the replacement. Can
   * be done recursive. Performs the replacement also in arrays and nested
   * OptionHandlers.
   *
   * @param handler	the OptionHandler to process
   * @param find	the object to look for
   * @param replace	the replacement
   * @param recursive	if true then all children (if any) of the actor will
   * 			be processed as well
   * @param excluded	the base classes to skip
   * @return		the number of replacements
   */
  public static int replace(OptionHandler handler, Comparable find, Comparable replace, boolean recursive, HashSet<Class> excluded) {
    int				result;
    boolean			updated;
    List<AbstractOption>	options;
    int				i;
    int				n;
    PropertyDescriptor		desc;
    Object			value;
    AbstractArgumentOption	argoption;

    result = 0;

    options = handler.getOptionManager().getOptionsList();
    for (i = 0; i < options.size(); i++) {
      updated = false;

      // superclass?
      if (!(options.get(i) instanceof AbstractArgumentOption))
	continue;
      argoption = (AbstractArgumentOption) options.get(i);

      // skipped?
      if (excluded.contains(argoption.getBaseClass()))
	continue;

      // correct class?
      if (!argoption.getBaseClass().equals(find.getClass())) {
	// nested OptionHandler?
	if (recursive && (ClassLocator.hasInterface(OptionHandler.class, argoption.getBaseClass()))) {
	  // get current value
	  desc = options.get(i).getDescriptor();
	  try {
	    value = desc.getReadMethod().invoke(handler, new Object[0]);
	  }
	  catch (Exception e) {
	    e.printStackTrace();
	    value = null;
	  }
	  if (value == null)
	    continue;

	  // check nested OptionHandler(s)
	  if (argoption.isMultiple()) {
	    for (n = 0; n < Array.getLength(value); n++)
	      result += replace((OptionHandler) Array.get(value, n), find, replace, recursive);
	  }
	  else {
	    result += replace((OptionHandler) value, find, replace, recursive);
	  }
	}
	continue;
      }

      // get current value
      desc = options.get(i).getDescriptor();
      try {
	value = desc.getReadMethod().invoke(handler, new Object[0]);
      }
      catch (Exception e) {
	e.printStackTrace();
	value = null;
      }
      if (value == null)
	continue;

      // the value we're looking for?
      if (argoption.isMultiple()) {
	for (n = 0; n < Array.getLength(value); n++) {
	  if (Array.get(value, n).equals(find)) {
	    Array.set(value, n, Utils.deepCopy(replace));
	    updated = true;
	    result++;
	  }
	}
      }
      else {
	if (value.equals(find)) {
	  value   = Utils.deepCopy(replace);
	  updated = true;
	  result++;
	}
      }

      // update values
      if (updated) {
	try {
	  desc.getWriteMethod().invoke(handler, value);
	}
	catch (Exception e) {
	  e.printStackTrace();
	}
      }
    }

    return result;
  }

  /**
   * Updates the name of the actor to make it unique among all the other
   * actors of the given handler.
   *
   * @param actor	the actor which name needs to be made unique
   * @param handler	the handler to take into account
   * @param index	the index that this actor will be placed
   * @return		true if the name was updated
   */
  public static boolean uniqueName(AbstractActor actor, ActorHandler handler, int index) {
    HashSet<String>	names;
    int			i;

    // get all names
    names = new HashSet<String>();
    for (i = 0; i < handler.size(); i++) {
      if (i != index)
	names.add(handler.get(i).getName());
    }

    return uniqueName(actor, names);
  }

  /**
   * Updates the name of the actor to make it unique among all the other
   * specified names.
   *
   * @param actor	the actor which name needs to be made unique
   * @param names	the existing names
   * @return		true if the name was updated
   */
  public static boolean uniqueName(AbstractActor actor, HashSet<String> names) {
    boolean		result;
    String		name;
    String		baseName;
    int			i;

    // create unique name
    baseName = actor.getName();
    i        = 0;
    do {
      if (i == 0)
	name = baseName;
      else
	name = baseName + "-" + i;
      i++;
    }
    while (names.contains(name));

    result = (!name.equals(actor.getName()));

    // update name
    if (result)
      actor.setName(name);

    return result;
  }

  /**
   * Updates the names of the actors to make then unique among all of them.
   * NB: any suffix matching "-XYZ" gets stripped first.
   *
   * @param actors	the actors which names need to be made unique
   * @return		true if at least one name was updated
   */
  public static boolean uniqueNames(AbstractActor[] actors) {
    boolean		result;
    HashSet<String>	names;
    int			i;

    result = false;

    names = new HashSet<String>();
    for (i = 0; i < actors.length; i++) {
      if (i > 0)
	result = uniqueName(actors[i], names) || result;
      names.add(actors[i].getName());
    }

    return result;
  }

  /**
   * Returns a list of actor handlers, starting from the current note (excluded).
   * The search goes up in the actor hierarchy, up to the root (i.e., the last
   * item in the returned list will be most likely a "Flow" actor).
   *
   * @param actor	the actor to start the search from
   * @return		the list of actor handlers found along the path to
   * 			the root actor
   */
  public static Vector<ActorHandler> findActorHandlers(AbstractActor actor) {
    return findActorHandlers(actor, false);
  }

  /**
   * Returns a list of actor handlers, starting from the current note (excluded).
   * The search goes up in the actor hierarchy, up to the root (i.e., the last
   * item in the returned list will be most likely a "Flow" actor).
   *
   * @param actor			the actor to start the search from
   * @param mustAllowStandalones	whether the handler must allow standalones
   * @return				the list of actor handlers found along the path
   * 					to the root actor
   */
  public static Vector<ActorHandler> findActorHandlers(AbstractActor actor, boolean mustAllowStandalones) {
    return findActorHandlers(actor, mustAllowStandalones, false);
  }

  /**
   * Returns a list of actor handlers, starting from the current note (excluded).
   * The search goes up in the actor hierarchy, up to the root (i.e., the last
   * item in the returned list will be most likely a "Flow" actor).
   *
   * @param actor			the actor to start the search from
   * @param mustAllowStandalones	whether the handler must allow standalones
   * @param includeSameLevel		allows adding of actor handlers that are on
   * 					the same level as the current actor, but
   * 					have a lower index in the parent
   * @return				the list of actor handlers found along the path
   * 					to the root actor
   */
  public static Vector<ActorHandler> findActorHandlers(AbstractActor actor, boolean mustAllowStandalones, boolean includeSameLevel) {
    Vector<ActorHandler>	result;
    AbstractActor		parent;
    AbstractActor		child;
    int				index;
    int				i;
    ActorHandler		handler;
    ActorHandler		subhandler;
    AbstractExternalActor	extActor;

    result = new Vector<ActorHandler>();

    child  = actor;
    parent = actor.getParent();
    while (parent != null) {
      if (parent instanceof ActorHandler) {
	handler = (ActorHandler) parent;
	if (includeSameLevel) {
	  index = handler.indexOf(child.getName());
	  for (i = index - 1; i >= 0; i--) {
	    subhandler = null;
	    if (handler.get(i) instanceof AbstractExternalActor) {
	      extActor = (AbstractExternalActor) handler.get(i);
	      if (extActor.getExternalActor() instanceof ActorHandler)
		subhandler = (ActorHandler) extActor.getExternalActor();
	    }
	    else if (handler.get(i) instanceof ActorHandler) {
	      subhandler = (ActorHandler) handler.get(i);
	    }
	    if (subhandler == null)
	      continue;
	    if (mustAllowStandalones) {
	      if (subhandler.getActorHandlerInfo().canContainStandalones())
		result.add(subhandler);
	    }
	    else {
	      result.add(subhandler);
	    }
	  }
	}
	if (mustAllowStandalones) {
	  if (handler.getActorHandlerInfo().canContainStandalones())
	    result.add(handler);
	}
	else {
	  result.add(handler);
	}
      }

      child  = parent;
      parent = parent.getParent();
    }

    return result;
  }

  /**
   * Checks whether this actor is a standalone.
   *
   * @param actor	the actor to check
   * @return		true if standalone
   */
  public static boolean isStandalone(AbstractActor actor) {
    return (!(actor instanceof InputConsumer)) && (!(actor instanceof OutputProducer));
  }

  /**
   * Checks whether this actor is a source (output).
   *
   * @param actor	the actor to check
   * @return		true if source
   */
  public static boolean isSource(AbstractActor actor) {
    return (!(actor instanceof InputConsumer)) && (actor instanceof OutputProducer);
  }

  /**
   * Checks whether this actor is a sink (input).
   *
   * @param actor	the actor to check
   * @return		true if sink
   */
  public static boolean isSink(AbstractActor actor) {
    return (actor instanceof InputConsumer) && (!(actor instanceof OutputProducer));
  }

  /**
   * Checks whether this actor is a transformer (input/output).
   *
   * @param actor	the actor to check
   * @return		true if transformer
   */
  public static boolean isTransformer(AbstractActor actor) {
    return (actor instanceof InputConsumer) && (actor instanceof OutputProducer);
  }

  /**
   * Checks whether this actor is a control actor.
   *
   * @param actor	the actor to check
   * @return		true if control actor
   */
  public static boolean isControlActor(AbstractActor actor) {
    return (actor instanceof ControlActor);
  }

  /**
   * Checks whether this actor is an actor handler.
   *
   * @param actor	the actor to check
   * @return		true if actor handler
   */
  public static boolean isActorHandler(AbstractActor actor) {
    return (actor instanceof ActorHandler);
  }

  /**
   * Writes the actor to a file.
   *
   * @param filename	the file to write to
   * @param actor	the actor to write
   * @return		true if successfully written
   */
  public static boolean write(String filename, AbstractActor actor) {
    boolean		result;
    NestedProducer	producer;

    producer = new NestedProducer();
    producer.setOutputClasspath(true);
    producer.produce(actor);
    result = producer.write(filename);

    return result;
  }

  /**
   * Reads an actor from a file.
   *
   * @param filename	the file to read the actor
   * @return		the actor or null in case of an error
   */
  public static AbstractActor read(String filename) {
    return read(filename, null);
  }

  /**
   * Reads an actor from a file.
   *
   * @param filename	the file to read the actor
   * @param errors	for storing (potential) errors, ignored if null
   * @return		the actor or null in case of an error
   */
  public static AbstractActor read(String filename, List<String> errors) {
    return read(filename, errors, null);
  }

  /**
   * Reads an actor from a file.
   *
   * @param filename	the file to read the actor
   * @param errors	for storing (potential) errors, ignored if null
   * @param warnings	for storing (potential) warnings, ignored if null
   * @return		the actor or null in case of an error
   */
  public static AbstractActor read(String filename, List<String> errors, List<String> warnings) {
    AbstractActor	result;
    NestedConsumer	consumer;

    consumer = new NestedConsumer();
    result   = (AbstractActor) consumer.read(filename);

    // transfer errors/warnings
    if (errors != null)
      errors.addAll(consumer.getErrors());
    if (warnings != null)
      warnings.addAll(consumer.getWarnings());

    return result;
  }

  /**
   * Locates global transformers.
   *
   * @param actor	the actor (and its sub-actors) to check
   * @return		the names of the global actors referenced and how
   * 			often they were referenced
   */
  public static Hashtable<String,Integer> findGlobalTransformers(AbstractActor actor) {
    Vector<AbstractActor>	actors;
    Hashtable<String,Integer>	count;
    String			name;

    actors = enumerate(actor);
    count  = new Hashtable<String,Integer>();
    for (AbstractActor current: actors) {
      if (current.getSkip())
	continue;
      if (!(current instanceof GlobalTransformer))
	continue;
      name = ((GlobalTransformer) current).getGlobalName().toString();
      if (!count.containsKey(name))
	count.put(name, 1);
      else
	count.put(name, count.get(name) + 1);
    }

    return count;
  }

  /**
   * Checks an actor handler's children whether they contain the actor type
   * we're looking for.
   *
   * @param handler	the actor handler to check
   * @param type	the type of actor to find the closest for
   * @return		the closest actor or null if not found
   */
  protected static AbstractActor findClosestType(ActorHandler handler, Class type) {
    AbstractActor		result;
    int				i;
    AbstractExternalActor	external;

    result = null;

    for (i = 0; i < handler.size(); i++) {
      if (handler.get(i).getClass().isAssignableFrom(type)) {
	result = handler.get(i);
	break;
      }
      else if (handler.get(i) instanceof AbstractExternalActor) {
	external = (AbstractExternalActor) handler.get(i);
	if (external.getExternalActor() instanceof ActorHandler) {
	  result = findClosestType((ActorHandler) external.getExternalActor(), type);
	  if (result != null)
	    break;
	}
      }
    }

    return result;
  }

  /**
   * Tries to find the closest type in the actor tree, starting with the current
   * actor.
   *
   * @param actor		the actor to start from
   * @param type		the type of actor to find the closest for
   * @return			the closest actor or null if not found
   */
  public static AbstractActor findClosestType(AbstractActor actor, Class type) {
    return findClosestType(actor, type, false);
  }

  /**
   * Tries to find the closest type in the actor tree, starting with the current
   * actor.
   *
   * @param actor		the actor to start from
   * @param type		the type of actor to find the closest for
   * @param includeSameLevel	whether to include actor handlers that are on
   * 				the same level as the actor, but with a lower
   * 				index in the parent
   * @return			the closest actor or null if not found
   */
  public static AbstractActor findClosestType(AbstractActor actor, Class type, boolean includeSameLevel) {
    AbstractActor		result;
    Vector<ActorHandler>	handlers;
    int				i;

    result   = null;
    handlers = ActorUtils.findActorHandlers(actor, true, includeSameLevel);
    for (i = 0; i < handlers.size(); i++) {
      result = findClosestType(handlers.get(i), type);
      if (result != null)
	break;
    }

    return result;
  }

  /**
   * Determines the initial location of the frame.
   *
   * @param frame	the frame to determine the location for
   * @param x		the position (-1: left, -2: middle, -3: right)
   * @param y		the position (-1: top, -2: middle, -3: bottom)
   * @return		the position
   */
  public static Point determineLocation(JFrame frame, int x, int y) {
    Point			result;
    Dimension			size;
    int				actX;
    int				actY;
    AbstractApplicationFrame	main;
    int				diffHeight;
    Rectangle			bounds;

    result = new Point();
    size   = frame.getSize();
    if ((size.getWidth() == 0) || (size.getHeight() == 0))
      size = frame.getPreferredSize();
    bounds = GUIHelper.getScreenBounds(frame);

    // X
    if (x == -1)
      actX = 0;
    else if (x == -2)
      actX = (int) ((bounds.width - size.getWidth()) / 2);
    else if (x == -3)
      actX = (int) (bounds.width - size.getWidth());
    else
      actX = x;

    // Y
    main       = Environment.getInstance().getApplicationFrame();
    diffHeight = ((main != null) ? main.getSize().height : 0);
    if (y == -1)
      actY = diffHeight;
    else if (y == -2)
      actY = (int) ((bounds.height - size.getHeight() - diffHeight) / 2);
    else if (y == -3)
      actY = (int) (bounds.height - size.getHeight());
    else
      actY = y;

    result.setLocation(actX, actY);

    return result;
  }

  /**
   * Returns the database connection object to use.
   *
   * @param actor	the actor start the search from (towards the root)
   * @param cls		the DatabaseConnection actor class to look for
   * @param defCon	the default database connection, in case none is found
   * 			in the flow
   * @return		the connection object to use
   */
  public static adams.db.AbstractDatabaseConnection getDatabaseConnection(AbstractActor actor, Class cls, adams.db.AbstractDatabaseConnection defCon) {
    adams.db.AbstractDatabaseConnection			result;
    adams.flow.standalone.AbstractDatabaseConnection	conn;

    conn = (adams.flow.standalone.AbstractDatabaseConnection) ActorUtils.findClosestType(actor, cls, true);
    if (conn != null)
      result = conn.getConnection();
    else
      result = defCon;
    if (!result.isConnected()) {
      try {
	result.connect();
      }
      catch (Exception e) {
	System.err.println("Failed to enable database connection (" + cls.getName() + ") for actor " + actor.getFullName() + ":");
	e.printStackTrace();
      }
    }

    return result;
  }

  /**
   * Removes all disabled actors (recursively) from the actor.
   * <p/>
   * NB: creates a copy of the actor first before removing the disabled actors.
   *
   * @param actor	the actor to process
   * @return		the cleaned up actor (or original, if nothing changed)
   */
  public static AbstractActor removeDisabledActors(AbstractActor actor) {
    AbstractActor		result;
    RemoveDisabledActors	processor;

    processor = new RemoveDisabledActors();
    processor.process(actor);
    result = processor.getModifiedActor();
    if (result == null)
      result = actor;

    return result;
  }

  /**
   * Ensures that the actor is enclosed in an "instantiable" wrapper, if it
   * doesn't implement the InstantiatableActor interface itself.
   *
   * @param actor	the actor to check and (potentially) enclose
   * @return		the processed actor
   * @see		InstantiatableActor
   */
  public static AbstractActor createExternalActor(AbstractActor actor) {
    AbstractActor	result;

    result = actor;

    if (!(actor instanceof InstantiatableActor)) {
      if (isStandalone(actor)) {
	result = new InstantiatableStandalone();
	((InstantiatableStandalone) result).set(0, actor);
      }
      else if (isSource(actor)) {
	result = new InstantiatableSource();
	((InstantiatableSource) result).set(0, actor);
      }
      else if (isTransformer(actor)) {
	result = new InstantiatableTransformer();
	((InstantiatableTransformer) result).set(0, actor);
      }
      else if (isSink(actor)) {
	result = new InstantiatableSink();
	((InstantiatableSink) result).set(0, actor);
      }
    }

    return result;
  }

  /**
   * Tries to locate the actor specified by the path parts.
   *
   * @param parent	the parent to start with
   * @param path	the path elements to traverse (below the parent)
   * @return		the located actor or null if none found
   */
  public static AbstractActor locate(ActorPath path, AbstractActor parent) {
    AbstractActor	result;
    ActorHandler	parentHandler;
    AbstractActor	child;
    int			index;
    int			i;

    result = null;
    if (!(parent instanceof ActorHandler))
      return result;

    parentHandler = (ActorHandler) parent;
    index         = -1;
    for (i = 0; i < parentHandler.size(); i++) {
      child = parentHandler.get(i);
      if (child.getName().equals(path.getFirstPathComponent())) {
	index = i;
	break;
      }
    }
    if (index != -1) {
      child = parentHandler.get(index);
      if (path.getPathCount() == 1)
	result = child;
      else
	result = locate(path.getChildPath(), child);
    }
    else {
      System.err.println("Malformed path?");
    }

    return result;
  }

  /**
   * Locates the actor in the actor tree based on the specified path.
   *
   * @param path	the path of the node to locate
   * @param root	the root actor start the search
   * @return		the located actor or null if none found
   */
  public static AbstractActor locate(String path, AbstractActor root) {
    return locate(new ActorPath(path), root);
  }

  /**
   * Checks whether the actor has a source as first actor (can be nested).
   *
   * @param actor	the actor to analyze
   * @return		null if ok, otherwise error message
   */
  public static String checkForSource(AbstractActor actor) {
    return checkForSource(new AbstractActor[]{actor});
  }

  /**
   * Checks whether the actors have a source as first actor (can be nested).
   *
   * @param actors	the actors to analyze
   * @return		null if ok, otherwise error message
   */
  public static String checkForSource(AbstractActor[] actors) {
    String		result;
    int			i;
    AbstractActor	actor;
    AbstractActor	previous;
    ActorHandler	handler;
    String		name;

    result = null;

    for (i = 0; i < actors.length; i++) {
      actor = actors[i];
      if (actor.getSkip())
	continue;
      if (ActorUtils.isStandalone(actor))
	continue;
      name = actor.getName();
      // drill down in actor handlers
      while (actor instanceof ActorHandler) {
	handler  = (ActorHandler) actor;
	previous = actor;
	actor    = handler.firstActive();
	if (actor == null) {
	  actor = previous;
	  break;
	}
	name += "." + actor.getName();
      }
      if (!ActorUtils.isSource(actor))
	result = "First active, non-standalone actor must be a source, but '" + name + "' is not!";
      break;
    }

    return result;
  }

  /**
   * Cleans up the flow, e.g., removing disabled actors, unused global actors.
   *
   * @param actor	the flow to clean up
   * @return		null if nothing changed, otherwise the updated flow
   */
  public static AbstractActor cleanUpFlow(AbstractActor actor) {
    AbstractActor	result;
    MultiProcessor	processor;

    result = null;

    processor = new MultiProcessor();
    processor.setSubProcessors(new AbstractActorProcessor[]{
	new RemoveDisabledActors(),
	new RemoveUnusedGlobalActors(),
	new FixDeprecatedCommandlineTransformers(),
	new FlattenStructure()
    });

    processor.process(actor);
    if (processor.isModified())
      result = processor.getModifiedActor();

    return result;
  }
}
