/*
 * GlobalSource.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.source;

import adams.flow.core.AbstractActor;
import adams.flow.core.AbstractGlobalActor;
import adams.flow.core.OutputProducer;
import adams.flow.core.Token;
import adams.flow.core.Unknown;

/**
 <!-- globalinfo-start -->
 * Broadcasts the generated output tokens from the source.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 * &nbsp;&nbsp;&nbsp;If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: GlobalSource
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-global &lt;java.lang.String&gt; (property: globalName)
 * &nbsp;&nbsp;&nbsp;The name of the global actor to use.
 * &nbsp;&nbsp;&nbsp;default: unknown
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3119 $
 */
public class GlobalSource
  extends AbstractGlobalActor
  implements OutputProducer {

  /** for serialization. */
  private static final long serialVersionUID = -99235787254225765L;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Broadcasts the generated output tokens from the source.";
  }

  /**
   * Tries to find the global actor referenced by its global name.
   * Makes sure that the actor produces output.
   *
   * @return		the global actor or null if not found
   */
  protected AbstractActor findGlobalActor() {
    AbstractActor	result;

    result = super.findGlobalActor();

    if (result != null) {
      if (!(result instanceof OutputProducer)) {
	getSystemErr().println("Global actor '" + result.getFullName() + "' does not produce any output!");
	result = null;
      }
    }

    return result;
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		depends on the global actor
   */
  public Class[] generates() {
    if (m_GlobalActor != null)
      return ((OutputProducer) m_GlobalActor).generates();
    else
      return new Class[]{Unknown.class};
  }

  /**
   * Executes the global actor. Derived classes might need to override this
   * method to ensure atomicity.
   *
   * @return		null if no error, otherwise error message
   */
  protected String executeGlobalActor() {
    String	result;

    result = m_GlobalActor.execute();

    return result;
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    return ((OutputProducer) m_GlobalActor).output();
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return ((OutputProducer) m_GlobalActor).hasPendingOutput();
  }
}
