/*
 * TextFileReader.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.io.File;
import java.util.Vector;

import adams.core.io.FileUtils;
import adams.core.io.PlaceholderFile;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Reads a text file and forwards the content. The content can either be forwarded as string array, line by line or as single string.<br/>
 * This actor takes the file to read as input.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String<br/>
 * &nbsp;&nbsp;&nbsp;java.io.File<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String[]<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: TextFileReader
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-output-type &lt;ARRAY_OF_LINES|LINE_BY_LINE|SINGLE_STRING&gt; (property: outputType)
 * &nbsp;&nbsp;&nbsp;The output type defines how to forward the content of the text file.
 * &nbsp;&nbsp;&nbsp;default: ARRAY_OF_LINES
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3319 $
 */
public class TextFileReader
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -184602726110144511L;

  /**
   * Enumeration that defines the type of output to produce.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 3319 $
   */
  public enum OutputType {
    /** array of lines. */
    ARRAY_OF_LINES,
    /** line by line. */
    LINE_BY_LINE,
    /** content as single string. */
    SINGLE_STRING
  }

  /** the type of output to produce. */
  protected OutputType m_OutputType;

  /** the output queue. */
  protected Vector m_Queue;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Reads a text file and forwards the content. The content can either be "
      + "forwarded as string array, line by line or as single string.\n"
      + "This actor takes the file to read as input.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "output-type", "outputType",
	    OutputType.ARRAY_OF_LINES);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Queue = new Vector();
  }

  /**
   * Resets the actor.
   * Clears the output queue.
   */
  protected void reset() {
    super.reset();

    m_Queue.clear();
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("outputType");
    if (variable != null)
      return variable;
    else
      return m_OutputType.toString();
  }

  /**
   * Sets the output type.
   *
   * @param value	the type
   */
  public void setOutputType(OutputType value) {
    m_OutputType = value;
    reset();
  }

  /**
   * Returns the output type.
   *
   * @return		the type
   */
  public OutputType getOutputType() {
    return m_OutputType;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputTypeTipText() {
    return "The output type defines how to forward the content of the text file.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.String.class, java.io.File.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{String.class, File.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.String[].class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    switch (m_OutputType) {
      case ARRAY_OF_LINES:
	return new Class[]{String[].class};

      case LINE_BY_LINE:
      case SINGLE_STRING:
	return new Class[]{String.class};

      default:
	throw new IllegalStateException("Unhandled output type: " + m_OutputType);
    }
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    Object		fileObj;
    File		file;
    Vector<String>	content;
    byte[]		bytes;
    int			i;
    boolean		ok;

    result = null;

    fileObj = m_InputToken.getPayload();
    if (fileObj instanceof File)
      file = (File) fileObj;
    else
      file = new PlaceholderFile((String) fileObj);

    m_Queue.clear();

    switch (m_OutputType) {
      case ARRAY_OF_LINES:
	content = FileUtils.loadFromFile(file);
	ok      = (content != null);
	if (ok)
	  m_Queue.add(content.toArray(new String[content.size()]));
	break;

      case LINE_BY_LINE:
	content = FileUtils.loadFromFile(file);
	ok      = (content != null);
	if (ok) {
	  for (i = 0; i < content.size(); i++)
	    m_Queue.add(content.get(i));
	}
	break;

      case SINGLE_STRING:
	bytes = FileUtils.loadFromBinaryFile(file);
	ok    = (bytes != null);
	if (ok)
	  m_Queue.add(new String(bytes));
	break;

      default:
	throw new IllegalStateException("Unhandled output type: " + m_OutputType);
    }

    if (!ok)
      result = "Error reading file: " + file;

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return (m_Queue.size() > 0);
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    result = new Token(m_Queue.get(0));
    m_Queue.remove(0);

    m_OutputToken = null;
    m_InputToken  = null;

    return result;
  }
}
