/*
 * Tree.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.flow.tree;

import java.awt.datatransfer.Transferable;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import javax.swing.ImageIcon;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.KeyStroke;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import adams.core.ClassLister;
import adams.core.io.FlowFile;
import adams.core.option.AbstractOptionProducer;
import adams.core.option.HtmlHelpProducer;
import adams.core.option.NestedConsumer;
import adams.core.option.NestedProducer;
import adams.core.option.OptionHandler;
import adams.flow.control.Breakpoint;
import adams.flow.control.Flow;
import adams.flow.core.AbstractActor;
import adams.flow.core.AbstractDisplay;
import adams.flow.core.AbstractExternalActor;
import adams.flow.core.AbstractGlobalActor;
import adams.flow.core.ActorExecution;
import adams.flow.core.ActorHandler;
import adams.flow.core.ActorHandlerInfo;
import adams.flow.core.ActorPath;
import adams.flow.core.ActorUtils;
import adams.flow.core.GlobalActorReference;
import adams.flow.core.InputConsumer;
import adams.flow.core.InstantiatableActor;
import adams.flow.core.MutableActorHandler;
import adams.flow.core.OutputProducer;
import adams.flow.processor.AbstractNameUpdater;
import adams.flow.processor.UpdateEventName;
import adams.flow.processor.UpdateGlobalActorName;
import adams.flow.sink.DisplayPanelManager;
import adams.flow.sink.DisplayPanelProvider;
import adams.flow.sink.ExternalSink;
import adams.flow.sink.GlobalSink;
import adams.flow.source.ExternalSource;
import adams.flow.source.GlobalSource;
import adams.flow.standalone.Events;
import adams.flow.standalone.ExternalStandalone;
import adams.flow.standalone.GlobalActors;
import adams.flow.template.AbstractActorTemplate;
import adams.flow.transformer.ExternalTransformer;
import adams.flow.transformer.GlobalTransformer;
import adams.gui.core.BaseMenu;
import adams.gui.core.BaseTreeNode;
import adams.gui.core.DragAndDropTree;
import adams.gui.core.DragAndDropTreeNodeCollection;
import adams.gui.core.GUIHelper;
import adams.gui.core.MouseUtils;
import adams.gui.core.dotnotationtree.AbstractItemFilter;
import adams.gui.dialog.HelpDialog;
import adams.gui.event.ActorChangeEvent;
import adams.gui.event.ActorChangeEvent.Type;
import adams.gui.event.ActorChangeListener;
import adams.gui.event.NodeDroppedEvent;
import adams.gui.event.NodeDroppedEvent.NotificationTime;
import adams.gui.event.NodeDroppedListener;
import adams.gui.flow.FlowEditorDialog;
import adams.gui.flow.FlowEditorPanel;
import adams.gui.flow.FlowPanel;
import adams.gui.goe.FlowHelper;
import adams.gui.goe.GenericObjectEditorDialog;
import adams.gui.goe.classtree.ActorClassTreeFilter;

/**
 * A custom tree for displaying the structure of a flow.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4260 $
 */
public class Tree
  extends DragAndDropTree {

  /** for serialization. */
  private static final long serialVersionUID = -6052602093735801950L;

  /**
   * Enumeration for how to insert a node.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4260 $
   */
  public enum InsertPosition {
    /** beneath the current path. */
    BENEATH,
    /** here at this position. */
    HERE,
    /** after this position. */
    AFTER
  }

  /** the tree itself. */
  protected Tree m_Self;

  /** the owner. */
  protected FlowPanel m_Owner;

  /** the listeners for changes in the actors. */
  protected HashSet<ActorChangeListener> m_ActorChangeListeners;

  /** whether the setup was modified or not. */
  protected boolean m_Modified;

  /** the file this actor is based on (if at all). */
  protected File m_File;

  /** the customizer for the node popup. */
  protected NodePopupMenuCustomizer m_NodePopupMenuCustomizer;

  /** the node found in the last search. */
  protected Node m_LastSearchNode;

  /** the last search string used. */
  protected String m_LastSearchString;

  /** the HTML color string of the actor names (e.g., 'black' or '#000000'). */
  protected String m_ActorNameColor;

  /** the HTML font tag size of the actor names (e.g., '3' or '-1'). */
  protected String m_ActorNameSize;

  /** the HTML color string of the quick info (e.g., 'green' or '#008800'). */
  protected String m_QuickInfoColor;

  /** the HTML font tag size of the quick info (e.g., '3' or '-2'). */
  protected String m_QuickInfoSize;

  /** the HTML color string of the annotations (e.g., 'blue' or '#0000FF'). */
  protected String m_AnnotationsColor;

  /** the HTML font tag size of the annotations (e.g., '3' or '-2'). */
  protected String m_AnnotationsSize;

  /** the HTML color string of the input/output info (e.g., 'green' or '#008800'). */
  protected String m_InputOutputColor;

  /** the HTML font tag size of the input/output info (e.g., '3' or '-2'). */
  protected String m_InputOutputSize;

  /** the HTML color string of the placeholders (e.g., 'navy' or '#0000FF'). */
  protected String m_PlaceholdersColor;

  /** the HTML font tag size of the placeholders (e.g., '3' or '-2'). */
  protected String m_PlaceholdersSize;

  /** the background HTML color string of the variable highlights (e.g., 'red' or '#FFDD88'). */
  protected String m_VariableHighlightBackground;

  /** whether to display the quick info or not. */
  protected boolean m_ShowQuickInfo;

  /** whether to show the annotations or not. */
  protected boolean m_ShowAnnotations;

  /** whether to display the input/output info or not. */
  protected boolean m_ShowInputOutput;

  /** the input/output class prefixes to remove. */
  protected String[] m_InputOutputPrefixes;

  /** the dialog for selecting a template for generating a flow fragment. */
  protected GenericObjectEditorDialog m_TemplateDialog;

  /** whether to store the flow as an object in the "state" or in nested format. */
  protected boolean m_StateUsesNested;

  /** the node that is currently being edited. */
  protected Node m_CurrentEditingNode;

  /** the parent of the currently edited node or node to be added. */
  protected Node m_CurrentEditingParent;

  /** the last template that was added via 'Add from template'. */
  protected AbstractActorTemplate m_LastTemplate;

  /** the position of the last template that was added via 'Add from template'. */
  protected InsertPosition m_LastTemplateInsertPosition;

  /**
   * Initializes the tree.
   *
   * @param owner	the owning panel
   */
  public Tree(FlowPanel owner) {
    this(owner, null);
  }

  /**
   * Initializes the tree.
   *
   * @param owner	the owning panel
   * @param root	the root actor, can be null
   */
  public Tree(FlowPanel owner, AbstractActor root) {
    super();

    m_Owner                       = owner;
    m_ActorNameColor              = "black";
    m_ActorNameSize               = "3";
    m_QuickInfoColor              = "#008800";
    m_QuickInfoSize               = "-2";
    m_AnnotationsColor            = "blue";
    m_AnnotationsSize             = "-2";
    m_PlaceholdersColor           = "navy";
    m_PlaceholdersSize            = "-2";
    m_InputOutputColor            = "grey";
    m_InputOutputSize             = "-2";
    m_VariableHighlightBackground = "#FFDD88";
    m_StateUsesNested             = true;
    m_InputOutputPrefixes         = new String[0];
    m_CurrentEditingNode          = null;
    m_CurrentEditingParent        = null;
    m_Modified                    = false;
    m_File                        = null;
    m_LastTemplate                = null;

    buildTree(root);
  }

  /**
   * Further initialization of the tree.
   */
  protected void initialize() {
    super.initialize();

    m_Self                    = this;
    m_Modified                = false;
    m_NodePopupMenuCustomizer = null;
    m_ActorChangeListeners    = new HashSet<ActorChangeListener>();
    m_LastSearchString        = "";
    m_LastSearchNode          = null;
    m_ShowQuickInfo           = true;
    m_ShowAnnotations         = true;
    m_ShowInputOutput         = false;

    putClientProperty("JTree.lineStyle", "None");
    setSelectionModel(new SelectionModel());
    setCellRenderer(new Renderer());
    setCellEditor(new CellEditor(this, (Renderer) getCellRenderer()));
    setShowsRootHandles(true);
    setToggleClickCount(0);  // to avoid double clicks from toggling expanded/collapsed state

    addMouseListener(new MouseAdapter() {
      public void mousePressed(MouseEvent e) {
        final TreePath selPath = m_Self.getPathForLocation(e.getX(), e.getY());

        if (m_Self.isEnabled() && MouseUtils.isRightClick(e)) {
          e.consume();
          showNodePopupMenu(e);
        }
        else if (m_Self.isEnabled() && MouseUtils.isDoubleClick(e)) {
          e.consume();
          editActor(selPath);
        }
        else {
          super.mousePressed(e);
        }
      }
    });

    addKeyListener(new KeyAdapter() {
      public void keyPressed(KeyEvent e) {
	TreePath path = getSelectionPath();
	TreePath[] paths = getSelectionPaths();

	if (path != null) {
	  KeyStroke ks = KeyStroke.getKeyStrokeForEvent(e);
	  boolean canRemove = canRemoveActors(paths);
	  boolean canPaste  = canPasteActor();
	  Node selNode = (Node) path.getLastPathComponent();
	  Node parent = (Node) selNode.getParent();
	  boolean isMutable = (selNode.getActor() instanceof MutableActorHandler);
	  boolean isParentMutable = (parent != null) && (parent.getActor() instanceof MutableActorHandler);
	  int numSel = ((getSelectionPaths() == null) ? 0 : getSelectionPaths().length);
	  boolean isSingleSel = (numSel == 1);

	  // cut
	  if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Cut"))) && canRemove) {
	    e.consume();
	    cutActors(paths);
	  }
	  // copy
	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Copy"))) &&  (numSel > 0)) {
	    e.consume();
  	    copyActors(paths);
	  }
	  // paste - beneath
	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Paste.Beneath"))) && isSingleSel && canPaste && isMutable) {
	    e.consume();
	    addActor(path, getActorFromClipboard(), InsertPosition.BENEATH);
	  }
	  // paste - here
	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Paste.Here"))) && isSingleSel && canPaste && isParentMutable) {
	    e.consume();
	    addActor(path, getActorFromClipboard(), InsertPosition.HERE);
	  }
	  // paste - after
	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Paste.After"))) && isSingleSel && canPaste && isParentMutable) {
	    e.consume();
	    addActor(path, getActorFromClipboard(), InsertPosition.AFTER);
	  }
	  // add - beneath
	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Add.Beneath"))) && isSingleSel && isMutable) {
	    e.consume();
	    addActor(path, null, InsertPosition.BENEATH);
	  }
	  // add - here
	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Add.Here"))) && isSingleSel && isParentMutable) {
	    e.consume();
	    addActor(path, null, InsertPosition.HERE);
	  }
	  // add - after
	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Add.After"))) && isSingleSel && isParentMutable) {
	    e.consume();
	    addActor(path, null, InsertPosition.AFTER);
	  }
	  // template - beneath
	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Template.Beneath"))) && isSingleSel && isMutable) {
	    e.consume();
	    addFromTemplate(path, null, InsertPosition.BENEATH);
	  }
	  // template - here
	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Template.Here"))) && isSingleSel && isParentMutable) {
	    e.consume();
	    addFromTemplate(path, null, InsertPosition.HERE);
	  }
	  // template - after
	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Template.After"))) && isSingleSel && isParentMutable) {
	    e.consume();
	    addFromTemplate(path, null, InsertPosition.AFTER);
	  }
	  // rename
  	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Rename"))) && isSingleSel) {
	    e.consume();
  	    renameActor(path);
  	  }
	  // edit actor
  	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Edit.Actor"))) && isSingleSel) {
	    e.consume();
  	    editActor(path);
  	  }
	  // edit flow
  	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Edit.Flow"))) && isSingleSel) {
	    e.consume();
	    if (selNode.getActor() instanceof AbstractExternalActor)
	      editFlow(path);
  	  }
	  // remove
  	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Remove"))) && canRemove) {
	    e.consume();
  	    removeActor(paths);
  	  }
	  // enable/disable
  	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("ToggleState"))) && (numSel > 0)) {
	    e.consume();
  	    toggleEnabledState(paths);
  	  }
	  // create global actor
  	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("CreateGlobalActor")))) {
	    e.consume();
  	    createGlobalActor(path);
  	  }
	  // externalize
  	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Externalize")))) {
	    e.consume();
  	    externalizeActor(path);
  	  }
	  // expand all
  	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("ExpandAll")))) {
	    e.consume();
	    expandAll(path);
  	  }
	  // collapse all
  	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("CollapseAll")))) {
	    e.consume();
	    collapseAll(path);
  	  }
	  // help
  	  else if (ks.equals(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Help")))) {
	    e.consume();
  	    help(path);
  	  }
	}

	if (!e.isConsumed())
	  super.keyPressed(e);
      }
    });

    addNodeDroppedListener(new NodeDroppedListener() {
      public void nodeDropped(NodeDroppedEvent e) {
	BaseTreeNode[] tnodes = (BaseTreeNode[]) e.getNodes();
	ArrayList<Node> nodes = new ArrayList<Node>();

	// update actor name, if necessary
	if (e.getNotificationTime() == NotificationTime.FINISHED) {
	  for (BaseTreeNode node: tnodes) {
	    if (node instanceof Node) {
	      if (uniqueName((Node) node))
		nodeStructureChanged((Node) node);
	    }
	  }
	}

	// undo + modified
	if (e.getNotificationTime() == NotificationTime.BEFORE) {
	  if (nodes.size() == 1)
	    addUndoPoint("Drag'n'Drop of actor '" + nodes.get(0).getActor().getName() + "'");
	  else
	    addUndoPoint("Drag'n'Drop of " + nodes.size() + " actors");
	}
	else {
	  m_Modified = true;
	  notifyActorChangeListeners(new ActorChangeEvent(m_Self, nodes.toArray(new Node[nodes.size()]), Type.MODIFY));
	}
      }
    });
  }

  /**
   * Sets the tree's selection model. When a <code>null</code> value is
   * specified an empty
   * <code>selectionModel</code> is used, which does not allow selections.
   *
   * @param selectionModel the <code>TreeSelectionModel</code> to use,
   *		or <code>null</code> to disable selections
   * @see TreeSelectionModel
   */
  public void setSelectionModel(TreeSelectionModel selectionModel) {
    if (!(selectionModel instanceof SelectionModel))
      throw new IllegalArgumentException(
	  "Only " + SelectionModel.class.getName() + " models are allowed");

    super.setSelectionModel(selectionModel);
  }

  /**
   * Builds the tree with the given root.
   *
   * @param root	the root actor, can be null
   */
  public void buildTree(AbstractActor root) {
    DefaultTreeModel		model;
    TreeModel			modelOld;
    DefaultMutableTreeNode	rootNode;

    modelOld = null;
    if (getModel() instanceof TreeModel)
      modelOld = (TreeModel) getModel();

    if (root == null) {
      model = new DefaultTreeModel(null);
    }
    else {
      rootNode = buildTree(null, root, true);
      model    = new DefaultTreeModel(rootNode);
    }

    setModel(model);

    if (model.getRoot() != null)
      expandPath(new TreePath(model.getRoot()));

    // clean up old model
    if (modelOld != null)
      modelOld.destroy();
  }

  /**
   * Builds the tree recursively.
   *
   * @param parent	the parent to add the actor to
   * @param actor	the actor to add
   * @param append	whether to append the sub-tree to the parent or just
   * 			return it (recursive calls always append the sub-tree!)
   * @return		the generated node
   */
  protected Node buildTree(Node parent, AbstractActor actor, boolean append) {
    return buildTree(parent, new AbstractActor[]{actor}, append)[0];
  }

  /**
   * Builds the tree recursively.
   *
   * @param parent	the parent to add the actor to
   * @param actors	the actors to add
   * @param append	whether to append the sub-tree to the parent or just
   * 			return it (recursive calls always append the sub-tree!)
   * @return		the generated nodes
   */
  protected Node[] buildTree(Node parent, AbstractActor[] actors, boolean append) {
    Node[]	result;
    int		n;
    int		i;

    result = new Node[actors.length];
    for (n = 0; n < actors.length; n++) {
      result[n] = new Node(this, actors[n]);
      if ((parent != null) && append)
	parent.add(result[n]);

      if (actors[n] instanceof ActorHandler) {
	for (i = 0; i < ((ActorHandler) actors[n]).size(); i++)
	  buildTree(result[n], ((ActorHandler) actors[n]).get(i), true);
      }
    }

    return result;
  }

  /**
   * Ensures that the name of the actor stored in the node is unique among
   * its siblings.
   *
   * @param node	the actor's name of this node is made unique
   * @return		true if the actor's name was modified
   */
  protected boolean uniqueName(Node node) {
    boolean		result;
    Node 		parent;
    AbstractActor	actor;
    HashSet<String> 	names;
    int			i;

    result = false;

    parent = (Node) node.getParent();
    if (parent != null) {
      if (parent.getActor() instanceof ActorHandler) {
	actor = node.getActor();
	names = new HashSet<String>();
	for (i = 0; i < parent.getChildCount(); i++) {
	  if (parent.getChildAt(i) == node)
	    continue;
	  names.add(((Node) parent.getChildAt(i)).getActor().getName());
	}
	result = ActorUtils.uniqueName(actor, names);
	if (result)
	  node.setActor(actor);
      }
    }

    return result;
  }

  /**
   * Returns the owning panel.
   *
   * @return		the panel
   */
  public FlowPanel getOwner() {
    return m_Owner;
  }

  /**
   * Returns the owning editor.
   *
   * @return		the editor
   */
  public FlowEditorPanel getEditor() {
    return m_Owner.getEditor();
  }

  /**
   * Sets the node popup menu customizer to use.
   *
   * @param value	the customizer, null to disable
   */
  public void setNodePopupMenuCustomizer(NodePopupMenuCustomizer value) {
    m_NodePopupMenuCustomizer = value;
  }

  /**
   * Returns the current node popup menu customizer in use, if any.
   *
   * @return		the customizer, can be null if default used
   */
  public NodePopupMenuCustomizer getNodePopupMenuCustomizer() {
    return m_NodePopupMenuCustomizer;
  }

  /**
   * Sets the HTML color string for the actor names.
   *
   * @param value	the HTML color string
   */
  public void setActorNameColor(String value) {
    m_ActorNameColor = value;
  }

  /**
   * Returns the HTML color string for the actor names.
   *
   * @return		the HTML color string
   */
  public String getActorNameColor() {
    return m_ActorNameColor;
  }

  /**
   * Sets the HTML font tag size string for the actor names.
   *
   * @param value	the HTML font tag size string
   */
  public void setActorNameSize(String value) {
    m_ActorNameSize = value;
  }

  /**
   * Returns the HTML font tag size string for the actor names.
   *
   * @return		the HTML font tag size string
   */
  public String getActorNameSize() {
    return m_ActorNameSize;
  }

  /**
   * Sets the HTML color string for the annotations.
   *
   * @param value	the HTML color string
   */
  public void setAnnotationsColor(String value) {
    m_AnnotationsColor = value;
  }

  /**
   * Returns the HTML color string for the annotations.
   *
   * @return		the HTML color string
   */
  public String getAnnotationsColor() {
    return m_AnnotationsColor;
  }

  /**
   * Sets the HTML font tag size string for the annotations.
   *
   * @param value	the HTML font tag size string
   */
  public void setAnnotationsSize(String value) {
    m_AnnotationsSize = value;
  }

  /**
   * Returns the HTML font tag size string for the annotations.
   *
   * @return		the HTML font tag size string
   */
  public String getAnnotationsSize() {
    return m_AnnotationsSize;
  }

  /**
   * Sets the HTML color string for the quick info.
   *
   * @param value	the HTML color string
   */
  public void setQuickInfoColor(String value) {
    m_QuickInfoColor = value;
  }

  /**
   * Returns the HTML color string for the quick info.
   *
   * @return		the HTML color string
   */
  public String getQuickInfoColor() {
    return m_QuickInfoColor;
  }

  /**
   * Sets the HTML font tag size string for the quick info.
   *
   * @param value	the HTML font tag size string
   */
  public void setQuickInfoSize(String value) {
    m_QuickInfoSize = value;
  }

  /**
   * Returns the HTML font tag size string for the quick info.
   *
   * @return		the HTML font tag size string
   */
  public String getQuickInfoSize() {
    return m_QuickInfoSize;
  }

  /**
   * Sets the HTML color string for the input/output information.
   *
   * @param value	the HTML color string
   */
  public void setInputOutputColor(String value) {
    m_InputOutputColor = value;
  }

  /**
   * Returns the HTML color string for the input/output information.
   *
   * @return		the HTML color string
   */
  public String getInputOutputColor() {
    return m_InputOutputColor;
  }

  /**
   * Sets the HTML font tag size string for the input/output information.
   *
   * @param value	the HTML font tag size string
   */
  public void setInputOutputSize(String value) {
    m_InputOutputSize = value;
  }

  /**
   * Returns the HTML font tag size string for the input/output information.
   *
   * @return		the HTML font tag size string
   */
  public String getInputOutputSize() {
    return m_InputOutputSize;
  }

  /**
   * Sets the HTML color string for the placeholders.
   *
   * @param value	the HTML color string
   */
  public void setPlaceholdersColor(String value) {
    m_PlaceholdersColor = value;
  }

  /**
   * Returns the HTML color string for the placeholders.
   *
   * @return		the HTML color string
   */
  public String getPlaceholdersColor() {
    return m_PlaceholdersColor;
  }

  /**
   * Sets the HTML font tag size string for the quick info.
   *
   * @param value	the HTML font tag size string
   */
  public void setPlaceholdersSize(String value) {
    m_PlaceholdersSize = value;
  }

  /**
   * Returns the HTML font tag size string for the placeholders.
   *
   * @return		the HTML font tag size string
   */
  public String getPlaceholdersSize() {
    return m_PlaceholdersSize;
  }

  /**
   * Sets the HTML background color string for the variable highlights.
   *
   * @param value	the HTML color string
   */
  public void setVariableHighlightBackground(String value) {
    m_VariableHighlightBackground = value;
  }

  /**
   * Returns the HTML background color string for the variable highlights.
   *
   * @return		the HTML color string
   */
  public String getVariableHighlightBackground() {
    return m_VariableHighlightBackground;
  }

  /**
   * Sets whether to use nested format or objects when generating the "state".
   *
   * @param value	if true then the nested format is used
   * @see		#getState()
   */
  public void setStateUsesNested(boolean value) {
    m_StateUsesNested = value;
  }

  /**
   * Returns whether the nested format or the objects are used when generating
   * the "state" of the tree.
   *
   * @return		true if the nested format is used instead of objects
   */
  public boolean getStateUsesNested() {
    return m_StateUsesNested;
  }

  /**
   * Shortcut method for notifying model about node structure change.
   *
   * @param node	the node the triggered the structural change
   */
  public void nodeStructureChanged(Node node) {
    if (node != null)
      node.invalidateRendering();
    ((DefaultTreeModel) getModel()).nodeStructureChanged(node);
  }

  /**
   * Returns the actor stored on the clipboard.
   *
   * @return		the actor or null if none available
   */
  protected AbstractActor getActorFromClipboard() {
    AbstractActor	result;
    NestedConsumer	consumer;

    result = null;

    try {
      if (GUIHelper.canPasteFromClipboard()) {
	consumer = new NestedConsumer();
	consumer.setQuiet(true);
	result = (AbstractActor) consumer.fromString(GUIHelper.pasteSetupFromClipboard());
	consumer.cleanUp();
      }
    }
    catch (Exception ex) {
      result = null;
    }

    return result;
  }

  /**
   * Puts the actor in nested form on the clipboard.
   *
   * @param actor	the actor to put on the clipboard
   */
  protected void putActorOnClipboard(AbstractActor actor) {
    putActorOnClipboard(new AbstractActor[]{actor});
  }

  /**
   * Puts the actors in nested form on the clipboard.
   *
   * @param actors	the actors to put on the clipboard
   */
  protected void putActorOnClipboard(AbstractActor[] actors) {
    ClipboardActorContainer	cont;

    if (actors.length == 1) {
      GUIHelper.copyToClipboard(AbstractOptionProducer.toString(NestedProducer.class, actors[0]));
    }
    else if (actors.length > 1) {
      cont = new ClipboardActorContainer();
      cont.setActors(actors);
      GUIHelper.copyToClipboard(cont.toNestedString());
    }
  }

  /**
   * Checks whether an actor can be pasted.
   *
   * @return		true if pasting is possible
   */
  protected boolean canPasteActor() {
    return (getActorFromClipboard() != null);
  }

  /**
   * Checks whether the specified actors can be removed, e.g., "cut" and placed
   * on the clipboard.
   *
   * @param paths	the paths to the actors
   * @return		true if the actors can be removed
   */
  protected boolean canRemoveActors(TreePath[] paths) {
    boolean	result;
    Node	node;
    Node	parent;

    result = (paths.length > 0);

    for (TreePath path: paths) {
      node   = (Node) path.getLastPathComponent();
      parent = (Node) node.getParent();
      result =    (parent != null)
               && (parent.getActor() instanceof MutableActorHandler);
      if (!result)
	break;
    }

    return result;
  }

  /**
   * Shows a popup if possible for the given mouse event.
   *
   * @param e		the event
   */
  protected void showNodePopupMenu(MouseEvent e) {
    JPopupMenu		menu;
    BaseMenu		submenu;
    JMenuItem		menuitem;
    final TreePath 	selPath;
    Node 		selNode;
    boolean		canRemove;
    boolean		canPaste;
    boolean		isMutable;
    boolean		isParentMutable;
    Node		parent;
    String[]		actors;
    int			i;
    final int		numSel;
    boolean		isSingleSel;
    final TreePath[] 	paths;
    Node		currNode;

    menu     = null;
    paths    = getSelectionPaths();
    numSel   = ((paths == null) ? 0 : paths.length);
    if (getPathForLocation(e.getX(), e.getY()) == null)
      return;
    currNode = (Node) getPathForLocation(e.getX(), e.getY()).getLastPathComponent();

    isSingleSel = (numSel == 1);
    if (isSingleSel)
      selPath = paths[0];
    else
      selPath = null;
    selNode = null;
    parent  = null;
    if (numSel > 0) {
      selNode = (Node) paths[0].getLastPathComponent();
      parent  = (Node) selNode.getParent();
    }

    // can the node be deleted?
    canRemove = isEditable() && (numSel > 0) && canRemoveActors(paths);

    // is clipboard content an actor?
    canPaste = isEditable() && (numSel > 0) && canPasteActor();

    // mutable actor handlers?
    isMutable       = isEditable() && (selNode != null) && (selNode.getActor() instanceof MutableActorHandler);
    isParentMutable = isEditable() && (parent != null) && (parent.getActor() instanceof MutableActorHandler);

    menu = new JPopupMenu();

    menuitem = new JMenuItem(currNode.getActor().getName());
    menuitem.setEnabled(false);
    menuitem.setIcon(GUIHelper.getIcon("flow.gif"));
    menu.add(menuitem);
    menu.addSeparator();

    if (isEditable())
      menuitem = new JMenuItem("Edit...");
    else
      menuitem = new JMenuItem("Show...");
    menuitem.setEnabled(isSingleSel);
    menuitem.setIcon(GUIHelper.getIcon("properties.gif"));
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Edit.Actor")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	editActor(selPath);
      }
    });
    menu.add(menuitem);

    if (currNode.getActor() instanceof AbstractExternalActor) {
      menuitem = new JMenuItem("Edit flow...");
      menuitem.setEnabled(isSingleSel);
      menuitem.setIcon(GUIHelper.getIcon("flow.gif"));
      menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Edit.Flow")));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  editFlow(selPath);
	}
      });
      menu.add(menuitem);
    }

    menu.addSeparator();

    menuitem = new JMenuItem("Add beneath...");
    menuitem.setEnabled(isSingleSel && isMutable);
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Add.Beneath")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	addActor(selPath, null, InsertPosition.BENEATH);
      }
    });
    menu.add(menuitem);

    menuitem = new JMenuItem("Add here...");
    menuitem.setEnabled(isSingleSel && isParentMutable);
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Add.Here")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	addActor(selPath, null, InsertPosition.HERE);
      }
    });
    menu.add(menuitem);

    menuitem = new JMenuItem("Add after...");
    menuitem.setEnabled(isSingleSel && isParentMutable);
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Add.After")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	addActor(selPath, null, InsertPosition.AFTER);
      }
    });
    menu.add(menuitem);

    menu.addSeparator();

    menuitem = new JMenuItem("Cut");
    menuitem.setIcon(GUIHelper.getIcon("cut.gif"));
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Cut")));
    menuitem.setEnabled(canRemove);
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	cutActors(paths);
      }
    });
    menu.add(menuitem);

    menuitem = new JMenuItem("Copy");
    menuitem.setIcon(GUIHelper.getIcon("copy.gif"));
    menuitem.setEnabled(numSel > 0);
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Copy")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	copyActors(paths);
      }
    });
    menu.add(menuitem);

    submenu = new BaseMenu("Paste");
    submenu.setIcon(GUIHelper.getIcon("paste.gif"));
    submenu.setEnabled(canPaste);
    menu.add(submenu);

    menuitem = new JMenuItem("Add beneath");
    menuitem.setEnabled(canPaste && isMutable);
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Paste.Beneath")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	addActor(selPath, getActorFromClipboard(), InsertPosition.BENEATH);
      }
    });
    submenu.add(menuitem);

    menuitem = new JMenuItem("Add here");
    menuitem.setEnabled(canPaste && isParentMutable);
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Paste.Here")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	addActor(selPath, getActorFromClipboard(), InsertPosition.HERE);
      }
    });
    submenu.addSeparator();
    submenu.add(menuitem);

    menuitem = new JMenuItem("Add after");
    menuitem.setEnabled(canPaste && isParentMutable);
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Paste.After")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	addActor(selPath, getActorFromClipboard(), InsertPosition.AFTER);
      }
    });
    submenu.add(menuitem);

    menu.addSeparator();

    menuitem = new JMenuItem();
    menuitem.setEnabled(isEditable() && (numSel > 0));
    if (!isSingleSel) {
      menuitem.setText("Toggle state");
    }
    else {
      if (!currNode.getActor().getSkip())
	menuitem.setText("Disable");
      else
	menuitem.setText("Enable");
    }
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("ToggleState")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	toggleEnabledState(paths);
      }
    });
    menu.add(menuitem);

    menuitem = new JMenuItem("Rename...");
    menuitem.setEnabled(isSingleSel && isEditable());
    menuitem.setIcon(GUIHelper.getEmptyIcon());
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Rename")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	renameActor(selPath);
      }
    });
    menu.add(menuitem);

    menu.addSeparator();

    menuitem = new JMenuItem("Remove");
    menuitem.setIcon(GUIHelper.getIcon("delete.gif"));
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Remove")));
    menuitem.setEnabled(canRemove);
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	removeActor(paths);
      }
    });
    menu.add(menuitem);

    menu.addSeparator();

    submenu = new BaseMenu("Template");
    submenu.setIcon(GUIHelper.getEmptyIcon());
    submenu.setEnabled(isSingleSel);
    menu.add(submenu);

    menuitem = new JMenuItem("Add beneath...");
    menuitem.setEnabled(isMutable);
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Template.Beneath")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	addFromTemplate(selPath, null, InsertPosition.BENEATH);
      }
    });
    submenu.add(menuitem);

    menuitem = new JMenuItem("Add here...");
    menuitem.setEnabled(isParentMutable);
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Template.Here")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	addFromTemplate(selPath, null, InsertPosition.HERE);
      }
    });
    submenu.addSeparator();
    submenu.add(menuitem);

    menuitem = new JMenuItem("Add after...");
    menuitem.setEnabled(isParentMutable);
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Template.After")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	addFromTemplate(selPath, null, InsertPosition.AFTER);
      }
    });
    submenu.add(menuitem);

    if (m_LastTemplate != null) {
      menuitem = new JMenuItem("Last template...");
      menuitem.setEnabled(((m_LastTemplateInsertPosition == InsertPosition.BENEATH) && isMutable) || isParentMutable);
      menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Template.Last")));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  addFromTemplate(selPath, m_LastTemplate, m_LastTemplateInsertPosition);
	}
      });
      menu.add(menuitem);
    }

    submenu = new BaseMenu("Enclose");
    submenu.setEnabled(isEditable() && (parent != null) && (numSel > 0));
    submenu.setIcon(GUIHelper.getEmptyIcon());
    menu.add(submenu);

    actors = ClassLister.getSingleton().getClassnames(ActorHandler.class);
    for (i = 0; i < actors.length; i++) {
      final ActorHandler actor = (ActorHandler) AbstractActor.forName(actors[i], new String[0]);
      if (actor instanceof Flow)
	continue;
      if ((paths != null) && (paths.length > 1) && (!(actor instanceof MutableActorHandler)))
	continue;
      menuitem = new JMenuItem(actors[i].replaceAll(".*\\.", ""));
      submenu.add(menuitem);
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  encloseActor(paths, actor);
	}
      });
    }
    submenu.sort();

    if (isSingleSel && (selNode.getActor() instanceof DisplayPanelProvider)) {
      submenu.addSeparator();
      menuitem = new JMenuItem(DisplayPanelManager.class.getSimpleName());
      submenu.add(menuitem);
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  encloseInDisplayPanelManager(paths[0]);
	}
      });
    }

    menuitem = new JMenuItem("Create global actor");
    menuitem.setEnabled(isSingleSel && (getOwner() != null));
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("CreateGlobalActor")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	createGlobalActor(selPath);
      }
    });
    menu.add(menuitem);

    menuitem = new JMenuItem("Externalize...");
    menuitem.setEnabled(isSingleSel && (getOwner() != null));
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Externalize")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	externalizeActor(selPath);
      }
    });
    menu.add(menuitem);

    menu.addSeparator();

    menuitem = new JMenuItem("Expand all");
    menuitem.setIcon(GUIHelper.getIcon("tree.gif"));
    menuitem.setEnabled(isSingleSel && (currNode.getChildCount() > 0));
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("ExpandAll")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	expandAll(selPath);
      }
    });
    menu.add(menuitem);

    menuitem = new JMenuItem("Collapse all");
    menuitem.setIcon(GUIHelper.getEmptyIcon());
    menuitem.setEnabled(isSingleSel && (currNode.getChildCount() > 0));
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("CollapseAll")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	collapseAll(selPath);
      }
    });
    menu.add(menuitem);

    menu.addSeparator();

    menuitem = new JMenuItem("Help...");
    menuitem.setIcon(GUIHelper.getIcon("help.gif"));
    menuitem.setEnabled(isSingleSel);
    menuitem.setAccelerator(GUIHelper.getKeyStroke(getEditor().getTreeShortcut("Help")));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	help(selPath);
      }
    });
    menu.add(menuitem);

    // customize the menu?
    if ((menu != null) && (m_NodePopupMenuCustomizer != null))
      m_NodePopupMenuCustomizer.customizeNodePopupMenu(this, menu, paths);

    if (menu != null)
      menu.show(this, e.getX(), e.getY());
  }

  /**
   * Adds an undo point with the given comment.
   *
   * @param comment	the comment for the undo point
   */
  protected void addUndoPoint(String comment) {
    if (getOwner() != null)
      getOwner().addUndoPoint("Saving undo data...", comment);
  }

  /**
   * Cuts the selected actors and places them on the clipboard.
   *
   * @param paths	the paths to the actors
   */
  protected void cutActors(TreePath[] paths) {
    Node		selNode;
    AbstractActor[]	actors;
    int			i;

    actors = new AbstractActor[paths.length];
    for (i = 0; i < paths.length; i++) {
      selNode   = (Node) paths[i].getLastPathComponent();
      actors[i] = selNode.getFullActor();
    }
    putActorOnClipboard(actors);
    removeActor(paths);
  }

  /**
   * Copies the selected actors and places them on the clipboard.
   *
   * @param paths	the paths to the actors
   */
  protected void copyActors(TreePath[] paths) {
    Node		selNode;
    AbstractActor[]	actors;
    int			i;

    actors = new AbstractActor[paths.length];
    for (i = 0; i < paths.length; i++) {
      selNode   = (Node) paths[i].getLastPathComponent();
      actors[i] = selNode.getFullActor();
    }
    putActorOnClipboard(actors);
  }

  /**
   * Toggles the enabled state of actors.
   *
   * @param paths	the paths to the actors
   */
  protected void toggleEnabledState(TreePath[] paths) {
    Node[]		nodes;
    AbstractActor	actor;
    int			i;

    nodes = new Node[paths.length];
    for (i = 0; i < paths.length; i++)
      nodes[i] = (Node) paths[i].getLastPathComponent();

    for (i = 0; i < nodes.length; i++) {
      actor = nodes[i].getActor();
      actor.setSkip(!actor.getSkip());
      nodes[i].setActor(actor);
      ((DefaultTreeModel) getModel()).nodeChanged(nodes[i]);
    }

    m_Modified = true;
    if (nodes.length == 1)
      notifyActorChangeListeners(new ActorChangeEvent(m_Self, nodes[0], Type.MODIFY));
    else
      notifyActorChangeListeners(new ActorChangeEvent(m_Self, nodes, Type.MODIFY_RANGE));
  }

  /**
   * Renames an actor.
   *
   * @param path	the path to the actor
   */
  protected void renameActor(TreePath path) {
    String			oldName;
    String 			newName;
    Node			selNode;
    AbstractActor		actor;
    AbstractNameUpdater		updater;
    Node			parent;

    selNode = (Node) path.getLastPathComponent();
    oldName = selNode.getActor().getName();
    newName = JOptionPane.showInputDialog("Please input new name:", oldName);
    if (newName != null) {
      actor = selNode.getActor();
      // make sure name is not empty
      if (newName.length() == 0)
	newName = actor.getDefaultName();
      getOwner().addUndoPoint("Saving undo data...", "Renaming actor " + actor.getName() + " to " + newName);
      actor.setName(newName);
      selNode.setActor(actor);
      ((DefaultTreeModel) getModel()).nodeChanged(selNode);
      m_Modified = true;
      notifyActorChangeListeners(new ActorChangeEvent(m_Self, selNode, Type.MODIFY));
      setSelectionPath(new TreePath(selNode.getPath()));
      // shall we update all occurrences?
      parent = (Node) selNode.getParent();
      if (parent != null) {
	updater = null;
	if (parent.getActor() instanceof GlobalActors)
	  updater = new UpdateGlobalActorName();
	else if (parent.getActor() instanceof Events)
	  updater = new UpdateEventName();
	if (updater != null) {
	  if (JOptionPane.showConfirmDialog(this, "Update all occurrences of '" + oldName + "' with '" + newName + "'?") == JOptionPane.YES_OPTION) {
	    updater.setOldName(oldName);
	    updater.setNewName(newName);
	    updater.process(getActor());
	    if (updater.isModified())
	      setActor(updater.getModifiedActor());
	  }
	}
      }
    }
  }

  /**
   * Tries to figure what actors fit best in the tree at the given position.
   *
   * @param path	the path where to insert the actors
   * @param position	how the actors are to be inserted
   * @return		the actors
   */
  protected AbstractActor[] suggestActors(TreePath path, InsertPosition position) {
    AbstractActor[]	result;
    AbstractActor	parent;
    Node		parentNode;
    Node		node;
    int			pos;
    AbstractActor[]	actors;
    int			i;
    AbstractActor[]	suggestions;

    result = null;

    if (result == null) {
      if (position == InsertPosition.BENEATH) {
	parentNode = (Node) path.getLastPathComponent();
	pos        = parentNode.getChildCount();
      }
      else {
	node       = (Node) path.getLastPathComponent();
	parentNode = (Node) node.getParent();
	pos        = parentNode.getIndex(node);
	if (position == InsertPosition.AFTER)
	  pos++;
      }

      parent  = parentNode.getActor();
      actors  = new AbstractActor[parentNode.getChildCount()];
      for (i = 0; i < actors.length; i++)
	actors[i] = ((Node) parentNode.getChildAt(i)).getActor();

      suggestions = ActorSuggestion.getSingleton().suggest(parent, pos, actors);
      if (suggestions.length > 0)
	result = suggestions;
    }

    // default is "Filter"
    if (result == null)
      result = ActorSuggestion.getSingleton().getDefaults();

    return result;
  }

  /**
   * Tries to figure what actor templates fit best in the tree at the given position.
   *
   * @param path	the path where to insert the actor templates
   * @param position	how the actor templates are to be inserted
   * @return		the actor templates
   */
  protected AbstractActorTemplate[] suggestActorTemplates(TreePath path, InsertPosition position) {
    AbstractActorTemplate[]	result;
    AbstractActor		parent;
    Node			parentNode;
    Node			node;
    int				pos;
    AbstractActor[]		actors;
    int				i;
    AbstractActorTemplate[]	suggestions;

    result = null;

    if (result == null) {
      if (position == InsertPosition.BENEATH) {
	parentNode = (Node) path.getLastPathComponent();
	pos        = parentNode.getChildCount();
      }
      else {
	node       = (Node) path.getLastPathComponent();
	parentNode = (Node) node.getParent();
	pos        = parentNode.getIndex(node);
	if (position == InsertPosition.AFTER)
	  pos++;
      }

      parent  = parentNode.getActor();
      actors  = new AbstractActor[parentNode.getChildCount()];
      for (i = 0; i < actors.length; i++)
	actors[i] = ((Node) parentNode.getChildAt(i)).getActor();

      suggestions = ActorTemplateSuggestion.getSingleton().suggest(parent, pos, actors);
      if (suggestions.length > 0)
	result = suggestions;
    }

    // default is "Filter"
    if (result == null)
      result = ActorTemplateSuggestion.getSingleton().getDefaults();

    return result;
  }

  /**
   * Checks whether standalones can be placed beneath the parent actor.
   * If the actor isn't a standalone, this method returns true, of course.
   * In case the actor is a standalone and the parent doesn't allow standalones
   * to be placed, an error message pops up and informs the user.
   *
   * @param actor	the actor to place beneath the parent
   * @param parent	the parent to place the actor beneath
   * @return		true if actor can be placed, false if not
   */
  protected boolean checkForStandalones(AbstractActor actor, Node parent) {
    return checkForStandalones(new AbstractActor[]{actor}, parent);
  }

  /**
   * Checks whether standalones can be placed beneath the parent actor.
   * If the actors contain no standalones, this method returns true, of course.
   * In case the actors contain a standalone and the parent doesn't allow standalones
   * to be placed, an error message pops up and informs the user.
   *
   * @param actors	the actors to place beneath the parent
   * @param parent	the parent to place the actor beneath
   * @return		true if actor can be placed, false if not
   */
  protected boolean checkForStandalones(AbstractActor[] actors, Node parent) {
    for (AbstractActor actor: actors) {
      if (    ActorUtils.isStandalone(actor)
	  && (parent != null)
	  && (parent.getActor() instanceof ActorHandler)
	  && !((ActorHandler) parent.getActor()).getActorHandlerInfo().canContainStandalones()) {

	GUIHelper.showErrorMessage(
	    m_Self, "Actor '" + parent.getFullName() + "' cannot contain standalones!");
	return false;
      }
    }

    return true;
  }

  /**
   * Configures a filter for the ClassTree.
   *
   * @param path	the path where to insert the actor
   * @param position	where to add the actor, if null "editing" an existing actor is assumed
   * @return		the configured filter
   */
  protected AbstractItemFilter configureFilter(TreePath path, InsertPosition position) {
    ActorClassTreeFilter	result;
    AbstractActor		before;
    AbstractActor		after;
    AbstractActor		parent;
    Node			parentNode;
    Node			node;
    int				index;
    ActorHandlerInfo		handlerInfo;

    result      = new ActorClassTreeFilter();
    after       = null;
    before      = null;
    parentNode  = null;
    handlerInfo = null;

    // edit/update current actor
    if (position == null) {
      node       = (Node) path.getLastPathComponent();
      parentNode = (Node) node.getParent();
      if (parentNode != null) {
	parent = parentNode.getActor();
	if (parent instanceof MutableActorHandler) {
	  handlerInfo = ((MutableActorHandler) parent).getActorHandlerInfo();
	  if (handlerInfo.getActorExecution() == ActorExecution.SEQUENTIAL) {
	    index = parentNode.getIndex(node);
	    if (index > 0)
	      before = ((Node) parentNode.getChildAt(index - 1)).getActor();
	    if (index + 1 < parentNode.getChildCount())
	      after = ((Node) parentNode.getChildAt(index + 1)).getActor();
	  }
	}
      }
    }
    // add beneath
    else if (position == InsertPosition.BENEATH) {
      parentNode = (Node) path.getLastPathComponent();
      if (parentNode.getChildCount() > 0)
	before = ((Node) parentNode.getChildAt(parentNode.getChildCount() - 1)).getActor();
    }
    // add here
    else if (position == InsertPosition.HERE) {
      node       = (Node) path.getLastPathComponent();
      parentNode = (Node) node.getParent();
      index      = parentNode.getIndex(node);
      after      = node.getActor();
      if (index > 0)
	before = ((Node) parentNode.getChildAt(index - 1)).getActor();
    }
    // add after
    else if (position == InsertPosition.AFTER) {
      node       = (Node) path.getLastPathComponent();
      parentNode = (Node) node.getParent();
      index      = parentNode.getIndex(node);
      before     = node.getActor();
      if (index + 1 < parentNode.getChildCount())
	after = ((Node) parentNode.getChildAt(index + 1)).getActor();
    }

    // check types
    if ((before != null) && !(before instanceof OutputProducer))
      before = null;
    if ((after != null) && !(after instanceof InputConsumer))
      after = null;

    // set constraints
    if (before != null)
      result.setAccepts(((OutputProducer) before).generates());
    else
      result.setAccepts(null);
    if (after != null)
      result.setGenerates(((InputConsumer) after).accepts());
    else
      result.setGenerates(null);

    if ((handlerInfo == null) && (parentNode != null)) {
      parent = parentNode.getActor();
      if (parent instanceof ActorHandler)
	handlerInfo = ((ActorHandler) parent).getActorHandlerInfo();
    }

    // standalones?
    result.setStandalonesAllowed(false);
    if ((handlerInfo != null) && handlerInfo.canContainStandalones()) {
      // standalones can only be added at the start
      if ((before == null) || ((before != null) && ActorUtils.isStandalone(before)))
	result.setStandalonesAllowed(true);
    }

    // sources?
    result.setSourcesAllowed(false);
    if ((handlerInfo != null) && handlerInfo.canContainSource()) {
      // source can only be added at the start
      if ((before == null) || ((before != null) && ActorUtils.isStandalone(before)))
	result.setSourcesAllowed(true);
    }

    // restrictions?
    if ((handlerInfo != null) && handlerInfo.hasRestrictions())
      result.setRestrictions(handlerInfo.getRestrictions());

    return result;
  }

  /**
   * Brings up the GOE dialog for adding an actor if no actor supplied,
   * otherwise just adds the given actor at the position specified
   * by the path.
   *
   * @param path	the path to the actor to add the new actor sibling
   * @param actor	the actor to add, if null a GOE dialog is presented
   * @param position	where to insert the actor
   */
  protected void addActor(TreePath path, AbstractActor actor, InsertPosition position) {
    GenericObjectEditorDialog	dialog;
    Node			node;
    Node			parent;
    int				index;
    Node[]			children;
    AbstractActor[]		actors;
    String			txt;

    if (actor == null) {
      node = (Node) path.getLastPathComponent();
      if (position == InsertPosition.BENEATH)
	m_CurrentEditingParent = node;
      else
	m_CurrentEditingParent = (Node) node.getParent();
      dialog = GenericObjectEditorDialog.createDialog(this);
      if (position == InsertPosition.HERE)
	dialog.setTitle("Add here...");
      else if (position == InsertPosition.AFTER)
	dialog.setTitle("Add after...");
      else if (position == InsertPosition.BENEATH)
	dialog.setTitle("Add beneath...");
      actors = suggestActors(path, position);
      dialog.getGOEEditor().setCanChangeClassInDialog(true);
      dialog.getGOEEditor().setClassType(AbstractActor.class);
      dialog.setProposedClasses(actors);
      dialog.setCurrent(actors[0]);
      dialog.setLocationRelativeTo(this);
      dialog.getGOEEditor().setFilter(configureFilter(path, position));
      dialog.setVisible(true);
      m_CurrentEditingParent = null;
      if (dialog.getResult() == GenericObjectEditorDialog.APPROVE_OPTION)
        addActor(path, (AbstractActor) dialog.getEditor().getValue(), position);
    }
    else {
      if (position == InsertPosition.BENEATH) {
	node = (Node) path.getLastPathComponent();

	// does actor handler allow standalones?
	if (actor instanceof ClipboardActorContainer)
	  actors = ((ClipboardActorContainer) actor).getActors();
	else
	  actors = new AbstractActor[]{actor};
	if (actors.length < 1)
	  return;
	if (!checkForStandalones(actors, node))
	  return;

	if (actors.length == 1)
	  txt = "'" + actors[0].getName() + "'";
	else
	  txt = actors.length + " actors";
	addUndoPoint("Adding " + txt + " to '" + node.getFullName() + "'");

	// FIXME: backup and update expanded state

	// add
	children = buildTree(node, actors, true);
	for (Node child: children)
	  uniqueName(child);
	nodeStructureChanged(node);
      }
      else {
	node   = (Node) path.getLastPathComponent();
	parent = (Node) node.getParent();
	index  = node.getParent().getIndex(node);
	if (position == InsertPosition.AFTER)
	  index++;

	// does actor handler allow standalones?
	if (actor instanceof ClipboardActorContainer)
	  actors = ((ClipboardActorContainer) actor).getActors();
	else
	  actors = new AbstractActor[]{actor};
	if (actors.length < 1)
	  return;
	if (!checkForStandalones(actors, node))
	  return;

	if (actors.length == 1)
	  txt = "'" + actors[0].getName() + "'";
	else
	  txt = actors.length + " actors";
	if (position == InsertPosition.AFTER)
	  addUndoPoint("Adding " + txt + " after " + ((Node) parent.getChildAt(index - 1)).getFullName() + "'");
	else
	  addUndoPoint("Adding " + txt + " before " + ((Node) parent.getChildAt(index)).getFullName() + "'");

	// FIXME: backup and update expanded state

	// insert
	children = buildTree(node, actors, false);
	for (Node child: children) {
	  parent.insert(child, index);
	  uniqueName(child);
	  index++;
	}
	nodeStructureChanged(parent);
	setSelectionPath(new TreePath(children[0].getPath()));
      }

      m_Modified = true;

      // notify listeners
      notifyActorChangeListeners(new ActorChangeEvent(m_Self, node, Type.MODIFY));
    }
  }

  /**
   * Brings up the GOE dialog for adding a template.
   *
   * @param path	the path to the actor to add the new template sibling
   * @param template	the template to use as default in dialog, use null to use suggestion
   * @param position	where to insert the template
   */
  protected void addFromTemplate(TreePath path, AbstractActorTemplate template, InsertPosition position) {
    AbstractActor		actor;
    AbstractActorTemplate[] 	templates;

    if (m_TemplateDialog == null) {
      m_TemplateDialog = GenericObjectEditorDialog.createDialog(this);
      m_TemplateDialog.getGOEEditor().setCanChangeClassInDialog(true);
      m_TemplateDialog.getGOEEditor().setClassType(AbstractActorTemplate.class);
    }

    if (template == null) {
      templates = suggestActorTemplates(path, position);
      template  = templates[0];
    }
    else {
      templates = new AbstractActorTemplate[]{template};
    }
    m_TemplateDialog.setProposedClasses(templates);
    m_TemplateDialog.setCurrent(template);
    if (position == InsertPosition.HERE)
      m_TemplateDialog.setTitle("Add from template here...");
    else if (position == InsertPosition.AFTER)
      m_TemplateDialog.setTitle("Add from template after...");
    else if (position == InsertPosition.BENEATH)
      m_TemplateDialog.setTitle("Add from template beneath...");
    m_TemplateDialog.setLocationRelativeTo(this);
    m_TemplateDialog.setVisible(true);
    if (m_TemplateDialog.getResult() != GenericObjectEditorDialog.APPROVE_OPTION)
      return;
    template = (AbstractActorTemplate) m_TemplateDialog.getEditor().getValue();

    try {
      actor                        = template.generate();
      m_LastTemplate               = template;
      m_LastTemplateInsertPosition = position;
    }
    catch (Exception e) {
      actor = null;
      e.printStackTrace();
      GUIHelper.showErrorMessage(this, "Failed to create actor from template: " + e);
    }
    if (actor != null)
      addActor(path, actor, position);
  }

  /**
   * Brings up the GOE dialog for editing the selected actor.
   *
   * @param path	the path to the actor
   */
  protected void editActor(TreePath path) {
    GenericObjectEditorDialog	dialog;
    Node 			currNode;
    Node			newNode;
    Node			parent;
    AbstractActor		actor;
    AbstractActor		actorOld;
    int				index;
    boolean			changed;
    ActorHandler		handler;
    ActorHandler		handlerOld;
    int				i;

    if (path == null)
      return;

    currNode               = (Node) path.getLastPathComponent();
    m_CurrentEditingNode   = currNode;
    m_CurrentEditingParent = (Node) currNode.getParent();
    actorOld = currNode.getActor().shallowCopy();
    dialog   = GenericObjectEditorDialog.createDialog(this);
    if (isEditable())
      dialog.setTitle("Edit...");
    else
      dialog.setTitle("Show...");
    dialog.getGOEEditor().setCanChangeClassInDialog(true);
    dialog.getGOEEditor().setClassType(AbstractActor.class);
    dialog.setProposedClasses(null);
    dialog.setCurrent(currNode.getActor());
    dialog.getGOEEditor().setReadOnly(!isEditable());
    dialog.getGOEEditor().setFilter(configureFilter(path, null));
    dialog.setLocationRelativeTo(null);
    dialog.setVisible(true);
    m_CurrentEditingNode   = null;
    m_CurrentEditingParent = null;
    if (dialog.getResult() == GenericObjectEditorDialog.APPROVE_OPTION) {
      actor = (AbstractActor) dialog.getEditor().getValue();
      // make sure name is not empty
      if (actor.getName().length() == 0)
	actor.setName(actor.getDefaultName());
      if (actor.equals(actorOld)) {
	actorOld.destroy();
	return;
      }
      parent = (Node) currNode.getParent();

      // does parent allow singletons?
      if (!checkForStandalones(actor, parent))
	return;

      addUndoPoint("Updating node '" + currNode.getFullName() + "'");

      // check whether actor class or actor structure (for ActorHandlers) has changed
      changed = (actor.getClass() != actorOld.getClass());
      if (!changed && (actor instanceof ActorHandler)) {
	handler    = (ActorHandler) actor;
	handlerOld = (ActorHandler) actorOld;
	changed    = (handler.size() != handlerOld.size());
	if (!changed) {
	  for (i = 0; i < handler.size(); i++) {
	    if (handler.get(i).getClass() != handlerOld.get(i).getClass()) {
	      changed = true;
	      break;
	    }
	  }
	}
      }

      if (changed) {
	if (parent == null) {
	  if (actor instanceof InstantiatableActor) {
	    buildTree(actor);
	    currNode = (Node) getModel().getRoot();
	  }
	  else {
	    GUIHelper.showErrorMessage(
		m_Self, "Root node must be an instantiatable actor!");
	    return;
	  }
	}
	else {
	  newNode = buildTree(null, actor, false);
	  index   = parent.getIndex(currNode);
	  parent.remove(index);
	  parent.insert(newNode, index);
	  currNode = newNode;
	}
      }
      else {
	currNode.setActor(actor);
      }
      uniqueName(currNode);
      m_Modified = true;
      nodeStructureChanged(currNode);
      notifyActorChangeListeners(new ActorChangeEvent(m_Self, currNode, Type.MODIFY));
      setSelectionPath(new TreePath(currNode.getPath()));
      refreshTabs();
    }
  }

  /**
   * Brings up a flow window for editing the selected external actor's flow.
   *
   * @param path	the path to the node
   */
  protected void editFlow(TreePath path) {
    Node			node;
    FlowEditorDialog 		dialog;
    AbstractExternalActor	actor;

    node = (Node) path.getLastPathComponent();
    if (node == null)
      return;
    actor = (AbstractExternalActor) node.getActor();
    if (actor == null)
      return;

    if (getParentDialog() != null)
      dialog = new FlowEditorDialog(getParentDialog());
    else
      dialog = new FlowEditorDialog(getParentFrame());
    dialog.getFlowEditorPanel().loadUnsafe(actor.getActorFile());
    dialog.setVisible(true);
    if (dialog.getFlowEditorPanel().getCurrentFile() != null) {
      if ((actor.getActorFile() == null) || (!actor.getActorFile().equals(dialog.getFlowEditorPanel().getCurrentFile()))) {
	actor.setActorFile(new FlowFile(dialog.getFlowEditorPanel().getCurrentFile()));
	m_Modified = true;
      }
    }

    // notify listeners
    notifyActorChangeListeners(new ActorChangeEvent(m_Self, node, Type.MODIFY));
  }

  /**
   * Encloses the currently selected actors in the specified actor handler.
   *
   * @param paths	the (paths to the) actors to wrap in the control actor
   * @param handler	the handler to use
   */
  protected void encloseActor(TreePath[] paths, ActorHandler handler) {
    AbstractActor[]	currActor;
    Node		parent;
    Node 		currNode;
    Node		newNode;
    int			index;
    String		msg;
    MutableActorHandler	mutable;
    int			i;

    parent    = null;
    currActor = new AbstractActor[paths.length];
    for (i = 0; i < paths.length; i++) {
      currNode     = (Node) paths[i].getLastPathComponent();
      currActor[i] = currNode.getFullActor().shallowCopy();
      if (parent == null)
	parent = (Node) currNode.getParent();

      if (ActorUtils.isStandalone(currActor[i])) {
	if (!handler.getActorHandlerInfo().canContainStandalones()) {
	  GUIHelper.showErrorMessage(
	      this,
	      "You cannot enclose a standalone actor in a "
	      + handler.getClass().getSimpleName() + "!");
	  return;
	}
      }
    }

    if (paths.length == 1)
      addUndoPoint("Enclosing node '" + ((Node) paths[0].getLastPathComponent()).getActor().getFullName() + "' in " + handler.getClass().getName());
    else
      addUndoPoint("Enclosing " + paths.length + " nodes in " + handler.getClass().getName());

    try {
      if (handler instanceof MutableActorHandler) {
	mutable = (MutableActorHandler) handler;
	mutable.removeAll();
	for (i = 0; i < currActor.length; i++)
	  mutable.add(i, currActor[i]);
      }
      else {
	handler.set(0, currActor[0]);
      }
      newNode = buildTree(null, (AbstractActor) handler, false);
      for (i = 0; i < paths.length; i++) {
	currNode = (Node) paths[i].getLastPathComponent();
	index    = parent.getIndex(currNode);
	parent.remove(index);
	if (i == 0)
	  parent.insert(newNode, index);
      }
      uniqueName(newNode);
      m_Modified = true;
      if (paths.length == 1) {
	nodeStructureChanged(newNode);
	setSelectionPath(new TreePath(newNode.getPath()));
	notifyActorChangeListeners(new ActorChangeEvent(m_Self, newNode, Type.MODIFY));
      }
      else {
	nodeStructureChanged(parent);
	setSelectionPath(new TreePath(parent.getPath()));
	notifyActorChangeListeners(new ActorChangeEvent(m_Self, parent, Type.MODIFY));
      }
    }
    catch (Exception e) {
      if (paths.length == 1)
	msg = "Failed to enclose actor '" + ((Node) paths[0].getLastPathComponent()).getActor().getFullName() + "'";
      else
	msg = "Failed to enclose " + paths.length + " actors";
      msg += " in a " + handler.getClass().getSimpleName() + ": ";
      System.err.println(msg);
      e.printStackTrace();
      GUIHelper.showErrorMessage(
	  this, msg + "\n" + e.getMessage());
    }
  }

  /**
   * Encloses the specified actor in a DisplayPanelManager actor.
   *
   * @param path	the path of the actor to enclose
   */
  protected void encloseInDisplayPanelManager(TreePath path) {
    AbstractActor	currActor;
    Node		currNode;
    DisplayPanelManager	manager;
    AbstractDisplay	display;

    currNode  = (Node) path.getLastPathComponent();
    currActor = currNode.getFullActor().shallowCopy();
    manager   = new DisplayPanelManager();
    manager.setName(currActor.getName());
    manager.setPanelProvider((DisplayPanelProvider) currActor);
    if (currActor instanceof AbstractDisplay) {
      display = (AbstractDisplay) currActor;
      manager.setWidth(display.getWidth() + 100);
      manager.setHeight(display.getHeight());
      manager.setX(display.getX());
      manager.setY(display.getY());
    }

    addUndoPoint("Enclosing node '" + currNode.getActor().getFullName() + "' in " + manager.getClass().getName());

    currNode.setActor(manager);

    setModified(true);
    nodeStructureChanged((Node) currNode.getParent());
    setSelectionPath(new TreePath(currNode.getPath()));
    notifyActorChangeListeners(new ActorChangeEvent(m_Self, currNode, Type.MODIFY));
  }

  /**
   * Turns the selected actor into a global actor.
   *
   * @param path	the (path to the) actor to turn into global actor
   */
  protected void createGlobalActor(TreePath path) {
    AbstractActor	currActor;
    Node 		currNode;
    Node		globalNode;
    Node		root;
    Vector<Node>	global;
    GlobalActors	globalActors;
    Node		moved;
    AbstractGlobalActor	replacement;

    currNode  = (Node) path.getLastPathComponent();
    currActor = currNode.getFullActor().shallowCopy();
    if (ActorUtils.isStandalone(currActor)) {
      GUIHelper.showErrorMessage(
	  this,
	  "Standalone actors cannot be turned into a global actor!");
      return;
    }
    if (currActor instanceof AbstractGlobalActor) {
      GUIHelper.showErrorMessage(
	  this,
	  "Actor points already to a global actor!");
      return;
    }
    if ((currNode.getParent() != null) && (((Node) currNode.getParent()).getActor() instanceof GlobalActors)) {
      GUIHelper.showErrorMessage(
	  this,
	  "Actor is already a global actor!");
      return;
    }

    addUndoPoint("Creating global actor from '" + currNode.getActor().getFullName());

    global = FlowHelper.findGlobalActors(currNode, (Node) currNode.getParent());

    // no GlobalActors available?
    if (global.size() == 0) {
      root = (Node) currNode.getRoot();
      if (!((ActorHandler) root.getActor()).getActorHandlerInfo().canContainStandalones()) {
	GUIHelper.showErrorMessage(
	    this,
	    "Root actor '" + root.getActor().getName() + "' cannot contain standalones!");
	return;
      }
      globalActors = new GlobalActors();
      globalNode   = new Node(this, globalActors);
      root.insert(globalNode, 0);
      uniqueName(globalNode);
    }
    else {
      globalNode = global.lastElement();
    }

    // move actor
    moved = new Node(this, currActor);
    globalNode.add(moved);
    uniqueName(moved);

    // create replacement
    replacement = null;
    if (ActorUtils.isSource(currActor))
      replacement = new GlobalSource();
    else if (ActorUtils.isTransformer(currActor))
      replacement = new GlobalTransformer();
    else if (ActorUtils.isSink(currActor))
      replacement = new GlobalSink();
    replacement.setGlobalName(new GlobalActorReference(moved.getActor().getName()));
    currNode.setActor(replacement);
    uniqueName(currNode);

    // update tree
    setModified(true);
    nodeStructureChanged(globalNode);
    notifyActorChangeListeners(new ActorChangeEvent(m_Self, globalNode, Type.MODIFY));
    nodeStructureChanged((Node) currNode);
    notifyActorChangeListeners(new ActorChangeEvent(m_Self, currNode, Type.MODIFY));
  }

  /**
   * Opens a new FlowEditor window with the currently selected sub-flow.
   * If the selected actor itself is not implementing the InstantiatableActor
   * interface, it gets enclosed in the appropriate instantiatable wrapper
   * actor.
   *
   * @param path	the (path to the) actor to externalize
   */
  protected void externalizeActor(TreePath path) {
    AbstractActor		currActor;
    Node 			currNode;
    AbstractExternalActor	extActor;
    FlowEditorDialog		dialog;

    currNode  = (Node) path.getLastPathComponent();
    currActor = currNode.getFullActor().shallowCopy();
    currActor = ActorUtils.createExternalActor(currActor);
    if (getParentDialog() != null)
      dialog = new FlowEditorDialog(getParentDialog());
    else
      dialog = new FlowEditorDialog(getParentFrame());
    dialog.getFlowEditorPanel().newTab();
    dialog.getFlowEditorPanel().setCurrentFlow(currActor);
    dialog.getFlowEditorPanel().setModified(true);
    dialog.setVisible(true);
    if (dialog.getFlowEditorPanel().getCurrentFile() == null)
      return;

    addUndoPoint("Externalizing node '" + currNode.getFullName() + "'");

    extActor = null;
    if (ActorUtils.isStandalone(currActor))
      extActor = new ExternalStandalone();
    else if (ActorUtils.isSource(currActor))
      extActor = new ExternalSource();
    else if (ActorUtils.isTransformer(currActor))
      extActor = new ExternalTransformer();
    else if (ActorUtils.isSink(currActor))
      extActor = new ExternalSink();
    extActor.setActorFile(new FlowFile(dialog.getFlowEditorPanel().getCurrentFile()));

    setModified(true);
    currNode.setActor(extActor);
    currNode.removeAllChildren();
    nodeStructureChanged(currNode);
    notifyActorChangeListeners(new ActorChangeEvent(m_Self, currNode, Type.MODIFY));
  }

  /**
   * Removes the node (incl. sub-tree).
   *
   * @param path	the path of the node to remove
   */
  protected void removeActor(TreePath path) {
    removeActor(new TreePath[]{path});
  }

  /**
   * Removes the nodes (incl. sub-tree).
   *
   * @param path	the paths of the nodes to remove
   */
  protected void removeActor(TreePath[] paths) {
    Node		node;
    int			index;
    Node		parent;
    List<Boolean>	state;
    int			row;
    Node		selNode;
    Node[]		nodes;
    int			i;

    nodes = new Node[paths.length];
    for (i = 0; i < paths.length; i++)
      nodes[i] = (Node) paths[i].getLastPathComponent();

    if (nodes.length == 1)
      addUndoPoint("Removing node '" + nodes[0].getActor().getFullName() + "'");
    else
      addUndoPoint("Removing nodes");

    // backup expanded state
    state = new Vector<Boolean>(getExpandedStateList());

    selNode = null;
    for (i = nodes.length - 1; i >= 0; i--) {
      node   = nodes[i];
      parent = (Node) node.getParent();
      index  = parent.getIndex(node);
      row    = getRowForPath(paths[i]);

      // remove node
      parent.remove(index);
      nodeStructureChanged(parent);

      // restore expanded state
      state.remove(row);

      // select appropriate node
      if (parent.getChildCount() > index) {
        selNode = (Node) parent.getChildAt(index);
      }
      else {
	if ((parent.getChildCount() > 0) && (parent.getChildCount() > index - 1))
	  selNode = (Node) parent.getChildAt(index - 1);
	else
	  selNode = parent;
      }
    }

    setExpandedStateList(state);

    if (selNode != null)
      setSelectionPath(new TreePath(selNode.getPath()));

    m_Modified = true;

    // notify listeners
    if (nodes.length == 1)
      notifyActorChangeListeners(new ActorChangeEvent(m_Self, nodes[0], Type.REMOVE));
    else
      notifyActorChangeListeners(new ActorChangeEvent(m_Self, nodes, Type.REMOVE_RANGE));
  }

  /**
   * Displays the help for the selected actor.
   *
   * @param path	the path to the actor
   */
  protected void help(TreePath path) {
    HelpDialog		dialog;
    HtmlHelpProducer 	producer;
    Node		node;
    AbstractActor	actor;

    node  = (Node) path.getLastPathComponent();
    actor = node.getActor();

    if (getParentDialog() != null)
      dialog = new HelpDialog(getParentDialog());
    else
      dialog = new HelpDialog(getParentFrame());
    producer = new HtmlHelpProducer();
    producer.produce((OptionHandler) actor);
    dialog.setHelp(producer.getOutput(), true);
    dialog.setTitle("Help on " + actor.getClass().getName());
    dialog.setLocation(
	getTopLevelAncestor().getLocationOnScreen().x + getTopLevelAncestor().getSize().width,
	getTopLevelAncestor().getLocationOnScreen().y);
    dialog.setSize(800, 600);
    dialog.setVisible(true);

  }

  /**
   * Adds the listener to the internal list of listeners.
   *
   * @param l		the listener to add
   */
  public void addActorChangeListener(ActorChangeListener l) {
    m_ActorChangeListeners.add(l);
  }

  /**
   * Removes the listener from the internal list of listeners.
   *
   * @param l		the listener to remove
   */
  public void removeActorChangeListener(ActorChangeListener l) {
    m_ActorChangeListeners.remove(l);
  }

  /**
   * Notifies all listeners.
   *
   * @param e		the event to send
   */
  public void notifyActorChangeListeners(ActorChangeEvent e) {
    Iterator<ActorChangeListener>	iter;

    iter = m_ActorChangeListeners.iterator();
    while (iter.hasNext())
      iter.next().actorChanged(e);
  }

  /**
   * Sets the flow actor to display.
   *
   * @param value	the flow actor
   */
  public void setActor(AbstractActor value) {
    buildTree(value);
  }

  /**
   * Returns the underlying flow.
   *
   * @return		the flow or null if none stored yet
   */
  public AbstractActor getActor() {
    AbstractActor	result;

    result = null;

    if (getModel().getRoot() != null)
      result = ((Node) getModel().getRoot()).getFullActor();

    return result;
  }

  /**
   * Returns whether the actor is of type Flow or not.
   *
   * @return		true if actor is a Flow
   */
  public boolean isFlow() {
    if (getModel().getRoot() == null)
      return false;
    else
      return (((Node) getModel().getRoot()).getActor() instanceof Flow);
  }

  /**
   * Sets whether the tree is modified or not.
   *
   * @param value	true if tree is modified
   */
  public void setModified(boolean value) {
    m_Modified = value;
  }

  /**
   * Returns whether the tree is modified.
   *
   * @return		true if modified
   */
  public boolean isModified() {
    return m_Modified;
  }

  /**
   * Sets the file this flow is associated with.
   *
   * @param value	the associated file, null if not associated with a file
   */
  public void setFile(File value) {
    m_File = value;
  }

  /**
   * Returns the file this flow is associated with.
   *
   * @return		the file, null if not associated with file
   */
  public File getFile() {
    return m_File;
  }

  /**
   * Returns the currently selected node.
   *
   * @return		the selected node, null if none selected
   */
  public Node getSelectedNode() {
    Node	result;

    result = null;

    if (getSelectionPath() != null)
      result = (Node) getSelectionPath().getLastPathComponent();

    return result;
  }

  /**
   * Returns the currently selected actor.
   *
   * @return		the selected actor, null if none selected
   */
  public AbstractActor getSelectedActor() {
    AbstractActor	result;
    Node		node;

    result = null;
    node   = getSelectedNode();

    if (node != null)
      result = node.getActor();

    return result;
  }

  /**
   * Returns the currently selected actors.
   *
   * @return		the selected actors, 0-length array if none selected
   */
  public AbstractActor[] getSelectedActors() {
    AbstractActor[]	result;
    int			i;

    if (getSelectionPaths() != null) {
      result = new AbstractActor[getSelectionPaths().length];
      for (i = 0; i < result.length; i++)
	result[i] = ((Node) getSelectionPath().getLastPathComponent()).getActor();
    }
    else {
      result = new AbstractActor[0];
    }

    return result;
  }

  /**
   * Returns the full name of the currently selected actor.
   *
   * @return		the full name of the selected actor, null if none selected
   */
  public String getSelectedFullName() {
    String	result;
    Node	node;

    result = null;
    node   = getSelectedNode();
    if (node != null)
      result = node.getFullName();

    return result;
  }

  /**
   * Tries to locate the node specified by the path parts.
   *
   * @param parent	the parent to start with
   * @param path	the path elements to traverse (below the parent)
   * @return		the located node or null if none found
   */
  protected Node locate(Node parent, ActorPath path) {
    Node		result;
    Node		child;
    int			index;
    int			i;

    result = null;

    index = -1;
    for (i = 0; i < parent.getChildCount(); i++) {
      child = (Node) parent.getChildAt(i);
      if (child.getActor().getName().equals(path.getFirstPathComponent())) {
	index = i;
	break;
      }
    }
    if (index != -1) {
      child = (Node) parent.getChildAt(index);
      if (path.getPathCount() == 1)
	result = child;
      else
	result = locate(child, path.getChildPath());
    }
    else {
      System.err.println("Malformed path?");
    }

    return result;
  }

  /**
   * Locates the node in the tree based on the specified path.
   *
   * @param path	the path of the node to locate
   * @return		the located node or null if none found
   */
  public Node locate(String path) {
    Node	result;
    ActorPath	actorPath;
    Node	root;

    result = null;

    actorPath = new ActorPath(path);
    root      = (Node) getModel().getRoot();
    if (actorPath.getFirstPathComponent().equals(root.getActor().getName())) {
      if (actorPath.getPathCount() == 1)
	result = root;
      else
	result = locate(root, actorPath.getChildPath());
    }

    return result;
  }

  /**
   * Locates and selectes the node in the tree based on the specified path.
   *
   * @param path	the path of the node to locate
   */
  public void locateAndDisplay(String path) {
    Node	node;
    TreePath	tpath;

    node = locate(path);

    if (node != null) {
      tpath = getPath(node);
      setSelectionPath(tpath);
      scrollPathToVisible(tpath);
    }
  }

  /**
   * Searches for a node which is matches the "search" string. The search
   * string can be interpreted as regular expression as well.
   * A starting point (i.e., subtree) in the tree can be given as well.
   *
   * @param subtree	the starting point (= subtree) of the search, uses
   * 			the root node if null
   * @param last	the node returned from the last search, if not null
   * 			then a "find next" search is performed
   * @param search	the search string
   * @param isRegExp	whether the search string is a regular expression
   * @return		the path of the first matching node, otherwise null
   * 			if no match found
   */
  public Node find(Node subtree, Node last, String search, boolean isRegExp) {
    Node	result;
    Node	current;
    Enumeration	enm;

    result = null;

    if (!isRegExp)
      search = search.toLowerCase();

    // search whole tree?
    if (subtree == null)
      subtree = (Node) getModel().getRoot();

    enm = subtree.preorderEnumeration();

    // start from a specific node? -> skip nodes before this one
    if (last != null) {
      while (enm.hasMoreElements()) {
	current = (Node) enm.nextElement();
	if (current == last)
	  break;
      }
    }

    // perform search
    while (enm.hasMoreElements()) {
      current = (Node) enm.nextElement();
      if (isRegExp) {
	if (current.getActor().getName().toLowerCase().matches(search)) {
	  result = current;
	  break;
	}
      }
      else {
	if (current.getActor().getName().toLowerCase().indexOf(search) > -1) {
	  result = current;
	  break;
	}
      }
    }

    return result;
  }

  /**
   * Tries to find the node in the tree and returns the path to it.
   *
   * @param node	the node to look for
   * @return		the path if the node was found, null otherwise
   */
  public TreePath getPath(Node node) {
    return new TreePath(((DefaultTreeModel) getModel()).getPathToRoot(node));
  }

  /**
   * Searches for actor names in the tree.
   */
  public void find() {
    String	search;
    TreePath	path;
    Node	node;

    path = getSelectionPath();
    if (path != null)
      node = (Node) path.getLastPathComponent();
    else
      node = null;

    search = JOptionPane.showInputDialog(
	"Please enter the search string ("
	+ ((node == null) ? ("whole flow") : ("below '" + node.getActor().getName()) + "'") + "):",
	m_LastSearchString);
    if (search == null)
      return;

    m_LastSearchString = search;
    m_LastSearchNode   = find(node, null, m_LastSearchString, false);
    if (m_LastSearchNode == null) {
      GUIHelper.showErrorMessage(
	  m_Self, "Search string '" + m_LastSearchString + "' not found!");
    }
    else {
      path = getPath(m_LastSearchNode);
      setSelectionPath(path);
      scrollPathToVisible(path);
    }
  }

  /**
   * Searches for the next actor in the tree.
   */
  public void findNext() {
    TreePath	path;

    m_LastSearchNode = find(null, m_LastSearchNode, m_LastSearchString, false);
    if (m_LastSearchNode == null) {
      GUIHelper.showErrorMessage(
	  m_Self, "Search string '" + m_LastSearchString + "' not found!");
    }
    else {
      path = getPath(m_LastSearchNode);
      setSelectionPath(path);
      scrollPathToVisible(path);
    }
  }

  /**
   * Returns the last search string in use.
   *
   * @return		the search string
   */
  public String getLastSearchString() {
    return m_LastSearchString;
  }

  /**
   * Returns the node that was found in the last search.
   *
   * @return		the node, can be null if no search performed yet or
   * 			last search unsuccessful
   */
  public Node getLastSearchNode() {
    return m_LastSearchNode;
  }

  /**
   * Sets whether to show the quick info or not.
   *
   * @param value	if true then the quick info will be displayed
   */
  public void setShowQuickInfo(boolean value) {
    m_ShowQuickInfo = value;
    nodeStructureChanged((Node) getModel().getRoot());
  }

  /**
   * Returns whether the quick info is shown or not.
   *
   * @return		true if the quick info is shown
   */
  public boolean getShowQuickInfo() {
    return m_ShowQuickInfo;
  }

  /**
   * Sets whether to show the annotations or not.
   *
   * @param value	if true then the annotations will be displayed
   */
  public void setShowAnnotations(boolean value) {
    m_ShowAnnotations = value;
    nodeStructureChanged((Node) getModel().getRoot());
  }

  /**
   * Returns whether the annotations are shown or not.
   *
   * @return		true if the annotations are shown
   */
  public boolean getShowAnnotations() {
    return m_ShowAnnotations;
  }

  /**
   * Sets whether to show the input/output information or not.
   *
   * @param value	if true then the input/output information will be displayed
   */
  public void setShowInputOutput(boolean value) {
    m_ShowInputOutput = value;
    nodeStructureChanged((Node) getModel().getRoot());
  }

  /**
   * Returns whether the input/output information is shown or not.
   *
   * @return		true if the input/output information is shown
   */
  public boolean getShowInputOutput() {
    return m_ShowInputOutput;
  }

  /**
   * Sets the class name prefixes to remove from the input/output info.
   *
   * @param value	the prefixes
   */
  public void setInputOutputPrefixes(String[] value) {
    m_InputOutputPrefixes = value.clone();
  }

  /**
   * Returns the class name prefixes to remove from the input/output info.
   *
   * @return		the prefixes
   */
  public String[] getInputOutputPrefixes() {
    return m_InputOutputPrefixes;
  }

  /**
   * Sets the scale factor for the icons.
   *
   * @param value	the scale factor (1.0 is actual size)
   */
  public void setIconScaleFactor(double value) {
    if (value != ((Renderer) getCellRenderer()).getScaleFactor())
      setCellRenderer(new Renderer(value));
  }

  /**
   * Returns the scale factor for the icons.
   *
   * @return		the scale factor (1.0 is actual size)
   */
  public double getIconScaleFactor() {
    return ((Renderer) getCellRenderer()).getScaleFactor();
  }

  /**
   * Rebuilds the subtree, including the specified node, due to changes in the
   * actor of this node.
   *
   * @param node	the node which actor changed
   */
  public void actorChanged(Node node) {
    Node	newNode;
    Node	parent;
    int		index;

    newNode = buildTree(null, node.getActor(), false);
    parent  = (Node) node.getParent();

    // already at root node?
    if (parent == null) {
      ((DefaultTreeModel) getModel()).setRoot(newNode);
    }
    else {
      index = parent.getIndex(node);
      if (index > -1) {
	parent.insert(newNode, index);
	parent.remove(index + 1);
      }
      else {
	System.err.println("Couldn't find node in parent's children??");
      }
    }

    // notify about change
    nodeStructureChanged(newNode);
    notifyActorChangeListeners(new ActorChangeEvent(this, node, Type.MODIFY));
  }

  /**
   * Sets the current state of the tree (tree, expansions, modified).
   *
   * @param value	the state to use
   */
  public void setState(Vector value) {
    AbstractActor	actor;
    boolean[]		expanded;
    Boolean		modified;
    NestedConsumer	consumer;
    File		file;

    if (m_StateUsesNested) {
      if (value.get(0) != null) {
	consumer = new NestedConsumer();
	consumer.setInput((Vector) value.get(0));
	actor = (AbstractActor) consumer.consume();
	consumer.cleanUp();
      }
      else {
	actor = null;
      }
    }
    else {
      actor = (AbstractActor) value.get(0);
    }
    expanded = (boolean[]) value.get(1);
    modified = (Boolean) value.get(2);
    file     = (File) value.get(3);

    setActor(actor);
    setExpandedState(expanded);
    setModified(modified);
    setFile(file);
  }

  /**
   * Returns the current state of the tree (tree, expansions, modified).
   * Is used for undo/redo.
   *
   * @return		the current state
   */
  public Vector getState() {
    Vector		result;
    AbstractActor	actor;
    NestedProducer	producer;

    result = new Vector();

    actor = getActor();
    if (m_StateUsesNested) {
      if (actor == null) {
	result.add(null);
      }
      else {
	producer = new NestedProducer();
	producer.produce(actor);
	result.add(producer.getOutput());
	actor.destroy();
	producer.cleanUp();
      }
    }
    else {
      result.add(actor);
    }
    result.add(getExpandedState());
    result.add(isModified());
    result.add(getFile());

    return result;
  }

  /**
   * Returns the string for the specified action.
   *
   * @param action	the action to get the string for
   * @return		the caption
   */
  protected String getDropMenuActionCaption(DropMenu action) {
    if (action == DropMenu.ADD)
      return "Add actor";
    else if (action == DropMenu.MOVE)
      return "Move actor";
    else
      return super.getDropMenuActionCaption(action);
  }

  /**
   * Returns the icon for the drop action.
   *
   * @param action	the action to get the icon for
   * @return		the icon or null if none available
   */
  protected ImageIcon getDropMenuActionIcon(DropMenu action) {
    if (action == DropMenu.ADD)
      return GUIHelper.getIcon("flow.gif");
    else if (action == DropMenu.CANCEL)
      return GUIHelper.getIcon("delete.gif");
    else
      return super.getDropMenuActionIcon(action);
  }

  /**
   * Returns whether dragging is enabled.
   *
   * @return		always true
   */
  protected boolean isDragEnabled() {
    return true;
  }

  /**
   * Returns whether dropping is enabled.
   *
   * @return		always true
   */
  protected boolean isDropEnabled() {
    return true;
  }

  /**
   * Checks whether the source node can be dragged at all.
   * <p/>
   * The parent of the source node must a MutableActorHandler
   *
   * @param source	the source node that is about to be dragged
   * @return		true if the source node can be dragged
   */
  protected boolean canStartDrag(BaseTreeNode source) {
    return    (source.getParent() != null)
           && (((Node) source.getParent()).getActor() instanceof MutableActorHandler);
  }

  /**
   * Checks whether the source can be dropped here.
   *
   * @param source	the source node
   * @param target	the target node
   * @param position	where to drop the data
   * @return		true if can be dropped
   */
  protected boolean canDrop(Transferable source, TreeNode target, DropPosition position) {
    boolean	result;
    Node	parent;

    result = super.canDrop(source, target, position);

    if (result) {
      parent = (Node) target.getParent();
      if (position == DropPosition.BENEATH)
	result = (((Node) target).getActor() instanceof MutableActorHandler);
      else if (position == DropPosition.AFTER)
	result = (parent != null) && (parent.getActor() instanceof MutableActorHandler);
      else if (position == DropPosition.HERE)
	result = (parent != null) && (parent.getActor() instanceof MutableActorHandler);
    }

    return result;
  }

  /**
   * Highlights all the nodes that have a variable that match the regular
   * expression on the variable name.
   *
   * @param parent	the parent to start the search in
   * @param nameRegExp	the regular expression to match variable names against
   */
  public void highlightVariables(Node parent, String nameRegExp) {
    Node	child;
    int		i;

    parent.findVariable(nameRegExp);

    for (i = 0; i < parent.getChildCount(); i++) {
      child = (Node) parent.getChildAt(i);
      child.findVariable(nameRegExp);
      highlightVariables(child, nameRegExp);
    }
  }

  /**
   * Highlights all the nodes that have a variable that match the regular
   * expression on the variable name.
   *
   * @param nameRegExp	the regular expression to match variable names against
   */
  public void highlightVariables(String nameRegExp) {
    highlightVariables((Node) getModel().getRoot(), nameRegExp);
    treeDidChange();
  }

  /**
   * Enables/diables all breakpoint actors.
   *
   * @param parent	the parent node to start recursion from
   * @param enable	if true all breakpoint actors get enabled, otherwise disabled
   */
  protected void enableBreakpoints(Node parent, boolean enable) {
    int		i;

    if (parent.getActor() instanceof Breakpoint)
      parent.getActor().setSkip(!enable);

    for (i = 0; i < parent.getChildCount(); i++)
      enableBreakpoints((Node) parent.getChildAt(i), enable);
  }

  /**
   * Enables/diables all breakpoint actors.
   *
   * @param enable	if true all breakpoint actors get enabled, otherwise disabled
   */
  public void enableBreakpoints(boolean enable) {
    enableBreakpoints((Node) getModel().getRoot(), enable);
    treeDidChange();
  }

  /**
   * Sets the tree editable or read-only. Also, forces the tree to redraw.
   *
   * @param value	if false the tree will be read-only
   */
  public void setEditable(boolean value) {
    super.setEditable(value);
    if (getParent() != null) {
      getParent().invalidate();
      getParent().repaint();
    }
  }

  /**
   * Returns the node currently being edited.
   *
   * @return		the node, null if none being edited
   */
  public Node getCurrentEditingNode() {
    return m_CurrentEditingNode;
  }

  /**
   * Returns the parent of the node currently being edited or being added.
   *
   * @return		the node, null if none being edited/added
   */
  public Node getCurrentEditingParent() {
    return m_CurrentEditingParent;
  }

  /**
   * Creates a new collection for transfer.
   *
   * @param nodes	the nodes to package
   * @return		the new collection
   */
  protected DragAndDropTreeNodeCollection newNodeCollection(BaseTreeNode[] nodes) {
    Node[]	nnodes;
    int		i;

    nnodes = new Node[nodes.length];
    for (i = 0; i < nodes.length; i++)
      nnodes[i] = (Node) nodes[i];

    return new TreeNodeCollection(nnodes);
  }

  /**
   * Creates a new TreeNode for this tree.
   *
   * @param data	the data to use
   * @return		the new nodes
   */
  protected BaseTreeNode[] newTreeNodes(Transferable data) {
    TreeNodeCollection	coll;

    coll = TreeNodeCollection.fromTransferable(this, data);

    if (coll == null)
      return new BaseTreeNode[0];
    else
      return coll.toArray(new Node[coll.size()]);
  }

  /**
   * Refreshes the tabs.
   */
  protected void refreshTabs() {
    if (getEditor() != null)
      getEditor().refreshTabs();
  }
}