/*
 * XYSequencePanel.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.sequence;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.GridLayout;
import java.awt.Point;
import java.awt.Dialog.ModalityType;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.io.File;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.swing.JColorChooser;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;

import adams.core.Properties;
import adams.core.option.OptionUtils;
import adams.data.sequence.XYSequence;
import adams.data.sequence.XYSequencePoint;
import adams.db.AbstractDatabaseConnection;
import adams.db.DatabaseConnection;
import adams.gui.chooser.BaseDirectoryChooser;
import adams.gui.chooser.BaseFileChooser;
import adams.gui.core.ParameterPanel;
import adams.gui.core.Undo;
import adams.gui.dialog.AbstractApprovalDialog;
import adams.gui.event.PaintListener;
import adams.gui.scripting.AbstractScriptingEngine;
import adams.gui.scripting.ScriptingEngine;
import adams.gui.visualization.container.ContainerListPopupMenuSupplier;
import adams.gui.visualization.container.ContainerManager;
import adams.gui.visualization.container.ContainerTable;
import adams.gui.visualization.container.DataContainerPanelWithSidePanel;
import adams.gui.visualization.core.AbstractColorProvider;
import adams.gui.visualization.core.AbstractPaintlet;
import adams.gui.visualization.core.AntiAliasingPaintlet;
import adams.gui.visualization.core.CoordinatesPaintlet;
import adams.gui.visualization.core.CoordinatesPaintlet.Coordinates;
import adams.gui.visualization.core.DefaultColorProvider;
import adams.gui.visualization.core.PlotPanel;
import adams.gui.visualization.core.PopupMenuCustomizer;
import adams.gui.visualization.core.plot.Axis;
import adams.gui.visualization.core.plot.TipTextCustomizer;

/**
 * A panel for displaying XY sequences.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4305 $
 */
public class XYSequencePanel
  extends DataContainerPanelWithSidePanel<XYSequence, XYSequenceContainerManager>
  implements PaintListener, ContainerListPopupMenuSupplier<XYSequenceContainerManager,XYSequenceContainer>,
             PopupMenuCustomizer, TipTextCustomizer {

  /** for serialization. */
  private static final long serialVersionUID = 7985845939008731534L;

  /** the sequence ID list. */
  protected XYSequenceContainerList m_SequenceContainerList;

  /** paintlet for drawing the total ion count graph. */
  protected AbstractXYSequencePaintlet m_XYSequencePaintlet;

  /** paintlet for drawing the X-axis. */
  protected CoordinatesPaintlet m_CoordinatesPaintlet;

  /** the undo manager. */
  protected Undo m_Undo;

  /** whether to adjust to visible data or not. */
  protected boolean m_AdjustToVisibleData;

  /** whether user can resize dialog. */
  protected boolean m_AllowResize;

  /**
   * Initializes the panel with no title.
   */
  public XYSequencePanel() {
    super();
  }

  /**
   * Initializes the panel with the specified title.
   *
   * @param title	the title to use
   */
  public XYSequencePanel(String title) {
    super(title);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    m_Undo                = null;
    m_AdjustToVisibleData = true;
    m_AllowResize         = false;

    super.initialize();
  }

  /**
   * Returns the default database connection.
   *
   * @return		the default database connection
   */
  protected AbstractDatabaseConnection getDefaultDatabaseConnection() {
    return DatabaseConnection.getSingleton();
  }

  /**
   * Returns the container manager to use.
   *
   * @return		the container manager
   */
  protected XYSequenceContainerManager newContainerManager() {
    return new XYSequenceContainerManager(this);
  }

  /**
   * Initializes the GUI.
   */
  protected void initGUI() {
    Properties	props;

    super.initGUI();

    props = getProperties();

    setAdjustToVisibleData(props.getBoolean("Plot.AdjustToVisibleData", false));

    m_SequenceContainerList = new XYSequenceContainerList();
    m_SequenceContainerList.setManager(getContainerManager());
    m_SequenceContainerList.setPopupMenuSupplier(this);

    m_SidePanel.setLayout(new GridLayout(1, 1));
    m_SidePanel.add(m_SequenceContainerList);

    getPlot().setPopupMenuCustomizer(this);
    getPlot().setTipTextCustomizer(this);

    // paintlets
    m_XYSequencePaintlet = new XYSequenceStickPaintlet();
    if (m_XYSequencePaintlet instanceof AntiAliasingPaintlet)
      ((AntiAliasingPaintlet) m_XYSequencePaintlet).setAntiAliasingEnabled(props.getBoolean("Plot.AntiAliasing", true));
    m_XYSequencePaintlet.setPanel(this);
    setPaintlet(
	(AbstractXYSequencePaintlet) AbstractPaintlet.forCommandLine(
	    props.getString("Plot.Paintlet", new XYSequenceStickPaintlet().toCommandLine())));

    m_CoordinatesPaintlet = new CoordinatesPaintlet();
    m_CoordinatesPaintlet.setYInvisible(true);
    m_CoordinatesPaintlet.setPanel(this);
    m_CoordinatesPaintlet.setXColor(props.getColor("Plot.CoordinatesColor." + Coordinates.X, Color.DARK_GRAY));
    m_CoordinatesPaintlet.setYColor(props.getColor("Plot.CoordinatesColor." + Coordinates.Y, Color.DARK_GRAY));

    try {
      getContainerManager().setColorProvider(
	  (AbstractColorProvider) OptionUtils.forAnyCommandLine(
	      AbstractColorProvider.class,
	      props.getProperty("Plot.ColorProvider", "adams.gui.visualization.core.DefaultColorProvider")));
    }
    catch (Exception e) {
      System.err.println(getClass().getName() + " - Failed to set the color provider:");
      getContainerManager().setColorProvider(new DefaultColorProvider());
    }
  }

  /**
   * Returns the current container manager.
   *
   * @return		the manager
   */
  public ContainerManager getSequenceManager() {
    return m_Manager;
  }

  /**
   * Sets the paintlet to use.
   *
   * @param value	the paintlet
   */
  public void setPaintlet(AbstractXYSequencePaintlet value) {
    Properties	props;

    removePaintlet(m_XYSequencePaintlet);
    m_XYSequencePaintlet = value;
    if (m_XYSequencePaintlet instanceof AntiAliasingPaintlet) {
      props = getProperties();
      ((AntiAliasingPaintlet) m_XYSequencePaintlet).setAntiAliasingEnabled(props.getBoolean("Plot.AntiAliasing", true));
    }
    m_XYSequencePaintlet.setPanel(this);
    update();
  }

  /**
   * Returns the paintlet in use.
   *
   * @return		the paintlet
   */
  public AbstractXYSequencePaintlet getPaintlet() {
    return m_XYSequencePaintlet;
  }

  /**
   * Sets the undo manager to use, can be null if no undo-support wanted.
   *
   * @param value	the undo manager to use
   */
  public void setUndo(Undo value) {
    m_Undo = value;
  }

  /**
   * Returns the current undo manager, can be null.
   *
   * @return		the undo manager, if any
   */
  public Undo getUndo() {
    return m_Undo;
  }

  /**
   * Returns whether an Undo manager is currently available.
   *
   * @return		true if an undo manager is set
   */
  public boolean isUndoSupported() {
    return (m_Undo != null);
  }

  /**
   * Returns true if the paintlets can be executed.
   *
   * @param g		the graphics context
   * @return		true if painting can go ahead
   */
  protected boolean canPaint(Graphics g) {
    return ((getPlot() != null) && (m_Manager != null));
  }

  /**
   * Sets whether the display is adjusted to only the visible data or
   * everything currently loaded.
   *
   * @param value	if true then plot is adjusted to visible data
   */
  public void setAdjustToVisibleData(boolean value) {
    m_AdjustToVisibleData = value;
    update();
  }

  /**
   * Returns whether the display is adjusted to only the visible spectrums
   * or all of them.
   *
   * @return		true if the plot is adjusted to only the visible data
   */
  public boolean getAdjustToVisibleData() {
    return m_AdjustToVisibleData;
  }

  /**
   * Updates the axes with the min/max of the new data.
   */
  public void prepareUpdate() {
    List<XYSequencePoint>	points;
    Iterator<XYSequencePoint>	iter;
    XYSequencePoint		point;
    double 			minX;
    double 			maxX;
    double 			minY;
    double 			maxY;
    int				i;
    boolean			determineYRange;
    boolean			determineXRange;

    determineYRange = !(m_XYSequencePaintlet instanceof PaintletWithFixedYRange);
    determineXRange = !(m_XYSequencePaintlet instanceof PaintletWithFixedXRange);

    if (determineXRange) {
      minX = Double.MAX_VALUE;
      maxX = -Double.MAX_VALUE;
    }
    else {
      minX = ((PaintletWithFixedXRange) m_XYSequencePaintlet).getMinX();
      maxX = ((PaintletWithFixedXRange) m_XYSequencePaintlet).getMaxX();
    }
    if (determineYRange) {
      minY = Double.MAX_VALUE;
      maxY = -Double.MAX_VALUE;
    }
    else {
      minY = ((PaintletWithFixedYRange) m_XYSequencePaintlet).getMinY();
      maxY = ((PaintletWithFixedYRange) m_XYSequencePaintlet).getMaxY();
    }

    if (determineXRange || determineYRange) {
      for (i = 0; i < getContainerManager().count(); i++) {
	if (m_AdjustToVisibleData) {
	  if (!getContainerManager().isVisible(i))
	    continue;
	}

	points = getContainerManager().get(i).getData().toList();

	if (points.size() == 0)
	  continue;

	// determine min/max
	if (determineXRange) {
	  if (XYSequencePoint.toDouble(((XYSequencePoint) points.get(0)).getX()) < minX)
	    minX = XYSequencePoint.toDouble(((XYSequencePoint) points.get(0)).getX());
	  if (XYSequencePoint.toDouble(((XYSequencePoint) points.get(points.size() - 1)).getX()) > maxX)
	    maxX = XYSequencePoint.toDouble(((XYSequencePoint) points.get(points.size() - 1)).getX());
	}

	if (determineYRange) {
	  iter = points.iterator();
	  while (iter.hasNext()) {
	    point = (XYSequencePoint) iter.next();
	    if (XYSequencePoint.toDouble(point.getY()) > maxY)
	      maxY = XYSequencePoint.toDouble(point.getY());
	    if (XYSequencePoint.toDouble(point.getY()) < minY)
	      minY = XYSequencePoint.toDouble(point.getY());
	  }
	}
      }
    }

    // center, if only 1 data point
    if (minX == maxX) {
      minX -= 1;
      maxX += 1;
    }

    // update axes
    getPlot().getAxis(Axis.LEFT).setMinimum(minY);
    getPlot().getAxis(Axis.LEFT).setMaximum(maxY);
    getPlot().getAxis(Axis.BOTTOM).setMinimum(minX);
    getPlot().getAxis(Axis.BOTTOM).setMaximum(maxX);
  }

  /**
   * Optional customizing of the menu that is about to be popped up.
   *
   * @param e		the mous event
   * @param menu	the menu to customize
   */
  public void customizePopupMenu(MouseEvent e, JPopupMenu menu) {
    JMenuItem	item;

    if (m_XYSequencePaintlet instanceof XYSequenceLinePaintlet) {
      final XYSequenceLinePaintlet paintlet = (XYSequenceLinePaintlet) m_XYSequencePaintlet;
      item = new JMenuItem();
      if (!paintlet.isMarkersDisabled())
	item.setText("Disable markers");
      else
	item.setText("Enable markers");
      item.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  paintlet.setMarkersDisabled(
	      !paintlet.isMarkersDisabled());
	  repaint();
	}
      });
      menu.add(item);
    }

    if (getAllowResize()) {
      item = new JMenuItem("Resize...");
      item.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          showResizeDialog();
        }
      });
      menu.add(item);
    }

    menu.addSeparator();

    item = new JMenuItem();
    if (isSidePanelVisible())
      item.setText("Hide side panel");
    else
      item.setText("Show side panel");
    item.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	setSidePanelVisible(!isSidePanelVisible());
      }
    });
    menu.add(item);

    item = new JMenuItem();
    if (m_AdjustToVisibleData)
      item.setText("Adjust to loaded data");
    else
      item.setText("Adjust to visible data");
    item.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	m_AdjustToVisibleData = !m_AdjustToVisibleData;
	update();
      }
    });
    menu.add(item);

    menu.addSeparator();

    item = new JMenuItem("Save visible sequences...");
    item.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	BaseDirectoryChooser chooser = new BaseDirectoryChooser();
	int retval = chooser.showSaveDialog(XYSequencePanel.this);
	if (retval == BaseFileChooser.APPROVE_OPTION) {
	  String prefix = chooser.getSelectedFile().getAbsolutePath();
	  for (int i = 0; i < getContainerManager().count(); i++) {
	    if (getContainerManager().isVisible(i)) {
	      XYSequence s = getContainerManager().get(i).getData();
	      s.write(prefix + File.separator + s.getID() + XYSequence.FILE_EXTENSION);
	    }
	  }
	}
      }
    });
    menu.add(item);
  }

  /**
   * Returns a popup menu for the table of the spectrum list.
   *
   * @param table	the affected table
   * @param row	the row the mouse is currently over
   * @return		the popup menu
   */
  public JPopupMenu getContainerListPopupMenu(ContainerTable table, int row) {
    JPopupMenu		result;
    JMenuItem		item;

    result = new JPopupMenu();
    final ContainerTable tableF = table;
    final int rowF = row;

    item = new JMenuItem("Toggle visibility");
    item.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	int[] indices = tableF.getSelectedRows();
	if (indices.length == 0)
	  indices = new int[]{rowF};
	for (int i = 0; i < indices.length; i++) {
	  XYSequenceContainer c = getContainerManager().get(indices[i]);
	  c.setVisible(!c.isVisible());
	}
      }
    });
    result.add(item);

    item = new JMenuItem("Choose color...");
    item.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	XYSequenceContainer cont = getContainerManager().get(rowF);
	Color c = JColorChooser.showDialog(
	    tableF,
	    "Choose color for " + cont.getData().getID(),
	    cont.getColor());
	if (c != null)
	  cont.setColor(c);
      }
    });
    result.add(item);

    result.addSeparator();

    item = new JMenuItem("Remove");
    item.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	int[] indices = tableF.getSelectedRows();
	if (indices.length == 0)
	  indices = new int[]{rowF + 1};
	m_SequenceContainerList.getTable().removeContainers(indices);
      }
    });
    result.add(item);

    item = new JMenuItem("Remove all");
    item.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	m_SequenceContainerList.getTable().removeAllContainers();
      }
    });
    result.add(item);

    return result;
  }

  /**
   * Processes the given tip text. Among the current mouse position, the
   * panel that initiated the call are also provided.
   *
   * @param panel	the content panel that initiated this call
   * @param mouse	the mouse position
   * @param tiptext	the tiptext so far
   * @return		the processed tiptext
   */
  public String processTipText(PlotPanel panel, Point mouse, String tiptext) {
    String			result;
    MouseEvent			event;
    String			hit;

    result = tiptext;
    event  = new MouseEvent(
			getPlot().getContent(),
			MouseEvent.MOUSE_MOVED,
			new Date().getTime(),
			0,
			(int) mouse.getX(),
			(int) mouse.getY(),
			0,
			false);

    hit = (String) m_XYSequencePaintlet.getHitDetector().detect(event);
    if (hit != null)
      result += hit;

    return result;
  }

  /**
   * Returns the container list.
   *
   * @return	the container list
   */
  public XYSequenceContainerList getContainerList() {
    return m_SequenceContainerList;
  }

  /**
   * Returns the current scripting engine, can be null.
   *
   * @return		the current engine
   */
  public AbstractScriptingEngine getScriptingEngine() {
    return ScriptingEngine.getSingleton(getDatabaseConnection());
  }

  /**
   * Sets whether the user can resize the plot (actually the parent dialog/frame)
   * via the popup menu.
   *
   * @param value	if true then user can resize the plot
   */
  public void setAllowResize(boolean value) {
    m_AllowResize = value;
  }

  /**
   * Returns whether the user can reize the plot (actually the parent dialog/frame)
   * via the popup menu.
   *
   * @return		true if the user can resize the plot
   */
  public boolean getAllowResize() {
    return m_AllowResize;
  }

  /**
   * Shows a dialog that allows the user to resize the plot.
   */
  protected void showResizeDialog() {
    AbstractApprovalDialog	dialog;
    ParameterPanel		paramPanel;
    JSpinner			spinnerWidth;
    JSpinner			spinnerHeight;

    if (getParentDialog() != null)
      dialog = AbstractApprovalDialog.getDialog(getParentDialog());
    else
      dialog = AbstractApprovalDialog.getDialog(getParentFrame());
    dialog.setModalityType(ModalityType.DOCUMENT_MODAL);
    dialog.setTitle("Resizing plot");

    paramPanel = new ParameterPanel();
    dialog.getContentPane().add(paramPanel, BorderLayout.CENTER);

    spinnerWidth = new JSpinner();
    ((SpinnerNumberModel) spinnerWidth.getModel()).setMinimum(1);
    ((SpinnerNumberModel) spinnerWidth.getModel()).setStepSize(10);
    ((SpinnerNumberModel) spinnerWidth.getModel()).setValue(getPlot().getContent().getWidth());
    paramPanel.addParameter("_Width", spinnerWidth);

    spinnerHeight = new JSpinner();
    ((SpinnerNumberModel) spinnerHeight.getModel()).setMinimum(1);
    ((SpinnerNumberModel) spinnerHeight.getModel()).setStepSize(10);
    ((SpinnerNumberModel) spinnerHeight.getModel()).setValue(getPlot().getContent().getHeight());
    paramPanel.addParameter("_Height", spinnerHeight);

    dialog.pack();
    dialog.setLocationRelativeTo(this);
    dialog.setVisible(true);
    if (dialog.getOption() == AbstractApprovalDialog.APPROVE_OPTION) {
      resizePlot(
	  new Dimension(
	      ((Number) spinnerWidth.getValue()).intValue(),
	      ((Number) spinnerHeight.getValue()).intValue()));
    }
  }

  /**
   * Resizes the plot (actually the parent dialog/frame) to the new dimensions.
   *
   * @param size	the new size
   */
  public void resizePlot(Dimension size) {
    Dimension	current;
    int		width;
    int		height;
    Dimension	newSize;

    // determine differences
    current = getPlot().getContent().getSize();
    width   = size.width  - current.width;
    height  = size.height - current.height;

    // determine current size
    current = null;
    if (getParentDialog() != null)
      current = getParentDialog().getSize();
    else if (getParentFrame() != null)
      current = getParentFrame().getSize();
    else if (getParentInternalFrame() != null)
      current = getParentInternalFrame().getSize();

    // update size
    if (current != null) {
      newSize = new Dimension(current.width + width, current.height + height);
      if (getParentDialog() != null)
	getParentDialog().setSize(newSize);
      else if (getParentFrame() != null)
	getParentFrame().setSize(newSize);
      else if (getParentInternalFrame() != null)
	getParentInternalFrame().setSize(newSize);
    }
  }
}
