/**
 * Range.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */
package adams.core;

import java.io.Serializable;
import java.util.Vector;

/**
 * A class for managing a range of 1-based indices, e.g., 1-5, 3,7,9 or 1-7,9
 * (including "first", "second", "third", "last_2", "last_1" and "last").
 * A range can be inverted by surrounding it with "inv(...)".
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4113 $
 */
public class Range
  implements Serializable, CustomDisplayStringProvider, Comparable<Range>, ExampleProvider {

  /** for serialization. */
  private static final long serialVersionUID = -7995710565507092711L;

  /**
   * Represents a sub-range, either a single number of from-to.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4113 $
   */
  public static class SubRange
    implements Serializable, Comparable<SubRange> {

    /** for serialization. */
    private static final long serialVersionUID = -1352323320597824993L;

    /** the "from" (0-based). */
    protected Integer m_From;

    /** the "to" (0-based). */
    protected Integer m_To;

    /**
     * Initializes the sub-range as single number (0-based).
     *
     * @param from	the single number of the sub-range
     */
    public SubRange(int from) {
      this(from, null);
    }

    /**
     * Initializes the sub-range as range between (0-based) numbers.
     *
     * @param from	the start of the sub-range (incl)
     * @param to	the end of the sub-range (incl)
     */
    public SubRange(int from, Integer to) {
      super();

      m_From = from;
      m_To   = to;
    }

    /**
     * Returns the "from" part of the sub-range.
     *
     * @return		the from
     */
    public Integer getFrom() {
      return m_From;
    }

    /**
     * Returns whether a "to" is available.
     *
     * @return		true if a "to" is available
     */
    public boolean hasTo() {
      return (m_To != null);
    }

    /**
     * Returns the "to" of the sub-range.
     *
     * @return		the "to", null if not set
     */
    public Integer getTo() {
      return m_To;
    }

    /**
     * Checks whether the given index is within the limits of the sub-range.
     *
     * @param index	the (0-based) index to check
     * @return		true if within range
     */
    public boolean isInRange(int index) {
      if (m_To == null)
	return (m_From == index);
      else
	return ((m_From <= index) && (index <= m_To));
    }

    /**
     * Compares this subrange with the specified subrange for order.  Returns a
     * negative integer, zero, or a positive integer as this subrange is less
     * than, equal to, or greater than the specified subrange.
     * Uses the "from" as point of comparison and if those are equal, then
     * the "to" (if available).
     *
     * @param   o the subrange to be compared.
     * @return  a negative integer, zero, or a positive integer as this object
     *		is less than, equal to, or greater than the specified object.
     */
    public int compareTo(SubRange o) {
      int	result;

      result = getFrom().compareTo(o.getFrom());

      if (result == 0) {
	if (hasTo() && o.hasTo()) {
	  result = getTo().compareTo(o.getTo());
	}
	else if (!hasTo()) {
	  result = -1;
	}
	else {
	  result = +1;
	}
      }

      return result;
    }

    /**
     * Indicates whether some other object is "equal to" this one.
     *
     * @param obj	the reference object with which to compare.
     * @return		true if this object is the same as the obj argument;
     * 			false otherwise.
     */
    public boolean equals(Object obj) {
      if (!(obj instanceof SubRange))
	return false;
      else
	return (compareTo((SubRange) obj) == 0);
    }

    /**
     * Returns a string representation of the sub-range.
     *
     * @return		the representation
     */
    public String toString() {
      if (m_To == null)
	return "num=" + m_From;
      else
	return "from=" + m_From + ", to=" + m_To;
    }
  }

  /** the special string "-". */
  public final static String RANGE = "-";

  /** the special string ",". */
  public final static String SEPARATOR = ",";

  /** the special string "first". */
  public final static String FIRST = "first";

  /** the special string "second". */
  public final static String SECOND = "second";

  /** the special string "third". */
  public final static String THIRD = "third";

  /** the special string "last_1" (2nd to last). */
  public final static String LAST_1 = "last_1";

  /** the special string "last_2" (3rd to last). */
  public final static String LAST_2 = "last_2";

  /** the special string "last". */
  public final static String LAST = "last";

  /** the start string for inversion. */
  public final static String INV_START = "inv(";

  /** the end string for inversion. */
  public final static String INV_END = ")";

  /** "first-last" constant. */
  public final static String ALL = FIRST + RANGE + LAST;

  /** the range string. */
  protected String m_Range;

  /** the actual range, without the inversion. */
  protected String m_ActualRange;

  /** the maximum for the 1-based range. */
  protected int m_Max;

  /** whether the range is inverted. */
  protected boolean m_Inverted;

  /** the range parts. */
  protected Vector<SubRange> m_SubRanges;

  /**
   * Initializes with no range.
   */
  public Range() {
    this("");
  }

  /**
   * Initializes with the given range, but no maximum.
   *
   * @param range	the range to use
   */
  public Range(String range) {
    this(range, -1);
  }

  /**
   * Initializes with the given range and maximum.
   *
   * @param range	the range to use
   * @param max		the maximum of the 1-based index (e.g., use "10" to
   * 			allow "1-10" or -1 for uninitialized)
   */
  public Range(String range, int max) {
    super();

    setRange(range);
    setMax(max);
  }

  /**
   * Sets the range.
   *
   * @param value	the range to use
   */
  public void setRange(String value) {
    m_SubRanges = null;
    m_Range     = clean(value);
    m_Inverted  = m_Range.startsWith(INV_START) && m_Range.endsWith(INV_END);
    if (m_Inverted)
      m_ActualRange = m_Range.substring(INV_START.length(), m_Range.length() - INV_END.length());
    else
      m_ActualRange = m_Range;
  }

  /**
   * Sets the selected indices. Generates a range string out of the array.
   *
   * @param indices	the indices (0-based)
   */
  public void setIndices(Integer[] indices) {
    int[]	intIndices;
    int		i;

    intIndices = new int[indices.length];
    for (i = 0; i < indices.length; i++)
      intIndices[i] = indices[i];

    setIndices(intIndices);
  }

  /**
   * Sets the selected indices. Generates a range string out of the array.
   *
   * @param indices	the indices (0-based)
   */
  public void setIndices(int[] indices) {
    StringBuilder	range;
    int			i;
    int			start;
    int			end;
    int			diff;

    range = new StringBuilder();

    start = -1;
    end   = -1;
    if (indices.length > 0)
      start = indices[0];
    for (i = 1; i < indices.length; i++) {
      diff = indices[i] - indices[i - 1];
      if (diff > 1) {
	if (range.length() > 0)
	  range.append(",");
	if (start != indices[i - 1])
	  range.append((start + 1) + "-" + (indices[i - 1] + 1));
	else
	  range.append((start + 1));
	start = indices[i];
	end   = -1;
      }
      else {
	end = indices[i];
      }
    }

    if (start != -1) {
      if (end != -1) {
	if (range.length() > 0)
	  range.append(",");
	if (start != end)
	  range.append((start + 1) + "-" + (end + 1));
	else
	  range.append((start + 1));
      }
      else {
	if (range.length() > 0)
	  range.append(",");
	range.append((start + 1));
      }
    }

    setRange(range.toString());
  }

  /**
   * Checks whether the range is inverted.
   *
   * @return		true if inverted
   */
  public boolean isInverted()  {
    return m_Inverted;
  }

  /**
   * Sets whether the range is inverted or not.
   *
   * @param value	if true then the range is inverted
   */
  public void setInverted(boolean value) {
    if (value)
      setRange(INV_START + m_ActualRange + INV_END);
    else
      setRange(m_ActualRange);
  }

  /**
   * Returns the currently set range.
   *
   * @return		the range in use
   */
  public String getRange() {
    return m_Range;
  }

  /**
   * Sets the maximum (1-max will be allowed).
   *
   * @param value	the maximum for the 1-based index
   */
  public void setMax(int value) {
    if (value != m_Max) {
      if (value <= 0)
	m_Max = -1;
      else
	m_Max = value;
      m_SubRanges = null;
    }
  }

  /**
   * Returns the maximum.
   *
   * @return		the maximum for the 1-based index
   */
  public int getMax() {
    return m_Max;
  }

  /**
   * Checks whether a valid range has been supplied.
   *
   * @return		true if a valid range is available
   */
  public boolean hasRange() {
    return (m_Range.length() > 0);
  }

  /**
   * Cleanses the given string. Only allows "first", "last", ",", "-" and numbers.
   *
   * @param s		the string to clean
   * @return		the cleansed string, "" if invalid one provided
   */
  protected String clean(String s) {
    StringBuilder	result;
    String		tmp;
    StringBuilder	tmp2;
    int			i;
    char		chr;
    String[]		ranges;
    String[]		parts;
    int			from;
    int			to;
    boolean		inverted;

    result = new StringBuilder();

    // remove all invalid characters
    tmp    = s.toLowerCase();
    tmp    = tmp.replace(FIRST, "A").replace(LAST, "Z");
    tmp    = tmp.replace(SECOND, "B").replace(LAST_1, "Y");
    tmp    = tmp.replace(THIRD, "C").replace(LAST_2, "X");
    tmp    = tmp.replace(INV_START, "P").replace(INV_END, "Q");
    tmp2   = new StringBuilder();
    for (i = 0; i < tmp.length(); i++) {
      chr = tmp.charAt(i);
      if (chr == 'A')
	tmp2.append(FIRST);
      else if (chr == 'B')
	tmp2.append(SECOND);
      else if (chr == 'C')
	tmp2.append(THIRD);
      else if (chr == 'X')
	tmp2.append(LAST_2);
      else if (chr == 'Y')
	tmp2.append(LAST_1);
      else if (chr == 'Z')
	tmp2.append(LAST);
      else if (chr == 'P')
	tmp2.append(INV_START);
      else if (chr == 'Q')
	tmp2.append(INV_END);
      else if ((chr == SEPARATOR.charAt(0)) || (chr == RANGE.charAt(0)) || (chr == '_'))
	tmp2.append(chr);
      else if ((chr >= '0') && (chr <= '9'))
	tmp2.append(chr);
    }

    // test for inversion
    inverted = false;
    if (tmp2.length() >= INV_START.length() + 1 + INV_END.length()) {
      inverted =    tmp2.toString().startsWith(INV_START)
                 && tmp2.toString().endsWith(INV_END);
      if (inverted)
	tmp2 = new StringBuilder(tmp2.substring(INV_START.length(), tmp2.length() - INV_END.length()));
    }

    // remove invalid sub-ranges
    ranges = tmp2.toString().split(SEPARATOR);
    for (i = 0; i < ranges.length; i++) {
      // single number?
      if (ranges[i].indexOf(RANGE) == -1) {
	try {
	  parse(ranges[i]);
	  if (result.length() > 0)
	    result.append(SEPARATOR);
	  result.append(ranges[i]);
	}
	catch (Exception e) {
	  // ignored
	}
      }
      // from-to?
      else {
	parts = ranges[i].split(RANGE);
	if (parts.length == 2) {
	  try {
	    from = parse(parts[0], Integer.MAX_VALUE);
	    to   = parse(parts[1], Integer.MAX_VALUE);
	    if (from <= to) {
	      if (result.length() > 0)
		result.append(SEPARATOR);
	      result.append(ranges[i]);
	    }
	  }
	  catch (Exception e) {
	    // ignored
	  }
	}
      }
    }

    if (inverted) {
      result.insert(0, INV_START);
      result.append(INV_END);
    }

    return result.toString();
  }

  /**
   * Parses the 1-based index, 'first' and 'last' are accepted as well.
   *
   * @param s		the string to parse
   * @param max		the maximum value to use
   * @return		the 0-based index
   */
  protected int parse(String s, int max) {
    int		result;

    if (s.equals(FIRST))
      result = 0;
    else if (s.equals(SECOND))
      result = 1;
    else if (s.equals(THIRD))
      result = 2;
    else if (s.equals(LAST_2))
      result = max - 3;
    else if (s.equals(LAST_1))
      result = max - 2;
    else if (s.equals(LAST))
      result = max - 1;
    else
      result = Integer.parseInt(s) - 1;

    return result;
  }

  /**
   * Parses the 1-based index, 'first' and 'last' are accepted as well.
   *
   * @param s		the string to parse
   * @return		the 0-based index
   */
  protected int parse(String s) {
    return parse(s, m_Max);
  }

  /**
   * Parses the string and generates the sub-ranges.
   *
   * @return		the parsed sub-ranges
   */
  protected Vector<SubRange> parse() {
    Vector<SubRange>	result;
    String[]		ranges;
    String[]		parts;
    int			i;
    int			from;
    int			to;

    result = new Vector<SubRange>();
    if (m_Max == -1)
      return result;

    ranges = m_ActualRange.split(SEPARATOR);
    for (i = 0; i < ranges.length; i++) {
      if (ranges[i].length() == 0)
	continue;
      if (ranges[i].indexOf(RANGE) == -1) {
	from = parse(ranges[i]);
	if ((from >= 0) && (from < m_Max))
	  result.add(new SubRange(from));
      }
      else {
	parts = ranges[i].split(RANGE);
	from  = parse(parts[0]);
	to    = parse(parts[1]);
	if ((from <= to) && (from >= 0) && (to < m_Max))
	  result.add(new SubRange(from, to));
	else if ((from <= to) && (from >= 0) && (from < m_Max))
	  result.add(new SubRange(from, m_Max));
      }
    }

    return result;
  }

  /**
   * Returns the sub-ranges, initializes them if necessary.
   *
   * @return		the sub-ranges
   */
  protected synchronized Vector<SubRange> getSubRanges() {
    if (m_SubRanges == null)
      m_SubRanges = parse();

    return m_SubRanges;
  }

  /**
   * Checks whether the provided 0-based index is within the range.
   *
   * @param index	the index to check
   * @return		true if in range
   */
  public boolean isInRange(int index) {
    boolean		result;
    int			i;
    Vector<SubRange>	ranges;

    result = m_Inverted;

    ranges = getSubRanges();

    for (i = 0; i < ranges.size(); i++) {
      if (m_Inverted) {
	if (ranges.get(i).isInRange(index)) {
	  result = false;
	  break;
	}
      }
      else {
	if (ranges.get(i).isInRange(index)) {
	  result = true;
	  break;
	}
      }
    }

    return result;
  }

  /**
   * Returns the integer indices. Gets always generated on-the-fly!
   *
   * @return		the indices, 0-length array if not possible
   */
  public int[] getIntIndices() {
    int[]		result;
    Vector<Integer>	indices;
    int			i;

    // collect indices
    indices = new Vector<Integer>();
    for (i = 0; i < m_Max; i++) {
      if (isInRange(i))
	indices.add(i);
    }

    // transfer indices
    result = new int[indices.size()];
    for (i = 0; i < indices.size(); i++)
      result[i] = indices.get(i);

    return result;
  }

  /**
   * Turns the range into a list of from-to segements. The indices are 0-based.
   * In case a subrange consists only of a single index, the second one is the
   * same.
   *
   * @return		the segments
   */
  public int[][] getIntSegments() {
    int[][]	result;
    int		i;
    SubRange	sub;

    if (getSubRanges() == null)
      return new int[0][];

    result = new int[getSubRanges().size()][2];
    for (i = 0; i < getSubRanges().size(); i++) {
      sub          = getSubRanges().get(i);
      result[i][0] = sub.getFrom();
      if (sub.hasTo())
	result[i][1] = sub.getTo();
      else
	result[i][1] = sub.getFrom();
    }

    return result;
  }

  /**
   * Compares this subrange with the specified subrange for order.  Returns a
   * negative integer, zero, or a positive integer as this subrange is less
   * than, equal to, or greater than the specified subrange.
   * Uses the "from" as point of comparison and if those are equal, then
   * the "to" (if available).
   *
   * @param   o the subrange to be compared.
   * @return  a negative integer, zero, or a positive integer as this object
   *		is less than, equal to, or greater than the specified object.
   */
  public int compareTo(Range o) {
    int		result;
    int[]	indicesThis;
    int[]	indicesOther;
    int		i;

    indicesThis  = getIntIndices();
    indicesOther = o.getIntIndices();

    result = new Integer(indicesThis.length).compareTo(new Integer(indicesOther.length));

    if (result == 0) {
      for (i = 0; i < indicesThis.length; i++) {
	result = new Integer(indicesThis[i]).compareTo(new Integer(indicesOther[i]));
	if (result != 0)
	  break;
      }
    }

    return result;
  }

  /**
   * Indicates whether some other object is "equal to" this one.
   *
   * @param obj		the reference object with which to compare.
   * @return		true if this object is the same as the obj argument;
   * 			false otherwise.
   */
  public boolean equals(Object obj) {
    if (!(obj instanceof Range))
      return false;
    else
      return (compareTo((Range) obj) == 0);
  }

  /**
   * Returns a string representation of the range.
   *
   * @return		the representation
   */
  public String toString() {
    return "range=" + m_Range + ", max=" + m_Max + ", inv=" + m_Inverted;
  }

  /**
   * Returns the custom display string.
   *
   * @return		the string
   */
  public String toDisplay() {
    return getRange();
  }

  /**
   * Returns the example.
   *
   * @return		the example
   */
  public String getExample() {
    return
        "A range is a comma-separated list of single 1-based indices or "
      + "sub-ranges of indices ('start-end'); "
      + "'inv(...)' inverts the range '...'; "
      + "the following placeholders can be used as well: "
      + FIRST + ", " + SECOND + ", " + THIRD + ", " + LAST_2 + ", " + LAST_1 + ", " + LAST;
  }

  /**
   * Turns the integer indices into a range object.
   *
   * @param indices	the 0-based indices
   * @return		the generated range
   */
  public static Range toRange(int[] indices) {
    Range		result;
    StringBuilder	range;
    int			i;
    Integer		start;
    Integer		current;

    if (indices.length == 0)
      return new Range();

    range   = new StringBuilder();
    i       = 1;
    start   = indices[0];
    current = indices[0];
    range.append((current + 1));
    while (i < indices.length) {
      if (indices[i] - current > 1) {
	if (start < current) {
	  range.append("-");
	  range.append((current + 1));
	}
	range.append(",");
	start   = indices[i];
	current = indices[i];
	range.append((current + 1));
      }
      else {
	current = indices[i];
      }

      i++;

      if (i == indices.length) {
	if (start < current) {
	  range.append("-");
	  range.append((current + 1));
	}
      }
    }

    result = new Range(range.toString());

    return result;
  }
}
