/*
 * SerializableObjectHelper.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.core;

import adams.core.option.OptionUtils;

/**
 * Helper class for serializing an object's member variables.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4005 $
 */
public class SerializableObjectHelper
  extends ConsoleObject {

  /** for serialization. */
  private static final long serialVersionUID = -6245176175718570463L;

  /** the owner. */
  protected SerializableObject m_Owner;

  /** whether the setup got loaded or generated already. */
  protected boolean m_SetupLoadedOrGenerated;

  /**
   * Initializes the helper.
   *
   * @param owner	the owning object
   */
  public SerializableObjectHelper(SerializableObject owner) {
    super();

    m_Owner                  = owner;
    m_SetupLoadedOrGenerated = false;
  }

  /**
   * Returns the owning object.
   *
   * @return		the owner this helper belongs to
   */
  public SerializableObject getOwner() {
    return m_Owner;
  }

  /**
   * Resets the helper, i.e., it will attempt to load the setup again.
   */
  public void reset() {
    m_SetupLoadedOrGenerated = false;
  }

  /**
   * Attempts to load the setup. If the serialization file object is pointing
   * to a directory or the override flag is set in the owner, then the
   * deserialization from file will be skipped and a fresh setup generated
   * instead. After the new setup has been generated, the data is serialized
   * only if the serialization file is to pointing to an actual file.
   *
   * @see	SerializableObject#getSerializationFile()
   * @see 	SerializableObject#getOverrideSerializedFile()
   * @see	#generateSetup()
   * @see	#saveSetup()
   */
  public void loadSetup() {
    Object[]	setup;
    Object[]	fullSetup;
    String	options;
    boolean	generate;

    // can we skip loading the setup?
    if (m_SetupLoadedOrGenerated && !m_Owner.getOverrideSerializedFile())
      return;

    generate = false;

    if (    m_Owner.getOverrideSerializedFile()
	 || m_Owner.getSerializationFile().isDirectory()
	 || !m_Owner.getSerializationFile().exists() ) {
      generate = true;
    }
    else {
      fullSetup = null;
      try {
	fullSetup = SerializationHelper.readAll(m_Owner.getSerializationFile().getAbsolutePath());
      }
      catch (Exception e) {
	fullSetup = null;
	getSystemErr().println("Error deserializing from '" + m_Owner.getSerializationFile() + "':");
	getSystemErr().printStackTrace(e);
      }

      // do we have to regenerate the setup, since loading failed?
      if (fullSetup == null) {
	generate = true;
      }
      else {
	setup = new Object[fullSetup.length - 1];
	if (fullSetup.length > 1)
	  System.arraycopy(fullSetup, 1, setup, 0, fullSetup.length - 1);
	options = (String) fullSetup[0];
	// options different? -> regenerate setup!
	if (!options.equals(OptionUtils.getCommandLine(m_Owner))) {
	  generate = true;
	}
	else {
	  m_Owner.setSerializationSetup(setup);
	  m_SetupLoadedOrGenerated = true;
	}
      }
    }

    // generate the setup again, if necessary
    if (generate) {
      generateSetup();
      if (!m_Owner.getSerializationFile().isDirectory())
	saveSetup();
      m_SetupLoadedOrGenerated = true;
    }
  }

  /**
   * Saves the owner's member variables to the specified file (ignored if
   * pointing to a directory).
   *
   * @see	SerializableObject#getSerializationFile()
   * @see	SerializableObject#retrieveSerializationSetup()
   */
  public void saveSetup() {
    Object[]	setup;
    Object[]	fullSetup;

    if (m_Owner.getSerializationFile().isDirectory())
      return;

    try {
      // add options (if available) to data that is to be serialized
      setup        = m_Owner.retrieveSerializationSetup();
      fullSetup    = new Object[setup.length + 1];
      fullSetup[0] = OptionUtils.getCommandLine(m_Owner);
      if (setup.length > 0)
	System.arraycopy(setup, 0, fullSetup, 1, setup.length);

      // write data to disk
      SerializationHelper.writeAll(m_Owner.getSerializationFile().getAbsolutePath(), fullSetup);
    }
    catch (Exception e) {
      getSystemErr().println("Error serializing to '" + m_Owner.getSerializationFile() + "':");
      getSystemErr().printStackTrace(e);
    }
  }

  /**
   * Forces the owner to regenerate all objects that are necessary for
   * serialization.
   */
  public void generateSetup() {
    m_Owner.initSerializationSetup();
  }

  /**
   * Generates some information.
   *
   * @return		the information
   */
  public String toString() {
    String	result;

    result =   "Owner=" + m_Owner.getClass().getName() + ", "
             + "File=" + m_Owner.getSerializationFile() + ", "
             + "IsDirectory=" + m_Owner.getSerializationFile().isDirectory() + ", "
             + "Override=" + m_Owner.getOverrideSerializedFile();

    return result;
  }
}
