/*
 * BaseRegExp.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.core.base;

import java.util.regex.Pattern;

import adams.core.Utils;

/**
 * Wrapper for a regular expression string to be editable in the GOE. Basically
 * the same as BaseString, but checks whether the string represents a valid
 * regular expression.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3978 $
 */
public class BaseRegExp
  extends BaseString {

  /** for serialization. */
  private static final long serialVersionUID = -8687858764646783666L;

  /** the match-all expression. */
  public final static String MATCH_ALL = ".*";

  /** for performing pattern matching. */
  protected Pattern m_Pattern;

  /**
   * Initializes the string with length 0.
   */
  public BaseRegExp() {
    this(MATCH_ALL);
  }

  /**
   * Initializes the object with the string to parse.
   *
   * @param s		the string to parse
   */
  public BaseRegExp(String s) {
    super(s);
  }

  /**
   * Checks whether the string value is a valid presentation for this class.
   *
   * @param value	the string value to check
   * @return		always true
   */
  public boolean isValid(String value) {
    try {
      Pattern.compile(value);
      return true;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Converts the string according to the specified conversion.
   * <p/>
   * Implementation performs no conversion, merely initializes
   * the pattern matcher.
   *
   * @param value	the string to convert
   * @return		the converted string
   * @see		#m_Pattern
   */
  protected String convert(String value) {
    m_Pattern = Pattern.compile(value);
    return super.convert(value);
  }

  /**
   * Returns the backquoted String value.
   *
   * @return		the backquoted String value
   */
  public String stringValue() {
    return Utils.backQuoteChars(getValue());
  }

  /**
   * Returns a tool tip for the GUI editor (ignored if null is returned).
   *
   * @return		the tool tip
   */
  public String getTipText() {
    return "Regular expression.";
  }

  /**
   * Checks whether the string matches the regular expression.
   *
   * @param s		the string to check against regexp
   * @return		true if the string matches the regexp
   */
  public boolean isMatch(String s) {
    return m_Pattern.matcher(s).matches();
  }

  /**
   * Checks whether the expression is the match-all expression.
   *
   * @return		true if the match-all expression
   */
  public boolean isMatchAll() {
    return getValue().equals(MATCH_ALL);
  }

  /**
   * Checks whether the expression empty.
   *
   * @return		true if empty expression
   */
  public boolean isEmpty() {
    return (getValue().length() == 0);
  }
}
