/**
 * DefaultCommandLineHandler.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core.option;


/**
 * Only uses the classname as commandline, no actual option handling.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2822 $
 */
public class DefaultCommandLineHandler
  extends AbstractCommandLineHandler {

  /** for serialization. */
  private static final long serialVersionUID = -5233496867185402778L;

  /**
   * Generates an object from the specified commandline.
   *
   * @param cmd		the commandline to create the object from
   * @return		the created object, null in case of error
   */
  public Object fromCommandLine(String cmd) {
    String	classname;

    if (cmd.indexOf(' ') == -1)
      classname = cmd;
    else
      classname = cmd.substring(0, cmd.indexOf(' '));

    return fromArray(new String[]{classname});
  }

  /**
   * Generates an object from the commandline options.
   *
   * @param args	the commandline options to create the object from
   * @return		the created object, null in case of error
   */
  public Object fromArray(String[] args) {
    Object	result;

    result = null;

    if (args.length > 0) {
      try {
	result = Class.forName(args[0]).newInstance();
      }
      catch (Exception e) {
	System.err.println("Failed to instantiate object from array (fromArray):");
	e.printStackTrace();
      }
    }

    return result;
  }

  /**
   * Generates a commandline from the specified object.
   *
   * @param obj		the object to create the commandline for
   * @return		the generated commandline
   */
  public String toCommandLine(Object obj) {
    return obj.getClass().getName();
  }

  /**
   * Generates an options array from the specified object.
   *
   * @param obj		the object to create the array for
   * @return		the generated array
   */
  public String[] toArray(Object obj) {
    return new String[]{toCommandLine(obj)};
  }

  /**
   * Returns the commandline options (without classname) of the specified object.
   *
   * @param obj		the object to get the options from
   * @return		always array with length 0
   */
  public String[] getOptions(Object obj) {
    return new String[0];
  }

  /**
   * Sets the options of the specified object.
   *
   * @param obj		the object to set the options for
   * @param args	the options
   * @return		always true, does nothing
   */
  public boolean setOptions(Object obj, String[] args) {
    return true;
  }

  /**
   * Splits the commandline into an array.
   *
   * @param cmdline	the commandline to split
   * @return		the generated array of options
   */
  public String[] splitOptions(String cmdline) {
    String[]	result;

    try {
      result = OptionUtils.splitOptions(cmdline);
    }
    catch (Exception e) {
      result = new String[0];
    }

    return result;
  }

  /**
   * Turns the option array back into a commandline.
   *
   * @param args	the options to turn into a commandline
   * @return		the generated commandline
   */
  public String joinOptions(String[] args) {
    return OptionUtils.joinOptions(args);
  }

  /**
   * Checks whether the given class can be processed.
   *
   * @param cls		the class to inspect
   * @return		true if the handler can process the class
   */
  public boolean handles(Class cls) {
    return true;
  }
}
