/**
 * DocBookProducer.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core.option;

import java.lang.reflect.Array;
import java.lang.reflect.Method;

import adams.core.AdditionalInformationHandler;
import adams.doc.docbook.AbstractElement;
import adams.doc.docbook.Article;
import adams.doc.docbook.Document;
import adams.doc.docbook.InformalTable;
import adams.doc.docbook.Paragraph;
import adams.doc.docbook.Section;

/**
 * Generates documentation in
 * <a href="http://www.docbook.org/" target="_blank">DocBook</a> format.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3487 $
 */
public class DocBookProducer
  extends AbstractRecursiveOptionProducer<String,AbstractElement>
  implements RecursiveOptionProducer, FileBasedProducer {

  /** for serialization. */
  private static final long serialVersionUID = -1354998807180416601L;

  /** the output vector. */
  protected Document m_Document;

  /** whether to output default values as well. */
  protected boolean m_OutputDefaultValues;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Generates comprehensive output in DocBook format, including the "
      + "descriptions for the options and the default values.\n\n"
      + "For more information on the DocBook XML format, see:\n"
      + "http://www.docbook.org/";
  }

  /**
   * Initializes the visitor.
   */
  protected void initialize() {
    super.initialize();

    m_Document            = new Document();
    m_OutputDefaultValues = false;
  }

  /**
   * Initializes the output data structure.
   *
   * @return		the created data structure
   */
  protected String initOutput() {
    return "";
  }

  /**
   * Sets whether to output default values as well.
   *
   * @param value	if true then default values are output as well
   */
  public void setOutputDefaultValues(boolean value) {
    m_OutputDefaultValues = value;
  }

  /**
   * Returns whyether default values are output as well.
   *
   * @return		true if default values are output as well
   */
  public boolean getOutputDefaultValues() {
    return m_OutputDefaultValues;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputDefaultValuesTipText() {
    return "Whether to output or suppress default values.";
  }

  /**
   * Starts a DocBook section. Adds a synopsis section if "globalInfo"
   * available. An additional information section is added if the object
   * implements AdditionalInformationHandler.
   *
   * @param parent	the parent to add the elements to
   * @param obj		the object to process
   * @return		the generated section element
   */
  protected Section startSection(AbstractElement parent, Object obj) {
    Section	result;
    Method	method;
    String	globalInfo;

    result = new Section(obj.getClass().getName());
    parent.add(result);

    try {
      method = obj.getClass().getMethod("globalInfo", new Class[0]);
      if (method != null) {
	globalInfo = (String) method.invoke(obj, new Object[0]);
	result.add(
	    new Section(
		"Synopsis",
		globalInfo));
      }
    }
    catch (Exception e) {
      // ignored
    }

    if (obj instanceof AdditionalInformationHandler) {
      result.add(
	  new Section(
	      "Additional information",
	      ((AdditionalInformationHandler) getInput()).getAdditionalInformation()));
    }

    return result;
  }

  /**
   * Adds the description of the option as paragraph.
   *
   * @param parent	the parent element to add to
   * @param option	the option to use
   */
  protected void addDescription(AbstractElement parent, AbstractOption option) {
    String		description;
    Method		method;

    try {
      method      = option.getToolTipMethod();
      description = (String) method.invoke(option.getOptionHandler(), new Object[0]);
    }
    catch (Exception e) {
      description = "-no description-";
    }

    parent.add(new Paragraph(description));
  }

  /**
   * Visits a boolean option.
   *
   * @param option	the boolean option
   * @return		the last internal data structure that was generated
   */
  public AbstractElement processOption(BooleanOption option) {
    AbstractElement	result;
    boolean		different;
    InformalTable	table;

    result = null;

    different = ((Boolean) getCurrentValue(option)).equals((Boolean) option.getDefaultValue());

    if (m_OutputDefaultValues || different) {
      result = new Section(option.getProperty());
      addDescription(result, option);
      table = new InformalTable(2);
      result.add(table);
      table.addRow(
	  new String[]{
	      "command-line flag",
	      "-" + option.getCommandline()
	  });
      table.addRow(
	  new String[]{
	      "value",
	      "" + getCurrentValue(option)
	  });
    }

    if (result != null)
      m_Nesting.peek().add(result);

    return result;
  }

  /**
   * Visits an argument option.
   *
   * @param option	the argument option
   * @return		the last internal data structure that was generated
   */
  public AbstractElement processOption(AbstractArgumentOption option) {
    AbstractElement	result;
    Object		currValue;
    Object		currValues;
    int			i;
    InformalTable	table;

    result = null;

    if (option.isVariableAttached() && !m_OutputVariableValues) {
      result = new Section(option.getProperty());
      addDescription(result, option);
      table = new InformalTable(2);
      result.add(table);
      table.addRow(
	  new String[]{
	      "command-line flag",
	      "-" + option.getCommandline()
	  });
      table.addRow(
	  new String[]{
	      "variable",
	      option.getVariable()
	  });
    }
    else {
      currValue = getCurrentValue(option);

      if (m_OutputDefaultValues || !isDefaultValue(option, currValue)) {
	currValues = null;

	if (currValue != null) {
	  result = new Section(option.getProperty());
	  addDescription(result, option);
	  table = new InformalTable(2);
	  result.add(table);
	  table.addRow(
	      new String[]{
		  "command-line flag",
		  "-" + option.getCommandline()
	      });

	  if (!option.isMultiple()) {
	    currValues = Array.newInstance(option.getBaseClass(), 1);
	    Array.set(currValues, 0, currValue);
	  }
	  else {
	    currValues = currValue;
	  }

	  for (i = 0; i < Array.getLength(currValues); i++) {
	    table.addRow(
		new String[]{
		    "value",
		    option.toString(Array.get(currValues, i))
		});
	  }
	}
      }
    }

    if (result != null)
      m_Nesting.peek().add(result);

    return result;
  }

  /**
   * Visits a class option.
   *
   * @param option	the class option
   * @return		the last internal data structure that was generated
   */
  public AbstractElement processOption(ClassOption option) {
    AbstractElement		result;
    Object			currValue;
    Object			currValues;
    int				i;
    Object			value;
    InformalTable		table;
    Section			nested;
    Section			section;
    AbstractCommandLineHandler	handler;

    result = null;

    if (option.isVariableAttached() && !m_OutputVariableValues) {
      result = new Section(option.getProperty());
      addDescription(result, option);
      table = new InformalTable(2);
      result.add(table);
      table.addRow(
	  new String[]{
	      "command-line flag",
	      "-" + option.getCommandline()
	  });
      table.addRow(
	  new String[]{
	      "variable",
	      option.getVariable()
	  });
    }
    else {
      currValue = getCurrentValue(option);

      if (m_OutputDefaultValues || !isDefaultValue(option, currValue)) {
	currValues = null;

	if (currValue != null) {
	  result = new Section(option.getProperty());
	  addDescription(result, option);
	  table = new InformalTable(2);
	  result.add(table);
	  table.addRow(
	      new String[]{
		  "command-line flag",
		  "-" + option.getCommandline()
	      });

	  if (!option.isMultiple()) {
	    currValues = Array.newInstance(option.getBaseClass(), 1);
	    Array.set(currValues, 0, currValue);
	  }
	  else {
	    currValues = currValue;
	  }

	  for (i = 0; i < Array.getLength(currValues); i++) {
	    value  = Array.get(currValues, i);
	    nested = startSection(result, value);
	    m_Nesting.push(nested);
	    if (value instanceof OptionHandler) {
	      section = new Section("Options");
	      nested.add(section);
	      m_Nesting.push(section);
	      doProduce(((OptionHandler) value).getOptionManager());
	      m_Nesting.pop();
	    }
	    else {
	      handler = AbstractCommandLineHandler.getHandler(value);
	      section = new Section("Command-line");
	      nested.add(section);
	      table = new InformalTable(2);
	      section.add(table);
	      table.addRow(
		  new String[]{
		      "class",
		      "" + value.getClass().getName()
		  });
	      table.addRow(
		  new String[]{
		      "options",
		      OptionUtils.joinOptions(handler.getOptions(value))
		  });
	    }
	    m_Nesting.pop();
	  }
	}
      }
    }

    if (result != null)
      m_Nesting.peek().add(result);

    return result;
  }

  /**
   * Hook-method before starting visiting options.
   */
  protected void preProduce() {
    Section	section;

    super.preProduce();

    m_Output = null;

    m_Document = new Document();
    m_Document.setRoot(new Article("Setup documentation"));

    startSection(m_Document.getRoot(), getInput());

    section = new Section("Options");
    m_Document.getRoot().add(section);
    m_Nesting.push(section);
  }

  /**
   * Returns the output generated from the visit.
   *
   * @return		the output
   */
  public String getOutput() {
    StringBuilder	buffer;

    if (m_Output == null) {
      buffer = new StringBuilder();
      m_Document.toXML(buffer);
      m_Output = buffer.toString();
    }

    return m_Output;
  }

  /**
   * Returns the output generated from the visit.
   *
   * @return		the output, null in case of an error
   */
  public String toString() {
    return getOutput();
  }

  /**
   * Cleans up data structures, frees up memory.
   */
  public void cleanUp() {
    super.cleanUp();

    m_Document = null;
  }

  /**
   * Returns the description of the file format.
   *
   * @return		the description
   */
  public String getFileFormat() {
    return "DocBook";
  }

  /**
   * Returns the default file extension (without the dot).
   *
   * @return		the default extension
   */
  public String getDefaultFileExtension() {
    return "xml";
  }

  /**
   * Returns the file extensions (without the dot).
   *
   * @return		the extensions
   */
  public String[] getFileExtensions() {
    return new String[]{getDefaultFileExtension()};
  }
}
