/**
 * DatabaseManager.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.db;

import java.io.Serializable;
import java.util.Hashtable;
import java.util.Iterator;

/**
 * Manages the database URL/connection object relations.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3766 $
 * @param <T> the type of database this manager is for
 */
public class DatabaseManager<T extends AbstractDatabaseConnection>
  implements Serializable {

  /** for serialization. */
  private static final long serialVersionUID = -8832349882994980783L;

  /** the database this manager is for. */
  protected String m_DatabaseName;

  /** for storing the database connection objects. */
  protected Hashtable<String,T> m_Connections;

  /** the default database connection. */
  protected T m_DefaultDatabaseConnection;

  /**
   * Initializes the manager.
   *
   * @param dbName	the name of the database this manager is for
   */
  public DatabaseManager(String dbName) {
    super();

    m_DatabaseName = dbName;
    m_Connections  = new Hashtable<String,T>();
  }

  /**
   * Returns the name of the database this manager is handling.
   *
   * @return		the database name
   */
  public String getDatabaseName() {
    return m_DatabaseName;
  }

  /**
   * Generates a URL that includes the user name.
   *
   * @param dbcon	the database connection object to create the URL for
   * @return		the complete URL
   */
  protected String createURL(T dbcon) {
    return createURL(dbcon.getURL(), dbcon.getUser());
  }

  /**
   * Generates a URL that includes the user name.
   *
   * @param url		the JDBC URL
   * @param user	the database user
   * @return		the complete URL
   */
  protected String createURL(String url, String user) {
    return user + "@" + url;
  }

  /**
   * Checks whether a database object for the specified URL is
   * available.
   *
   * @param url		the URL to check
   * @param user	the database user
   * @return		true if a database object is available
   */
  public boolean has(String url, String user) {
    if (url == null)
      return false;
    else
      return m_Connections.containsKey(createURL(url, user));
  }

  /**
   * Gets the database object for the specified URL.
   *
   * @param url		the URL to get the database for
   * @param user	the database user
   * @return		the database object if available, otherwise null
   */
  public T get(String url, String user) {
    if (url == null)
      return null;
    else
      return m_Connections.get(createURL(url, user));
  }

  /**
   * Adds the database object (uses the current URL as key).
   *
   * @param dbcon	the database object to add
   * @return		the previous database, null if no previous one stored
   */
  public T add(T dbcon) {
    if (dbcon == null)
      return null;

    dbcon.setOwner(this);

    return m_Connections.put(createURL(dbcon), dbcon);
  }

  /**
   * Sets the default database connection.
   *
   * @param value	the default connection object
   */
  public void setDefault(T value) {
    m_DefaultDatabaseConnection = value;
    m_DefaultDatabaseConnection.setOwner(this);
    if (value != null)
      add(value);
  }

  /**
   * Returns the default database connection.
   *
   * @return		the default
   */
  public T getDefault() {
    return m_DefaultDatabaseConnection;
  }

  /**
   * Returns an iterator over all databases.
   *
   * @return		the iterator
   */
  public Iterator<T> iterator() {
    return m_Connections.values().iterator();
  }

  /**
   * Returns a short string representation of the manager.
   *
   * @return		the string representation
   */
  public String toString() {
    return getDatabaseName() + ": " + m_Connections.keySet();
  }
}
