/*
 * GlobalActorHelper.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.core;

import java.util.Vector;

import adams.core.ConsoleObject;
import adams.flow.control.AbstractDirectedControlActor;
import adams.flow.standalone.GlobalActors;

/**
 * Helper class for global actors.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3470 $
 */
public class GlobalActorHelper
  extends ConsoleObject {

  /** for serialization. */
  private static final long serialVersionUID = -763479272812116920L;

  /**
   * Checks a control actor's children whether they contain the global actor
   * that we're looking for.
   *
   * @param group	the group to check
   * @param name	the name of the global actor
   * @return		the global actor or null if not found
   */
  public AbstractActor findGlobalActor(ActorHandler group, GlobalActorReference name) {
    AbstractActor		result;
    int				i;
    GlobalActors		global;
    int				index;
    AbstractExternalActor	external;

    result = null;

    for (i = 0; i < group.size(); i++) {
      if (group.get(i) instanceof GlobalActors) {
	global = (GlobalActors) group.get(i);
	index  = global.indexOf(name.toString());
	if (index > -1) {
	  result = global.get(index);
	  break;
	}
      }
      else if (group.get(i) instanceof AbstractExternalActor) {
	external = (AbstractExternalActor) group.get(i);
	if (external.getExternalActor() instanceof ActorHandler) {
	  result = findGlobalActor((ActorHandler) external.getExternalActor(), name);
	  if (result != null)
	    break;
	}
      }
    }

    return result;
  }

  /**
   * Tries to find the global actor referenced by its global name.
   *
   * @param root	the root to search in
   * @param name	the name of the global actor
   * @return		the global actor or null if not found
   */
  public AbstractActor findGlobalActor(AbstractActor root, GlobalActorReference name) {
    AbstractActor	result;

    result = null;

    if (root == null) {
      getSystemErr().println("No root container found!");
    }
    else if (!(root instanceof AbstractDirectedControlActor)) {
      getSystemErr().println(
	  "Root is not a container ('" + root.getFullName() + "'/"
	  + root.getClass().getName() + ")!");
      root = null;
    }

    if (root != null)
      result = findGlobalActor((ActorHandler) root, name);

    return result;
  }

  /**
   * Tries to find the referenced global actor. First all possible actor
   * handlers are located recursively (up to the root) that allow also
   * singletons. This list of actors is then searched for the global actor.
   *
   * @param actor	the actor to start from
   * @param name	the name of the global actor
   * @return		the global actor or null if not found
   * @see		ActorUtils#findActorHandlers(AbstractActor, boolean)
   */
  public AbstractActor findGlobalActorRecursive(AbstractActor actor, GlobalActorReference name) {
    AbstractActor		result;
    Vector<ActorHandler>	handlers;
    int				i;

    result   = null;
    handlers = ActorUtils.findActorHandlers(actor, true);
    for (i = 0; i < handlers.size(); i++) {
      result = findGlobalActor(handlers.get(i), name);
      if (result != null)
	break;
    }

    return result;
  }
}
