/*
 * LogViewer.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.sink;

import java.awt.BorderLayout;
import java.awt.Dialog.ModalityType;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.Collections;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import adams.core.Properties;
import adams.core.Utils;
import adams.core.io.LogEntryWriter;
import adams.db.LogEntry;
import adams.flow.core.Token;
import adams.gui.chooser.BaseDirectoryChooser;
import adams.gui.chooser.BaseFileChooser;
import adams.gui.core.BasePanel;
import adams.gui.core.BaseTable;
import adams.gui.core.BaseTableWithButtons;
import adams.gui.core.ExtensionFileFilter;
import adams.gui.dialog.TextDialog;
import adams.gui.tools.LogEntryViewerTableModel;

/**
 <!-- globalinfo-start -->
 * Actor that stores LogEntry objects in the Log table.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.db.LogEntry<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Logger
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4095 $
 */
public class LogViewer
  extends AbstractDisplay {

  /** for serialization. */
  private static final long serialVersionUID = -1980631598893105134L;

  /** the table with the entries. */
  protected BaseTableWithButtons m_TableEntries;

  /** the button for displaying the message details. */
  protected JButton m_ButtonDisplay;

  /** the button for saving the selected entries (or all if none selected). */
  protected JButton m_ButtonSave;

  /** the dialog for displaying the message details. */
  protected TextDialog m_MessageDialog;

  /** the base file chooser for saving the log entries. */
  protected BaseFileChooser m_BaseFileChooser;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Actor that displays LogEntry objects, but doesn't store them in the database.";
  }

  /**
   * Returns the default width for the dialog.
   *
   * @return		the default width
   */
  protected int getDefaultWidth() {
    return 600;
  }

  /**
   * Returns the default height for the dialog.
   *
   * @return		the default height
   */
  protected int getDefaultHeight() {
    return 400;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.db.LogEntry.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{LogEntry.class};
  }

  /**
   * Clears the content of the panel.
   */
  public void clearPanel() {
    ((LogEntryViewerTableModel) m_TableEntries.getModel()).clear();
  }

  /**
   * Displays the message of the given entry in a separate window.
   *
   * @param entry	the entry to display
   */
  protected void displayMessage(LogEntry entry) {
    StringBuilder 	content;
    Properties 		props;
    Vector<String> 	keys;
    int			i;

    // build content
    content = new StringBuilder();
    props   = entry.getMessageAsProperties();
    keys    = new Vector<String>(props.keySetAll());
    Collections.sort(keys);
    for (i = 0; i < keys.size(); i++) {
      if (i > 0)
	content.append("\n");
      content.append(keys.get(i) + ":\n");
      content.append(Utils.indent(props.getProperty(keys.get(i)), 3));
    }

    // display content
    if (m_MessageDialog == null) {
      if (m_TableEntries.getParentDialog() != null)
	m_MessageDialog = new TextDialog(m_TableEntries.getParentDialog(), ModalityType.DOCUMENT_MODAL);
      else
	m_MessageDialog = new TextDialog(m_TableEntries.getParentFrame(), true);
    }
    m_MessageDialog.setTitle("Message (" + entry.getDatabaseID() + "/" + entry.getGenerationAsString() + ")");
    m_MessageDialog.setContent(content.toString());
    m_MessageDialog.setSize(400, 300);
    m_MessageDialog.setLocationRelativeTo(m_TableEntries);
    m_MessageDialog.setVisible(true);
  }

  /**
   * Creates the panel to display in the dialog.
   *
   * @return		the panel
   */
  protected BasePanel newPanel() {
    BasePanel	result;

    result = new BasePanel(new BorderLayout());
    result.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));

    m_TableEntries = new BaseTableWithButtons(new LogEntryViewerTableModel());
    m_TableEntries.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    m_TableEntries.setAutoResizeMode(BaseTable.AUTO_RESIZE_OFF);
    m_TableEntries.setInfoVisible(true);
    m_TableEntries.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
      public void valueChanged(ListSelectionEvent e) {
	m_ButtonDisplay.setEnabled(m_TableEntries.getSelectedRowCount() == 1);
      }
    });
    result.add(m_TableEntries, BorderLayout.CENTER);

    m_ButtonDisplay = new JButton("Display...");
    m_ButtonDisplay.setMnemonic('D');
    m_ButtonDisplay.setEnabled(false);
    m_ButtonDisplay.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	int row = m_TableEntries.getSelectedRow();
	LogEntryViewerTableModel model = (LogEntryViewerTableModel) m_TableEntries.getModel();
	LogEntry entry = model.getLogEntryAt(row);
	displayMessage(entry);
      }
    });
    m_TableEntries.addToButtonsPanel(m_ButtonDisplay);
    m_TableEntries.setDoubleClickButton(m_ButtonDisplay);

    m_ButtonSave = new JButton("Save...");
    m_ButtonSave.setMnemonic('S');
    m_ButtonSave.setEnabled(true);
    m_ButtonSave.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	int retVal = m_BaseFileChooser.showSaveDialog(m_TableEntries);
	if (retVal != BaseDirectoryChooser.APPROVE_OPTION)
	  return;
	String filename = m_BaseFileChooser.getSelectedFile().getAbsolutePath();
	LogEntryWriter.rewrite(filename);
	int[] rows = m_TableEntries.getSelectedRows();
	LogEntryViewerTableModel model = (LogEntryViewerTableModel) m_TableEntries.getModel();
	if (rows.length == 0) {
	  for (int i = 0; i < m_TableEntries.getRowCount(); i++)
	    LogEntryWriter.write(filename, model.getLogEntryAt(i));
	}
	else {
	  for (int i = 0; i < rows.length; i++)
	    LogEntryWriter.write(filename, model.getLogEntryAt(rows[i]));
	}
      }
    });
    m_TableEntries.addToButtonsPanel(m_ButtonSave);

    m_TableEntries.setOptimalColumnWidth();

    m_BaseFileChooser = new BaseFileChooser();
    m_BaseFileChooser.addChoosableFileFilter(new ExtensionFileFilter("CSV files", "csv"));
    m_BaseFileChooser.addChoosableFileFilter(new ExtensionFileFilter("Log files", "log"));
    m_BaseFileChooser.setDefaultExtension("log");
    m_BaseFileChooser.setAutoAppendExtension(true);
    m_BaseFileChooser.setCurrentDirectory(new File("."));
    m_BaseFileChooser.setSelectedFile(new File(getName() + ".log"));

    return result;
  }

  /**
   * Displays the token (the panel and dialog have already been created at
   * this stage).
   *
   * @param token	the token to display
   */
  protected void display(Token token) {
    LogEntry	entry;

    entry = (LogEntry) token.getPayload();
    ((LogEntryViewerTableModel) m_TableEntries.getModel()).add(entry, true);
    m_TableEntries.setOptimalColumnWidth();
  }

  /**
   * Removes all graphical components.
   */
  protected void cleanUpGUI() {
    if (m_MessageDialog != null) {
      m_MessageDialog.setVisible(false);
      m_MessageDialog = null;
    }

    super.cleanUpGUI();
  }
}
