/*
 * ArrayElement.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import adams.core.Index;
import adams.flow.core.Token;
import adams.flow.core.Unknown;

/**
 <!-- globalinfo-start -->
 * Picks an element from an array and forwards only this element.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * <pre>   adams.flow.core.Unknown[]</pre>
 * - generates:<br/>
 * <pre>   adams.flow.core.Unknown</pre>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 *         If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 *         The name of the actor.
 *         default: ArrayElement
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 *         The annotations to attach to this actor.
 *         default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 *         If set to true, transformation is skipped and the input token is just forwarded
 *          as it is.
 * </pre>
 *
 * <pre>-index &lt;java.lang.String&gt; (property: index)
 *         The 1-based index of the element to pick ('first' and 'last' are accepted
 *          as well).
 *         default: 1
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2414 $
 */
public class ArrayElement
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -3483275959738804271L;

  /** the element index to pick. */
  protected String m_Index;

  /** for interpreting the index. */
  protected Index m_ActualIndex;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Picks an element from an array and forwards only this element.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "index", "index",
	    "1");
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_ActualIndex = new Index();
  }

  /**
   * Sets the index (1-based).
   *
   * @param value	the index
   */
  public void setIndex(String value) {
    m_Index = value;
    reset();
  }

  /**
   * Returns the index (1-based).
   *
   * @return		the index
   */
  public String getIndex() {
    return m_Index;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String indexTipText() {
    return "The 1-based index of the element to pick ('first' and 'last' are accepted as well).";
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("index");

    return "Index = " + ((variable == null) ? m_Index : variable);
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown[].class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Unknown[].class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.core.Unknown.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Unknown.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    Object[]	array;

    result = null;

    try {
      array = (Object[]) m_InputToken.getPayload();
      m_ActualIndex.setIndex(m_Index);
      m_ActualIndex.setMax(array.length);
      m_OutputToken = new Token(array[m_ActualIndex.getIntIndex()]);
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    return result;
  }
}
