/**
 * StatusBar.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.core;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;
import javax.swing.border.BevelBorder;

import adams.core.StatusMessageHandler;
import adams.gui.dialog.TextDialog;

/**
 * A bar for displaying a status message.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3224 $
 */
public class BaseStatusBar
  extends BasePanel
  implements StatusMessageHandler {

  /** for serialization. */
  private static final long serialVersionUID = 4700122818727471957L;

  /**
   * Interface for classes that process the status string to bring it into
   * a displayable format.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 3224 $
   */
  public static interface StatusProcessor {

    /**
     * Processes the status string.
     *
     * @param msg	the status string to process
     * @return		the processed string
     */
    public String process(String msg);
  }

  /** the empty status. */
  public final static String EMPTY_STATUS = " ";

  /** the default title. */
  public final static String DEFAULT_TITLE = "Status";

  /** a label for displaying a status. */
  protected JLabel m_LabelStatus;

  /** a mouse listener for displaying the message in a dialog. */
  protected MouseListener m_MouseListener;

  /** the title for the status dialog. */
  protected String m_StatusDialogTitle;

  /** for custom processing the status message before displaying it. */
  protected StatusProcessor m_StatusProcessor;

  /** the default dimension for displaying the status. */
  protected Dimension m_DialogSize;

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_MouseListener     = null;
    m_StatusDialogTitle = DEFAULT_TITLE;
    m_StatusProcessor   = null;
    m_DialogSize        = new Dimension(400, 300);
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    JPanel	panel;

    super.initGUI();

    setLayout(new BorderLayout());
    setBorder(BorderFactory.createBevelBorder(BevelBorder.LOWERED));
    panel         = new JPanel(new FlowLayout(FlowLayout.RIGHT));
    m_LabelStatus = new JLabel(EMPTY_STATUS);
    panel.add(m_LabelStatus, BorderLayout.WEST);
    add(panel, BorderLayout.WEST);
  }

  /**
   * Removes any status message currently being displayed.
   */
  protected void clearStatus() {
    m_LabelStatus.setText(EMPTY_STATUS);
  }

  /**
   * Returns whether a status is currently being displayed.
   *
   * @return		true if a status message is being displayed
   */
  public boolean hasStatus() {
    return !m_LabelStatus.getText().equals(EMPTY_STATUS);
  }

  /**
   * Displays a message.
   *
   * @param msg		the message to display
   */
  public void setStatus(final String msg) {
    Runnable	run;

    run = new Runnable() {
      public void run() {
	if ((msg == null) || (msg.length() == 0))
	  clearStatus();
	else
	  m_LabelStatus.setText(msg);
      }
    };
    SwingUtilities.invokeLater(run);
  }

  /**
   * Returns the currently displayed status.
   *
   * @return		the status, null if none displayed
   */
  public String getStatus() {
    String	result;

    result = null;

    if (!m_LabelStatus.getText().equals(EMPTY_STATUS))
      result = m_LabelStatus.getText();

    return result;
  }

  /**
   * Displays a message. Just an alias for setStatus(String)
   *
   * @param msg		the message to display
   * @see		#setStatus(String)
   */
  public void showStatus(String msg) {
    setStatus(msg);
  }

  /**
   * Displays the status in a dialog.
   */
  protected void displayStatus() {
    String 	status;
    Component 	parent;
    TextDialog	dialog;

    status = getStatus();
    if (m_StatusProcessor != null)
      status = m_StatusProcessor.process(status);

    if (getParentDialog() != null) {
      parent = getParentDialog();
      dialog = new TextDialog((Dialog) parent);
    }
    else {
      parent = getParentFrame();
      dialog = new TextDialog((Frame) parent);
    }
    dialog.setTitle(m_StatusDialogTitle);
    dialog.setContent(status);
    dialog.setEditable(false);
    dialog.setSize(m_DialogSize);
    dialog.setLocationRelativeTo(parent);
    dialog.setVisible(true);
  }

  /**
   * Adds the mouse listener to the status bar (the JLabel, actually).
   *
   * @param l		the listener to add
   */
  public void addMouseListener(MouseListener l) {
    m_LabelStatus.addMouseListener(l);
  }

  /**
   * Removes the mouse listener from the status bar (the JLabel, actually).
   *
   * @param l		the listener to removes
   */
  public void removeMouseListener(MouseListener l) {
    m_LabelStatus.removeMouseListener(l);
  }

  /**
   * Sets whether to turn mouse listener on.
   *
   * @param value	if true then the mouse listener is active
   */
  public void setMouseListenerActive(boolean value) {
    if (value != (m_MouseListener != null)) {
      if (value) {
	m_MouseListener = new MouseAdapter() {
	  public void mouseClicked(MouseEvent e) {
	    if (MouseUtils.isDoubleClick(e) && (m_LabelStatus.getText().length() > 0)) {
	      e.consume();
	      displayStatus();
	    }
	    else if (MouseUtils.isRightClick(e)) {
	      e.consume();
	      JPopupMenu menu = getPopup();
	      menu.show(m_LabelStatus, e.getX(), e.getY());
	    }
	    else {
	      super.mouseClicked(e);
	    }
	  }
	};
	addMouseListener(m_MouseListener);
      }
      else {
	removeMouseListener(m_MouseListener);
      }
    }
  }

  /**
   * Returns whether the mouse listener is active.
   *
   * @return		true if the mouse listener is active
   */
  public boolean isMouseListenerActive() {
    return (m_MouseListener != null);
  }

  /**
   * Returns the popup menu for the status.
   *
   * @return		the popup menu
   */
  protected JPopupMenu getPopup() {
    JPopupMenu	result;
    JMenuItem	menuitem;

    result = new JPopupMenu();

    menuitem = new JMenuItem("Show status");
    menuitem.setEnabled(m_LabelStatus.getText().length() > 0);
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	displayStatus();
      }
    });
    result.add(menuitem);

    result.addSeparator();

    menuitem = new JMenuItem("Clear status");
    menuitem.setEnabled(m_LabelStatus.getText().length() > 0);
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	clearStatus();
      }
    });
    result.add(menuitem);

    return result;
  }

  /**
   * Sets the title to use for the dialog when displaying the status bar in a
   * dialog.
   *
   * @param value	the title, use null for default
   */
  public void setStatusDialogTitle(String value) {
    if (value == null)
      m_StatusDialogTitle = DEFAULT_TITLE;
    else
      m_StatusDialogTitle = value;
  }

  /**
   * Returns the currently set title for the status dialog.
   *
   * @return		the title
   */
  public String getStatusDialogTitle() {
    return m_StatusDialogTitle;
  }

  /**
   * Sets the code for processing the status message before displaying it.
   *
   * @param value	the processor to use, null to turn off
   */
  public void setStatusProcessor(StatusProcessor value) {
    m_StatusProcessor = value;
  }

  /**
   * Returns the code for processing the status message before displaying it.
   *
   * @return		the processor in use, null if none set
   */
  public StatusProcessor getStatusProcessor() {
    return m_StatusProcessor;
  }

  /**
   * Sets the size for the dialog.
   *
   * @param value	the size to use
   */
  public void setDialogSize(Dimension value) {
    m_DialogSize = new Dimension(value);
  }

  /**
   * Returns the size for the dialog.
   *
   * @return		the size in use
   */
  public Dimension getDialogSize() {
    return m_DialogSize;
  }
}
