/**
 * ImportDialog.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.flow;

import java.awt.BorderLayout;
import java.awt.Dialog;
import java.awt.Frame;
import java.io.File;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import adams.core.option.AbstractOptionConsumer;
import adams.core.option.FileBasedConsumer;
import adams.core.option.NestedConsumer;
import adams.gui.chooser.FileChooserPanel;
import adams.gui.core.ExtensionFileFilter;
import adams.gui.core.ParameterPanel;
import adams.gui.dialog.AbstractApprovalDialog;
import adams.gui.goe.GenericObjectEditorPanel;

/**
 * A dialog for importing a flow from various formats.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4095 $
 */
public class ImportDialog
  extends AbstractApprovalDialog {

  /** for serialization. */
  private static final long serialVersionUID = 201725070566669323L;

  /** the panel with the file. */
  protected FileChooserPanel m_PanelFile;

  /** the GOE panel for the import format. */
  protected GenericObjectEditorPanel m_PanelGOE;

  /**
   * Creates a modeless dialog without a title with the specified Dialog as
   * its owner.
   *
   * @param owner	the owning dialog
   */
  public ImportDialog(Dialog owner) {
    super(owner);
  }

  /**
   * Creates a dialog with the specified owner Dialog and modality.
   *
   * @param owner	the owning dialog
   * @param modality	the type of modality
   */
  public ImportDialog(Dialog owner, ModalityType modality) {
    super(owner, modality);
  }

  /**
   * Creates a modeless dialog without a title with the specified Frame as
   * its owner.
   *
   * @param owner	the owning frame
   */
  public ImportDialog(Frame owner) {
    super(owner);
  }

  /**
   * Creates a dialog with the specified owner Frame, modality and an empty
   * title.
   *
   * @param owner	the owning frame
   * @param modal	whether the dialog is modal or not
   */
  public ImportDialog(Frame owner, boolean modal) {
    super(owner, modal);
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    ParameterPanel	params;

    super.initGUI();

    setTitle("Import flow");

    params = new ParameterPanel();
    getContentPane().add(params, BorderLayout.CENTER);

    m_PanelFile = new FileChooserPanel();
    m_PanelFile.setPromptOverwriteFile(true);
    params.addParameter("_File", m_PanelFile);

    m_PanelGOE = new GenericObjectEditorPanel(AbstractOptionConsumer.class, new NestedConsumer(), true);
    m_PanelGOE.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
	updateFileChooser();
      }
    });
    params.addParameter("For_mat", m_PanelGOE);

    pack();
  }

  /**
   * Updates the file chooser's extension file filters.
   */
  protected void updateFileChooser() {
    FileBasedConsumer	producer;
    ExtensionFileFilter	filter;

    m_PanelFile.setDefaultExtension(null);
    m_PanelFile.removeChoosableFileFilters();
    m_PanelFile.setAcceptAllFileFilterUsed(true);
    m_PanelFile.setAutoAppendExtension(false);

    if (m_PanelGOE.getCurrent() instanceof FileBasedConsumer) {
      producer = (FileBasedConsumer) m_PanelGOE.getCurrent();
      filter   = new ExtensionFileFilter(producer.getFileFormat(), producer.getFileExtensions());
      m_PanelFile.setDefaultExtension(producer.getDefaultFileExtension());
      m_PanelFile.setAutoAppendExtension(true);
      m_PanelFile.addChoosableFileFilter(filter);
      m_PanelFile.setFileFilter(filter);
    }
  }

  /**
   * Sets the producer to generate the output format.
   *
   * @param value	the producer to use
   */
  public void setConsumer(AbstractOptionConsumer value) {
    if (value == null)
      value = new NestedConsumer();
    m_PanelGOE.setCurrent(value);
  }

  /**
   * Returns the producer currently in use.
   *
   * @return		the current producer
   */
  public AbstractOptionConsumer getConsumer() {
    return (AbstractOptionConsumer) m_PanelGOE.getCurrent();
  }

  /**
   * Sets the file to save the options to.
   *
   * @param value	the file
   */
  public void setFile(File value) {
    m_PanelFile.setCurrent(value);
  }

  /**
   * Returns the current file to save the options to.
   *
   * @return		the file
   */
  public File getFile() {
    return m_PanelFile.getCurrent();
  }
}
