/**
 * VariableSupport.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.goe;

import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyEditor;

import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;

import adams.core.Variables;
import adams.core.option.AbstractArgumentOption;
import adams.core.option.AbstractOption;
import adams.gui.core.GUIHelper;

/**
 * Helper class for managing variables in the GOE.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3663 $
 */
public class VariableSupport {

  /** the indicator to be displayed in a JLabel when the option has a variable. */
  public final static String CAPTION_INDICATOR = "*";

  /** the indicator to be displayed in a JLabel's tiptext when the option has a variable. */
  public final static String HINT_INDICATOR = ", variable: ";

  /**
   * Returns the PropertySheetPanel parent of this container.
   *
   * @param c		the container to get the enclosing PropertySheetPanel for
   * @return		the parent sheet or null if none found
   */
  public static PropertySheetPanel findParent(Container c) {
    return (PropertySheetPanel) GUIHelper.getParent(c, PropertySheetPanel.class);
  }

  /**
   * Checks whether the editor's option has a variable defined.
   *
   * @param parent	the parent sheet panel of the editor
   * @param editor	the property editor this menu is generated for
   * @return		the popup menu
   */
  public static boolean hasVariable(PropertySheetPanel parent, PropertyEditor editor) {
    if (parent != null)
      return (parent.findOption(editor) != null);
    else
      return false;
  }

  /**
   * Creates a popup menu for variable management.
   *
   * @param parent	the parent sheet panel of the editor
   * @param editor	the property editor this menu is generated for
   * @return		the popup menu
   */
  public static JPopupMenu createPopup(PropertySheetPanel parent, PropertyEditor editor) {
    JPopupMenu		result;
    AbstractOption	option;

    result = null;

    if (parent != null) {
      option = parent.findOption(editor);
      if (option != null)
	result = updatePopup(parent, editor, new JPopupMenu());
    }

    return result;
  }

  /**
   * Updates a popup menu and adds menu items for variable management.
   *
   * @param parent	the parent sheet panel of the editor
   * @param editor	the property editor this menu is generated for
   * @param menu	the menu to update
   * @return		the updated popup menu
   */
  public static JPopupMenu updatePopup(final PropertySheetPanel parent, PropertyEditor editor, JPopupMenu menu) {
    JPopupMenu			result;
    JMenuItem			item;
    AbstractOption 		option;
    JLabel			label;
    AbstractArgumentOption	argoption;

    result = menu;
    if (parent == null)
      return result;

    option = parent.findOption(editor);
    label  = parent.findLabel(editor);

    if (option instanceof AbstractArgumentOption) {
      argoption = (AbstractArgumentOption) option;

      if (!argoption.isMultiple()) {
	final AbstractArgumentOption fArgOption = argoption;
	final JLabel fLabel = label;
	final PropertySheetPanel fParent = parent;

	if (result.getComponentCount() > 0)
	  result.addSeparator();

	if (!argoption.isVariableAttached()) {
	  item = new JMenuItem("Set variable...", GUIHelper.getIcon("variable.gif"));
	  item.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
	      String name = JOptionPane.showInputDialog(fParent, "Please enter the variable name:");
	      if (name == null)
		return;
	      if (!Variables.isValidName(name)) {
		GUIHelper.showErrorMessage(
		    fParent,
		    "Not a valid variable name: " + name + "\n"
		    + "Allowed characters:\n" + Variables.CHARS);
		return;
	      }
	      fArgOption.setVariable(name);
	      updateLabel(fLabel, fArgOption.getVariableName());
	      parent.getPropertyChangeSupport().firePropertyChange("", null, null);
	    }
	  });
	  result.add(item);
	}
	else {
	  item = new JMenuItem("Change variable '" + fArgOption.getVariableName() + "'...", GUIHelper.getIcon("variable.gif"));
	  item.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
	      String oldName = fArgOption.getVariableName();
	      String name = JOptionPane.showInputDialog(fParent, "Please enter the new variable name:", oldName);
	      if (name == null)
		return;
	      if (!Variables.isValidName(name)) {
		GUIHelper.showErrorMessage(
		    fParent,
		    "Not a valid variable name: " + name + "\n"
		    + "Allowed characters:\n" + Variables.CHARS);
		return;
	      }
	      fArgOption.setVariable(name);
	      updateLabel(fLabel, fArgOption.getVariableName());
	      parent.getPropertyChangeSupport().firePropertyChange("", null, null);
	    }
	  });
	  result.add(item);

	  item = new JMenuItem("Remove variable '" + argoption.getVariableName() + "'", GUIHelper.getIcon("delete.gif"));
	  item.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
	      fArgOption.setVariable(null);
	      updateLabel(fLabel, null);
	      parent.getPropertyChangeSupport().firePropertyChange("", null, null);
	    }
	  });
	  result.add(item);
	}
      }
    }

    return result;
  }

  /**
   * Updates the label's caption and hint depending on whether the option has
   * a variable or not.
   *
   * @param label	the label to process
   * @param variable	the variable name, null if none present
   */
  public static void updateLabel(JLabel label, String variable) {
    boolean	hintPresent;
    String	text;
    boolean	hasVariable;

    hasVariable = (variable != null);

    // caption
    text        = label.getText();
    hintPresent = text.endsWith(CAPTION_INDICATOR);
    if (hintPresent)
      text = text.substring(0, text.length() - CAPTION_INDICATOR.length());
    if (hasVariable)
      text = text + CAPTION_INDICATOR;
    label.setText(text);

    // hint
    text        = label.getToolTipText();
    hintPresent = (text.indexOf(HINT_INDICATOR) > -1);
    if (hintPresent)
      text = text.substring(0, text.indexOf(HINT_INDICATOR));
    if (hasVariable)
      text = text + HINT_INDICATOR + variable;
    label.setToolTipText(text);
  }
}
