/*
 * ClassLocator.java
 * Copyright (C) 2005 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.core;

import java.lang.reflect.Modifier;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * This class is used for discovering classes that implement a certain
 * interface or a derived from a certain class. Based on the
 * <code>weka.core.ClassDiscovery</code> class.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3726 $
 * @see StringCompare
 * @see weka.core.ClassDiscovery
 */
public class ClassLocator {

  /** whether to output some debug information. */
  public final static boolean VERBOSE = false;

  /** for caching queries (classname-packagename &lt;-&gt; Vector with classnames). */
  protected static Hashtable<String,Vector<String>> m_Cache;

  /** the overall class cache. */
  protected static ClassCache m_ClassCache;

  /** notify if VERBOSE is still on */
  static {
    if (VERBOSE)
      System.err.println(ClassLocator.class.getName() + ": VERBOSE ON");
  }

  /**
   * Checks whether the "otherclass" is a subclass of the given "superclass".
   *
   * @param superclass      the superclass to check against
   * @param otherclass      this class is checked whether it is a subclass
   *                        of the the superclass
   * @return                TRUE if "otherclass" is a true subclass
   */
  public static boolean isSubclass(String superclass, String otherclass) {
    try {
      return isSubclass(Class.forName(superclass), Class.forName(otherclass));
    }
    catch (Throwable t) {
      return false;
    }
  }

  /**
   * Checks whether the "otherclass" is a subclass of the given "superclass".
   *
   * @param superclass      the superclass to check against
   * @param otherclass      this class is checked whether it is a subclass
   *                        of the the superclass
   * @return                TRUE if "otherclass" is a true subclass
   */
  public static boolean isSubclass(Class superclass, Class otherclass) {
    Class       currentclass;
    boolean     result;

    result       = false;
    currentclass = otherclass;
    do {
      result = currentclass.equals(superclass);

      // topmost class reached?
      if (currentclass.equals(Object.class) || (currentclass.getSuperclass() == null))
        break;

      if (!result)
        currentclass = currentclass.getSuperclass();
    }
    while (!result);

    return result;
  }

  /**
   * Checks whether the given class implements the given interface.
   *
   * @param intf      the interface to look for in the given class
   * @param cls       the class to check for the interface
   * @return          TRUE if the class contains the interface
   */
  public static boolean hasInterface(String intf, String cls) {
    try {
      return hasInterface(Class.forName(intf), Class.forName(cls));
    }
    catch (Throwable t) {
      return false;
    }
  }

  /**
   * Checks whether the given class implements the given interface.
   *
   * @param intf      the interface to look for in the given class
   * @param cls       the class to check for the interface
   * @return          TRUE if the class contains the interface
   */
  public static boolean hasInterface(Class intf, Class cls) {
    return intf.isAssignableFrom(cls);
  }

  /**
   * Checks the given packages for classes that inherited from the given class,
   * in case it's a class, or implement this class, in case it's an interface.
   *
   * @param classname       the class/interface to look for
   * @param pkgnames        the packages to search in
   * @return                a list with all the found classnames
   */
  public static Vector<String> find(String classname, String[] pkgnames) {
    Vector<String>	result;
    Class		cls;

    result = new Vector<String>();

    try {
      cls    = Class.forName(classname);
      result = find(cls, pkgnames);
    }
    catch (Throwable t) {
      System.err.println("Failed to instantiate '" + classname + "'/" + Utils.arrayToString(pkgnames) + " (find):");
      t.printStackTrace();
    }

    return result;
  }

  /**
   * Checks the given packages for classes that inherited from the given class,
   * in case it's a class, or implement this class, in case it's an interface.
   *
   * @param cls             the class/interface to look for
   * @param pkgnames        the packages to search in
   * @return                a list with all the found classnames
   */
  public static Vector<String> find(Class cls, String[] pkgnames) {
    Vector<String>	result;
    int			i;
    HashSet<String>	names;

    result = new Vector<String>();

    names = new HashSet<String>();
    for (i = 0; i < pkgnames.length; i++)
      names.addAll(findInPackage(cls, pkgnames[i]));

    // sort result
    result.addAll(names);
    Collections.sort(result, new StringCompare());

    return result;
  }

  /**
   * Checks the given package for classes that inherited from the given class,
   * in case it's a class, or implement this class, in case it's an interface.
   *
   * @param classname       the class/interface to look for
   * @param pkgname         the package to search in
   * @return                a list with all the found classnames
   */
  public static Vector<String> findInPackage(String classname, String pkgname) {
    Vector<String>	result;
    Class		cls;

    result = new Vector<String>();

    try {
      cls    = Class.forName(classname);
      result = findInPackage(cls, pkgname);
    }
    catch (Throwable t) {
      System.err.println("Failed to instantiate '" + classname + "'/" + pkgname + " (findInPackage):");
      t.printStackTrace();
    }

    return result;
  }

  /**
   * Checks the given package for classes that inherited from the given class,
   * in case it's a class, or implement this class, in case it's an interface.
   *
   * @param cls             the class/interface to look for
   * @param pkgname         the package to search in
   * @return                a list with all the found classnames
   */
  public static Vector<String> findInPackage(Class cls, String pkgname) {
    Vector<String>	result;
    int			i;
    Class		clsNew;

    // already cached?
    result = getCache(cls, pkgname);

    if (result == null) {
      if (VERBOSE)
	System.out.println(
	    "Searching for '" + cls.getName() + "' in '" + pkgname + "':");

      result = new Vector<String>();
      if (m_ClassCache.getClassnames(pkgname) != null)
	result.addAll(m_ClassCache.getClassnames(pkgname));

      // check classes
      i = 0;
      while (i < result.size()) {
	try {
	  // no inner classes
	  if (result.get(i).indexOf('$') > -1) {
	    result.remove(i);
	    continue;
	  }

	  clsNew = Class.forName((String) result.get(i));
	  // no abstract classes
	  if (Modifier.isAbstract(clsNew.getModifiers())) {
	    m_ClassCache.remove(result.get(i));
	    result.remove(i);
	  }
	  // must implement interface
	  else if ( (cls.isInterface()) && (!hasInterface(cls, clsNew)) ) {
	    result.remove(i);
	  }
	  // must be derived from class
	  else if ( (!cls.isInterface()) && (!isSubclass(cls, clsNew)) ) {
	    result.remove(i);
	  }
	  else {
	    i++;
	  }
	}
	catch (Throwable t) {
	  System.err.println("Failed to instantiate '" + result.get(i) + "' (find):");
	  t.printStackTrace();
	}
      }

      // sort result
      Collections.sort(result, new StringCompare());

      // add to cache
      addCache(cls, pkgname, result);
    }

    return result;
  }

  /**
   * Lists all packages it can find in the classpath.
   *
   * @return                a list with all the found packages
   */
  public static Vector<String> findPackages() {
    Vector<String>	result;
    Enumeration<String>	packages;

    initCache();

    result   = new Vector<String>();
    packages = m_ClassCache.packages();
    while (packages.hasMoreElements())
      result.add(packages.nextElement());
    Collections.sort(result, new StringCompare());

    return result;
  }

  /**
   * initializes the cache for the classnames.
   */
  protected static void initCache() {
    if (m_Cache == null)
      m_Cache = new Hashtable<String,Vector<String>>();
    if (m_ClassCache == null)
      m_ClassCache = new ClassCache();
  }

  /**
   * adds the list of classnames to the cache.
   *
   * @param cls		the class to cache the classnames for
   * @param pkgname	the package name the classes were found in
   * @param classnames	the list of classnames to cache
   */
  protected static void addCache(Class cls, String pkgname, Vector<String> classnames) {
    initCache();
    m_Cache.put(cls.getName() + "-" + pkgname, classnames);
  }

  /**
   * returns the list of classnames associated with this class and package, if
   * available, otherwise null.
   *
   * @param cls		the class to get the classnames for
   * @param pkgname	the package name for the classes
   * @return		the classnames if found, otherwise null
   */
  protected static Vector<String> getCache(Class cls, String pkgname) {
    initCache();
    return m_Cache.get(cls.getName() + "-" + pkgname);
  }

  /**
   * Possible calls:
   * <ul>
   *    <li>
   *      weka.core.ClassLocator &lt;packages&gt;<br/>
   *      Prints all the packages in the current classpath
   *    </li>
   *    <li>
   *      weka.core.ClassLocator &lt;classname&gt; &lt;packagename(s)&gt;<br/>
   *      Prints the classes it found.
   *    </li>
   * </ul>
   *
   * @param args	the commandline arguments
   */
  public static void main(String[] args) {
    Vector<String>	list;
    Vector<String>	packages;
    int         	i;
    StringTokenizer	tok;

    if ((args.length == 1) && (args[0].equals("packages"))) {
      list = findPackages();
      for (i = 0; i < list.size(); i++)
	System.out.println(list.get(i));
    }
    else if (args.length == 2) {
      // packages
      packages = new Vector<String>();
      tok = new StringTokenizer(args[1], ",");
      while (tok.hasMoreTokens())
        packages.add(tok.nextToken());

      // search
      list = ClassLocator.find(
  		args[0],
  		packages.toArray(new String[packages.size()]));

      // print result, if any
      System.out.println(
          "Searching for '" + args[0] + "' in '" + args[1] + "':\n"
          + "  " + list.size() + " found.");
      for (i = 0; i < list.size(); i++)
        System.out.println("  " + (i+1) + ". " + list.get(i));
    }
    else {
      System.out.println("\nUsage:");
      System.out.println(
	  ClassLocator.class.getName() + " packages");
      System.out.println("\tlists all packages in the classpath");
      System.out.println(
	  ClassLocator.class.getName() + " <classname> <packagename(s)>");
      System.out.println("\tlists classes derived from/implementing 'classname' that");
      System.out.println("\tcan be found in 'packagename(s)' (comma-separated list)");
      System.out.println();
      System.exit(1);
    }
  }
}
