/*
 * Performance.java
 * Copyright (C) 2008 University of Waikato, Hamilton, New Zealand
 */

package adams.core;

import adams.env.Environment;
import adams.env.PerformanceDefinition;


/**
 * A convenience class for accessing the performance tuning parameters.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4237 $
 */
public class Performance {

  /** the name of the props file. */
  public final static String FILENAME = "Performance.props";

  /** the properties. */
  protected static Properties m_Properties;

  /**
   * loads the props file.
   */
  protected synchronized static void initialize() {
    if (m_Properties == null) {
      m_Properties = Environment.getInstance().read(PerformanceDefinition.KEY);
    }
  }

  /**
   * Returns the specified boolean property.
   *
   * @param key		the key of the property to retrieve
   * @param defValue	the default value
   */
  public static boolean getBoolean(String key, boolean defValue) {
    return m_Properties.getBoolean(key, defValue);
  }

  /**
   * Returns the specified integer property.
   *
   * @param key		the key of the property to retrieve
   * @param defValue	the default value
   */
  public static int getInteger(String key, int defValue) {
    return m_Properties.getInteger(key, defValue);
  }

  /**
   * Returns the specified double property.
   *
   * @param key		the key of the property to retrieve
   * @param defValue	the default value
   */
  public static double getDouble(String key, double defValue) {
    return m_Properties.getDouble(key, defValue);
  }

  /**
   * Returns the maximum number of processors to use, -1 is all available.
   *
   * @return		the maximum number
   */
  public static int getMaxNumProcessors() {
    initialize();

    return m_Properties.getInteger("maxNumProcessors", -1);
  }

  /**
   * Returns the minimum number of JobComplete events in the JobCompleteManager.
   *
   * @return		the minimum number
   */
  public static int getMinKeepJobComplete() {
    initialize();

    return m_Properties.getInteger("minKeepJobComplete", 50);
  }

  /**
   * Returns the maximum number of JobComplete events in the JobCompleteManager.
   *
   * @return		the maximum number
   */
  public static int getMaxKeepJobComplete() {
    initialize();

    return m_Properties.getInteger("maxKeepJobComplete", 100);
  }

  /**
   * Returns whether only events of failed jobs are kept or all.
   *
   * @return		true if only events of failed jobs are kept
   */
  public static boolean getKeepOnlyFailedJobComplete() {
    initialize();

    return m_Properties.getBoolean("keepOnlyFailedJobComplete", true);
  }

  /**
   * Returns whether multiprocessing is enabled.
   *
   * @return		true if multiprocessing enabled
   */
  public static boolean getMultiProcessingEnabled() {
    initialize();

    return m_Properties.getBoolean("multiProcessingEnabled", true);
  }
}
