/*
 * Breakpoint.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.HashMap;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import adams.core.CleanUpHandler;
import adams.core.Utils;
import adams.core.Variables;
import adams.core.base.BaseString;
import adams.core.option.DebugNestedProducer;
import adams.flow.core.ActorUtils;
import adams.flow.core.ControlActor;
import adams.flow.core.Token;
import adams.flow.core.Unknown;
import adams.flow.transformer.AbstractTransformer;
import adams.gui.core.BaseDialog;
import adams.gui.core.BaseFrame;
import adams.gui.core.BasePanel;
import adams.gui.core.GUIHelper;
import adams.gui.core.MenuBarProvider;
import adams.gui.dialog.TextDialog;
import adams.gui.flow.StoragePanel;
import adams.gui.tools.ExpressionWatchPanel;
import adams.gui.tools.VariableManagementPanel;
import adams.gui.tools.ExpressionWatchPanel.ExpressionType;
import adams.gui.visualization.debug.InspectionPanel;
import adams.parser.BooleanExpression;

/**
 <!-- globalinfo-start -->
 * Allows to pause the execution of the flow when this actor is reached and the condition evaluates to 'true'.<br/>
 * It is possible to define watches as well.<br/>
 * <br/>
 * The expression has the following underlying grammar:<br/>
 *  expr_list ::= expr_list expr_part | expr_part ;<br/>
 *  expr_part ::= boolexpr ;<br/>
 * <br/>
 *  boolexpr ::=    ( boolexpr )<br/>
 *                | true<br/>
 *                | false<br/>
 *                | numexpr &lt; numexpr<br/>
 *                | numexpr &lt;= numexpr<br/>
 *                | numexpr &gt; numexpr<br/>
 *                | numexpr &gt;= numexpr<br/>
 *                | numexpr = numexpr<br/>
 *                | numexpr != numexpr<br/>
 *                | str &lt; str<br/>
 *                | str &lt;= str<br/>
 *                | str &gt; str<br/>
 *                | str &gt;= str<br/>
 *                | str = str<br/>
 *                | str != str<br/>
 *                | ! boolexpr<br/>
 *                | boolexpr &amp; boolexpr<br/>
 *                | boolexpr | boolexpr<br/>
 *                ;<br/>
 * <br/>
 *  numexpr   ::=   ( numexpr )<br/>
 *                | NUMBER<br/>
 *                | -numexpr<br/>
 *                | constexpr<br/>
 *                | opexpr<br/>
 *                | varexpr<br/>
 *                | funcexpr<br/>
 *                ;<br/>
 * <br/>
 *  constexpr ::=   PI<br/>
 *                | E<br/>
 *                ;<br/>
 * <br/>
 *  opexpr    ::=   numexpr + numexpr<br/>
 *                | numexpr - numexpr<br/>
 *                | numexpr * numexpr<br/>
 *                | numexpr &#47; numexpr<br/>
 *                | numexpr ^ numexpr<br/>
 *                ;<br/>
 * <br/>
 *  varexpr  ::=  VARIABLE ;<br/>
 * <br/>
 *  funcexpr ::=    abs ( numexpr )<br/>
 *                | sqrt ( numexpr )<br/>
 *                | log ( numexpr )<br/>
 *                | exp ( numexpr )<br/>
 *                | sin ( numexpr )<br/>
 *                | cos ( numexpr )<br/>
 *                | tan ( numexpr )<br/>
 *                | rint ( numexpr )<br/>
 *                | floor ( numexpr )<br/>
 *                | pow ( numexpr , numexpr )<br/>
 *                | ceil ( numexpr )<br/>
 *                | ifelse ( boolexpr , numexpr (if true) , numexpr (if false) )<br/>
 *                | length ( str )<br/>
 *                ;<br/>
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Breakpoint
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-x &lt;int&gt; (property: x)
 * &nbsp;&nbsp;&nbsp;The X position of the dialog (&gt;=0: absolute, -1: left, -2: center, -3: right
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -3
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 *
 * <pre>-y &lt;int&gt; (property: y)
 * &nbsp;&nbsp;&nbsp;The Y position of the dialog (&gt;=0: absolute, -1: top, -2: center, -3: bottom
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 *
 * <pre>-expression &lt;java.lang.String&gt; (property: expression)
 * &nbsp;&nbsp;&nbsp;The expression to evaluate; if the expression evaluates to 'true', the execution
 * &nbsp;&nbsp;&nbsp;of the flow is paused and a control window is opened.
 * &nbsp;&nbsp;&nbsp;default: true
 * </pre>
 *
 * <pre>-watch &lt;adams.core.base.BaseString&gt; [-watch ...] (property: watches)
 * &nbsp;&nbsp;&nbsp;The expression to display initially in the watch dialog; the type of the
 * &nbsp;&nbsp;&nbsp;watch needs to be specified as well.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-watch-type &lt;VARIABLE|BOOLEAN|NUMERIC|STRING&gt; [-watch-type ...] (property: watchTypes)
 * &nbsp;&nbsp;&nbsp;The types of the watch expressions; determines how the expressions get evaluated
 * &nbsp;&nbsp;&nbsp;and displayed.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-dialog &lt;SOURCE|EXPRESSIONS|VARIABLES|STORAGE|INSPECT_TOKEN|INSPECT_FLOW&gt; [-dialog ...] (property: dialogs)
 * &nbsp;&nbsp;&nbsp;The dialogs to display automatically when the breakpoint is reached.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3239 $
 */
public class Breakpoint
  extends AbstractTransformer
  implements ControlActor {

  /** for serialization. */
  private static final long serialVersionUID = 1670185555433805533L;

  /**
   * A little dialog for controlling the breakpoint. Cannot be static class!
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 3239 $
   */
  public class ControlPanel
    extends BasePanel
    implements CleanUpHandler {

    /** for serialization. */
    private static final long serialVersionUID = 1000900663466801934L;

    /** the button for continuing execution. */
    protected JButton m_ButtonContinue;

    /** the button for stopping execution. */
    protected JButton m_ButtonStop;

    /** the button for disabling/enabling the breakpoint. */
    protected JButton m_ButtonDisableEnable;

    /** the button for displaying dialog with watch expressions. */
    protected JButton m_ButtonExpressions;

    /** the dialog for watching expressions. */
    protected BaseDialog m_DialogExpressions;

    /** the breakpoint condition. */
    protected JTextField m_TextBreakpointExpression;

    /** the button for updating the breakpoing condition. */
    protected JButton m_ButtonBreakpointExpressionUpdate;

    /** the button to bring up the source code of the current flow. */
    protected JButton m_ButtonSource;

    /** the button to bring up the variable management dialog. */
    protected JButton m_ButtonVariables;

    /** the button to bring up the storage dialog. */
    protected JButton m_ButtonStorage;

    /** the button to bring up the dialog for inspecting the current token. */
    protected JButton m_ButtonInspectToken;

    /** the button to bring up the dialog for inspecting the flow. */
    protected JButton m_ButtonInspectFlow;

    /** the panel with the watch expressions. */
    protected ExpressionWatchPanel m_PanelExpressions;

    /** the panel with the variables. */
    protected VariableManagementPanel m_PanelVariables;

    /** the panel for inspecting the current token. */
    protected InspectionPanel m_PanelInspectionToken;

    /** the panel for inspecting the current flow. */
    protected InspectionPanel m_PanelInspectionFlow;

    /** the panel for displaying the temporary storage. */
    protected StoragePanel m_PanelStorage;

    /** the dialog for the source. */
    protected TextDialog m_DialogSource;

    /**
     * Initializes the widgets.
     */
    protected void initGUI() {
      JPanel	panelCondition;
      JPanel	panelButtons;
      JPanel	panelAllButtons;
      JPanel	panel;

      super.initGUI();

      setLayout(new BorderLayout());

      // buttons
      // 1. execution
      panelAllButtons = new JPanel(new BorderLayout());
      add(panelAllButtons, BorderLayout.CENTER);
      panel = new JPanel(new BorderLayout());
      panel.setBorder(BorderFactory.createTitledBorder("Execution"));
      panelAllButtons.add(panel, BorderLayout.NORTH);
      panelButtons = new JPanel(new GridLayout(2, 2));
      panel.add(panelButtons, BorderLayout.NORTH);

      m_ButtonContinue = new JButton("Continue", GUIHelper.getIcon("run.gif"));
      m_ButtonContinue.setMnemonic('C');
      m_ButtonContinue.setToolTipText("Continues with the flow execution");
      m_ButtonContinue.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          continueFlowExecution();
        }
      });
      panelButtons.add(m_ButtonContinue);

      m_ButtonStop = new JButton("Stop", GUIHelper.getIcon("stop.gif"));
      m_ButtonStop.setMnemonic('S');
      m_ButtonStop.setToolTipText("Stops the flow execution immediately");
      m_ButtonStop.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  stopFlowExecution();
	}
      });
      panelButtons.add(m_ButtonStop);

      m_ButtonDisableEnable = new JButton("Disable", GUIHelper.getIcon("debug_off.png"));
      m_ButtonDisableEnable.setMnemonic('D');
      m_ButtonDisableEnable.setToolTipText("Disable the breakpoint");
      m_ButtonDisableEnable.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  disableEnableBreakpoint();
	}
      });
      panelButtons.add(m_ButtonDisableEnable);

      // condition
      panelCondition = new JPanel(new FlowLayout(FlowLayout.LEFT));
      panel.add(panelCondition, BorderLayout.CENTER);

      m_TextBreakpointExpression = new JTextField(10);
      m_TextBreakpointExpression.getDocument().addDocumentListener(new DocumentListener() {
        public void removeUpdate(DocumentEvent e) {
          update();
        }
        public void insertUpdate(DocumentEvent e) {
          update();
        }
        public void changedUpdate(DocumentEvent e) {
          update();
        }
      });
      m_ButtonBreakpointExpressionUpdate = new JButton("Update");
      m_ButtonBreakpointExpressionUpdate.setToolTipText("Update the breakpoint condition");
      m_ButtonBreakpointExpressionUpdate.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          updateExpression();
        }
      });
      panelCondition.add(new JLabel("Condition"));
      panelCondition.add(m_TextBreakpointExpression);
      panelCondition.add(m_ButtonBreakpointExpressionUpdate);

      // 2. runtime information
      panelButtons = new JPanel(new GridLayout(3, 2));
      panelButtons.setBorder(BorderFactory.createTitledBorder("Runtime information"));
      panelAllButtons.add(panelButtons, BorderLayout.CENTER);

      m_ButtonSource = new JButton("Source", GUIHelper.getIcon("source.png"));
      m_ButtonSource.setMnemonic('o');
      m_ButtonSource.setToolTipText("Display current flow state as source (nested format)");
      m_ButtonSource.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          showSource();
        }
      });
      panelButtons.add(m_ButtonSource);

      m_ButtonExpressions = new JButton("Expressions", GUIHelper.getIcon("glasses.gif"));
      m_ButtonExpressions.setMnemonic('x');
      m_ButtonExpressions.setToolTipText("Display dialog for watch expressions");
      m_ButtonExpressions.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  showWatchExpressions();
        }
      });
      panelButtons.add(m_ButtonExpressions);

      m_ButtonVariables = new JButton("Variables", GUIHelper.getIcon("variable.gif"));
      m_ButtonVariables.setMnemonic('V');
      m_ButtonVariables.setToolTipText("Display dialog with currently active variables");
      m_ButtonVariables.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          showVariables();
        }
      });
      panelButtons.add(m_ButtonVariables);

      m_ButtonStorage = new JButton("Storage", GUIHelper.getIcon("disk.png"));
      m_ButtonStorage.setMnemonic('t');
      m_ButtonStorage.setToolTipText("Display dialog with items currently stored in temporary storage");
      m_ButtonStorage.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          showStorage();
        }
      });
      panelButtons.add(m_ButtonStorage);

      m_ButtonInspectToken = new JButton("Inspect token", GUIHelper.getIcon("properties.gif"));
      m_ButtonInspectToken.setMnemonic('I');
      m_ButtonInspectToken.setToolTipText("Display dialog for inspecting the current token");
      m_ButtonInspectToken.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          inspectToken();
        }
      });
      panelButtons.add(m_ButtonInspectToken);

      m_ButtonInspectFlow = new JButton("Inspect flow", GUIHelper.getIcon("flow.gif"));
      m_ButtonInspectFlow.setMnemonic('n');
      m_ButtonInspectFlow.setToolTipText("Display dialog for inspecting the current flow");
      m_ButtonInspectFlow.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          inspectFlow();
        }
      });
      panelButtons.add(m_ButtonInspectFlow);

      // watches
      m_PanelExpressions = new ExpressionWatchPanel();
    }

    /**
     * Returns the underlying flow.
     *
     * @return		the flow
     */
    protected Flow getFlow() {
      return (Flow) getRoot();
    }

    /**
     * Updates the enabled status of the buttons, text fields.
     */
    protected void update() {
      Flow	flow;

      flow = getFlow();

      m_ButtonContinue.setEnabled(flow.isPaused());
      m_ButtonStop.setEnabled(!flow.isStopped());
      m_ButtonDisableEnable.setEnabled(!flow.isStopped());
      m_ButtonExpressions.setEnabled(flow.isPaused());
      m_ButtonBreakpointExpressionUpdate.setEnabled(m_TextBreakpointExpression.getText().length() > 0);
    }

    /**
     * Continues the flow execution.
     */
    protected void continueFlowExecution() {
      // breakpoint condition updated?
      if (!m_TextBreakpointExpression.getText().equals(getExpression()))
	m_Expression = m_TextBreakpointExpression.getText();
      getFlow().resumeExecution();
      update();
    }

    /**
     * Stops the flow execution.
     */
    protected void stopFlowExecution() {
      Runnable	runnable;

      if (getParentDialog() != null)
	getParentDialog().setVisible(false);
      else if (getParentFrame() != null)
	getParentFrame().setVisible(false);

      runnable = new Runnable() {
	public void run() {
	  getFlow().stopExecution("User stopped flow!");
	}
      };
      SwingUtilities.invokeLater(runnable);
    }

    /**
     * Disable/enable the breakpoint.
     */
    protected void disableEnableBreakpoint() {
      Flow	flow;

      // cannot use setSkip(boolean) as this will reset the actor
      // and remove the control panel
      setDisabled(!isDisabled());
      if (isDisabled()) {
	m_ButtonDisableEnable.setText("Enable");
	m_ButtonDisableEnable.setMnemonic('E');
	m_ButtonDisableEnable.setToolTipText("Enable the breakpoint");
	m_ButtonDisableEnable.setIcon(GUIHelper.getIcon("debug.png"));
      }
      else {
	m_ButtonDisableEnable.setText("Disable");
	m_ButtonDisableEnable.setMnemonic('D');
	m_ButtonDisableEnable.setToolTipText("Disable the breakpoint");
	m_ButtonDisableEnable.setIcon(GUIHelper.getIcon("debug_off.png"));
      }

      flow = getFlow();
      if (flow.isPaused())
	flow.resumeExecution();

      update();
    }

    /**
     * Displays dialog with watch expressions.
     */
    protected void showWatchExpressions() {
      if (m_DialogExpressions == null) {
	m_PanelExpressions.setVariables(getVariables());
	if (getParentDialog() != null)
	  m_DialogExpressions = new BaseDialog(getParentDialog());
	else
	  m_DialogExpressions = new BaseDialog(getParentFrame());
	m_DialogExpressions.setTitle("Expression watch");
	m_DialogExpressions.setSize(600, 400);
	m_DialogExpressions.getContentPane().setLayout(new BorderLayout());
	m_DialogExpressions.getContentPane().add(m_PanelExpressions, BorderLayout.CENTER);
	m_DialogExpressions.pack();
	m_DialogExpressions.setLocationRelativeTo(null);
      }

      m_DialogExpressions.setVisible(true);
    }

    /**
     * Displays the source code (nested format) of the current flow.
     */
    protected void showSource() {
      String			content;
      DebugNestedProducer	producer;

      producer = new DebugNestedProducer();
      producer.produce(getFlow());
      content = producer.toString();
      producer.cleanUp();

      if (m_DialogSource == null) {
	if (getParentDialog() != null)
	  m_DialogSource = new TextDialog(getParentDialog());
	else
	  m_DialogSource = new TextDialog(getParentFrame());
	m_DialogSource.setTitle("Current state");
	m_DialogSource.setEditable(false);
	m_DialogSource.setTabSize(2);
	m_DialogSource.setLocationRelativeTo(null);
      }
      m_DialogSource.setContent(content);
      m_DialogSource.setVisible(true);
    }

    /**
     * Displays the current variables in the system.
     */
    protected void showVariables() {
      BaseDialog	dialog;

      if (m_PanelVariables == null) {
	m_PanelVariables = new VariableManagementPanel();
	m_PanelVariables.setVariables(getVariables());
	if (getParentDialog() != null)
	  dialog = new BaseDialog(getParentDialog());
	else
	  dialog = new BaseDialog(getParentFrame());
	dialog.setTitle("Variables");
	dialog.getContentPane().setLayout(new BorderLayout());
	dialog.getContentPane().add(m_PanelVariables, BorderLayout.CENTER);
	dialog.pack();
	dialog.setLocationRelativeTo(null);
      }
      m_PanelVariables.getParentDialog().setVisible(true);
    }

    /**
     * Inspects the current token.
     */
    protected void inspectToken() {
      BaseDialog	dialog;

      if (m_PanelInspectionToken == null) {
	m_PanelInspectionToken = new InspectionPanel();
	if (getParentDialog() != null)
	  dialog = new BaseDialog(getParentDialog());
	else
	  dialog = new BaseDialog(getParentFrame());
	dialog.setTitle("Current token");
	dialog.getContentPane().setLayout(new BorderLayout());
	dialog.getContentPane().add(m_PanelInspectionToken, BorderLayout.CENTER);
	dialog.setJMenuBar(m_PanelInspectionToken.getMenuBar());
	dialog.setSize(800, 500);
	dialog.setLocationRelativeTo(null);
      }
      m_PanelInspectionToken.setCurrent(m_CurrentToken);
      m_PanelInspectionToken.getParentDialog().setVisible(true);
    }

    /**
     * Inspects the current flow.
     */
    protected void inspectFlow() {
      BaseDialog	dialog;

      if (m_PanelInspectionFlow == null) {
	m_PanelInspectionFlow = new InspectionPanel();
	if (getParentDialog() != null)
	  dialog = new BaseDialog(getParentDialog());
	else
	  dialog = new BaseDialog(getParentFrame());
	dialog.setTitle("Current flow");
	dialog.getContentPane().setLayout(new BorderLayout());
	dialog.getContentPane().add(m_PanelInspectionFlow, BorderLayout.CENTER);
	dialog.setJMenuBar(m_PanelInspectionFlow.getMenuBar());
	dialog.setSize(800, 500);
	dialog.setLocationRelativeTo(null);
      }
      m_PanelInspectionFlow.setCurrent(m_Self.getRoot());
      m_PanelInspectionFlow.getParentDialog().setVisible(true);
    }

    /**
     * Shows the current temporary storage.
     */
    protected void showStorage() {
      BaseDialog	dialog;

      if (m_PanelStorage == null) {
	m_PanelStorage = new StoragePanel();
	if (getParentDialog() != null)
	  dialog = new BaseDialog(getParentDialog());
	else
	  dialog = new BaseDialog(getParentFrame());
	dialog.setTitle("Temporary storage");
	dialog.getContentPane().setLayout(new BorderLayout());
	dialog.getContentPane().add(m_PanelStorage, BorderLayout.CENTER);
	dialog.pack();
	dialog.setLocationRelativeTo(null);
      }
      m_PanelStorage.setHandler(getStorageHandler());
      m_PanelStorage.getParentDialog().setVisible(true);
    }

    /**
     * Called by actor when breakpoint reached.
     */
    public void breakpointReached() {
      m_PanelExpressions.refreshAllExpressions();
      m_TextBreakpointExpression.setText(getExpression());
      if (m_PanelInspectionToken != null)
	m_PanelInspectionToken.setCurrent(m_CurrentToken);
      if (m_PanelStorage != null)
	m_PanelStorage.setHandler(getStorageHandler());

      update();

      // show dialogs
      for (Dialog d: m_Dialogs) {
	switch (d) {
	  case SOURCE:
	    showSource();
	    break;
	  case EXPRESSIONS:
	    showWatchExpressions();
	    break;
	  case INSPECT_TOKEN:
	    inspectToken();
	    break;
	  case STORAGE:
	    showStorage();
	    break;
	  case VARIABLES:
	    showVariables();
	    break;
	  default:
	    throw new IllegalStateException("Unhandled dialog type: " + d);
	}
      }
    }

    /**
     * Updates the breakpoint condition.
     */
    public void updateExpression() {
      setExpression(m_TextBreakpointExpression.getText());
    }

    /**
     * Adds a new expression.
     *
     * @param expr	the expression
     * @param type	the type of the expression
     * @see		ExpressionWatchPanel#addExpression(String, ExpressionType)
     */
    public void addWatch(String expr, ExpressionType type) {
      m_PanelExpressions.addExpression(expr, type);
    }

    /**
     * Cleans up data structures, frees up memory.
     */
    public void cleanUp() {
      if (m_DialogSource != null) {
	m_DialogSource.dispose();
	m_DialogSource = null;
      }
      if (m_PanelExpressions != null) {
	m_PanelExpressions.closeParent();
	m_PanelExpressions = null;
	m_DialogExpressions = null;
      }
      if (m_PanelVariables != null) {
	m_PanelVariables.closeParent();
	m_PanelVariables = null;
      }
      if (m_PanelStorage != null) {
	m_PanelStorage.cleanUp();
	m_PanelStorage.closeParent();
	m_PanelStorage = null;
      }
      if (m_PanelInspectionToken != null) {
	m_PanelInspectionToken.closeParent();
	m_PanelInspectionToken = null;
      }
      if (m_PanelInspectionFlow != null) {
	m_PanelInspectionFlow.closeParent();
	m_PanelInspectionFlow = null;
      }
    }
  }

  /**
   * The dialogs available for the actor.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 3239 $
   */
  public enum Dialog {
    /** the source dialog. */
    SOURCE,
    /** the expressions dialog. */
    EXPRESSIONS,
    /** the variables dialog. */
    VARIABLES,
    /** the storage dialog. */
    STORAGE,
    /** the dialog for inspecting the current token. */
    INSPECT_TOKEN,
    /** the dialog for inspecting the current flow. */
    INSPECT_FLOW
  }

  /** the breakpoint expression to evaluate. */
  protected String m_Expression;

  /** the watch expressions. */
  protected BaseString[] m_Watches;

  /** the watch expression types. */
  protected ExpressionType[] m_WatchTypes;

  /** the X position of the dialog. */
  protected int m_X;

  /** the Y position of the dialog. */
  protected int m_Y;

  /** the panel to display. */
  protected BasePanel m_Panel;

  /** the dialog that's being displayed. */
  protected BaseFrame m_Frame;

  /** whether the breakpoint is disabled. */
  protected boolean m_Disabled;

  /** the current token. */
  protected transient Token m_CurrentToken;

  /** the dialogs to display automatically. */
  protected Dialog[] m_Dialogs;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Allows to pause the execution of the flow when this actor is reached "
      + "and the condition evaluates to 'true'.\n"
      + "It is possible to define watches as well.\n\n"
      + "The expression has the following underlying grammar:\n"
      + new BooleanExpression().getGrammar();
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "x", "x",
	    getDefaultX(), -3, null);

    m_OptionManager.add(
	    "y", "y",
	    getDefaultY(), -3, null);

    m_OptionManager.add(
	    "expression", "expression",
	    "true");

    m_OptionManager.add(
	    "watch", "watches",
	    new BaseString[0]);

    m_OptionManager.add(
	    "watch-type", "watchTypes",
	    new ExpressionType[0]);

    m_OptionManager.add(
	    "dialog", "dialogs",
	    new Dialog[0]);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;
    int		i;

    result = "";

    variable = getOptionManager().getVariableForProperty("expression");
    if (variable != null)
      result = variable;
    else if ((m_Expression != null) && (m_Expression.length() > 0))
      result = m_Expression;

    if (m_Watches.length > 0) {
      if (result.length() > 0)
	result += ", ";
      result += m_Watches.length + " watch";
      if (m_Watches.length > 1)
	result += "es";
    }

    if (m_Dialogs.length > 0) {
      result += ", dialog";
      if (m_Dialogs.length > 1)
	result += "s";
      result += ": ";
      for (i = 0; i < m_Dialogs.length; i++) {
	if (i > 0)
	  result += ", ";
	result += m_Dialogs[i];
      }
    }

    return result;
  }

  /**
   * Returns the default X position for the dialog.
   *
   * @return		the default X
   */
  protected int getDefaultX() {
    return -3;
  }

  /**
   * Returns the default Y position for the dialog.
   *
   * @return		the default Y position
   */
  protected int getDefaultY() {
    return -1;
  }

  /**
   * Sets the X position of the dialog.
   *
   * @param value 	the X position
   */
  public void setX(int value) {
    m_X = value;
    reset();
  }

  /**
   * Returns the currently set X position of the dialog.
   *
   * @return 		the X position
   */
  public int getX() {
    return m_X;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String xTipText() {
    return "The X position of the dialog (>=0: absolute, -1: left, -2: center, -3: right).";
  }

  /**
   * Sets the Y position of the dialog.
   *
   * @param value 	the Y position
   */
  public void setY(int value) {
    m_Y = value;
    reset();
  }

  /**
   * Returns the currently set Y position of the dialog.
   *
   * @return 		the Y position
   */
  public int getY() {
    return m_Y;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String yTipText() {
    return "The Y position of the dialog (>=0: absolute, -1: top, -2: center, -3: bottom).";
  }

  /**
   * Sets the expression to evaluate. Automatically wraps expressions in
   * parentheses that consists only of a variable. Otherwise, the expresssion
   * would get interpreted as attached variable for the expression option.
   *
   * @param value	the expression
   */
  public void setExpression(String value) {
    if (Variables.isPlaceholder(value))
      value = "(" + value + ")";
    m_Expression = value;
    reset();
  }

  /**
   * Returns the expression to evaluate.
   *
   * @return		the expression
   */
  public String getExpression() {
    return m_Expression;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String expressionTipText() {
    return
        "The expression to evaluate; if the expression evaluates to 'true', "
      + "the execution of the flow is paused and a control window is opened.";
  }

  /**
   * Sets the watch expressions for the watch dialog.
   *
   * @param value	the expressions
   */
  public void setWatches(BaseString[] value) {
    int		i;

    for (i = 0; i < value.length; i++) {
      if (Variables.isPlaceholder(value[i].getValue()))
	value[i] = new BaseString("(" + value[i].getValue() + ")");
    }

    m_Watches = value;
    reset();
  }

  /**
   * Returns the watch expressions for the watch dialog.
   *
   * @return		the expressions
   */
  public BaseString[] getWatches() {
    return m_Watches;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String watchesTipText() {
    return
        "The expression to display initially in the watch dialog; the type of "
      + "the watch needs to be specified as well.";
  }

  /**
   * Sets the types of the watch expressions.
   *
   * @param value	the types
   */
  public void setWatchTypes(ExpressionType[] value) {
    m_WatchTypes = value;
    reset();
  }

  /**
   * Returns the types of the watch expressions.
   *
   * @return		the types
   */
  public ExpressionType[] getWatchTypes() {
    return m_WatchTypes;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String watchTypesTipText() {
    return
        "The types of the watch expressions; determines how the expressions "
      + "get evaluated and displayed.";
  }

  /**
   * Sets the dialogs to display automatically.
   *
   * @param value	the dialogs
   */
  public void setDialogs(Dialog[] value) {
    m_Dialogs = value;
    reset();
  }

  /**
   * Returns the dialogs to display automatically.
   *
   * @return		the dialogs
   */
  public Dialog[] getDialogs() {
    return m_Dialogs;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String dialogsTipText() {
    return "The dialogs to display automatically when the breakpoint is reached.";
  }

  /**
   * Creates the panel to display in the dialog.
   *
   * @return		the panel
   */
  protected BasePanel newPanel() {
    ControlPanel	result;
    int			i;

    result = new ControlPanel();

    for (i = 0; i < m_Watches.length; i++)
      result.addWatch(m_Watches[i].getValue(), m_WatchTypes[i]);

    return result;
  }

  /**
   * Determines whether to execute the 'then' branch.
   *
   * @param input	the input token to use for evaluation
   * @return		true if the 'then' branch should get executed
   */
  protected boolean evalCondition(Token input) {
    boolean	result;
    String	exp;
    HashMap	symbols;

    exp     = getVariables().expand(m_Expression);
    symbols = new HashMap();
    if (input.getPayload() instanceof Integer)
      symbols.put("X", ((Integer) input.getPayload()).doubleValue());
    else if (input.getPayload() instanceof Double)
      symbols.put("X", ((Double) input.getPayload()).doubleValue());

    try {
      result = BooleanExpression.evaluate(exp, symbols);
      if (getDebugLevel() >= 2)
        debug(
            "exp: " + m_Expression + "\n"
            + "  --> " + exp + "\n"
            + "  = " + result, 2);
    }
    catch (Exception e) {
      getSystemErr().println("Error evaluating boolean expression: " + exp);
      getSystemErr().printStackTrace(e);
      m_Expression = "true";
      getSystemErr().println("Reseting expression to: " + m_Expression);
      if (m_Panel != null)
	((ControlPanel) m_Panel).updateExpression();
      result = false;
    }

    return result;
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null) {
      if (m_Watches.length != m_WatchTypes.length)
	Utils.adjustArray(m_WatchTypes, m_Watches.length, ExpressionType.STRING);
    }

    return result;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Unknown.class};
  }

  /**
   * Sets whether the breakpoint is disabled or not.
   *
   * @param value	if true then the breakpoint gets disabled
   */
  protected void setDisabled(boolean value) {
    m_Disabled = value;
  }

  /**
   * Returns whether the breakpoint is diabled or not.
   *
   * @return		true if the breakpoint is disabled
   */
  protected boolean isDisabled() {
    return m_Disabled;
  }

  /**
   * Creates the actual frame.
   *
   * @param panel	the panel to display in the frame
   * @return		the created frame
   */
  protected BaseFrame createFrame(BasePanel panel) {
    BaseFrame	result;
    ImageIcon	icon;

    result = new BaseFrame(getName());
    result.getContentPane().setLayout(new BorderLayout());
    result.getContentPane().add(panel, BorderLayout.CENTER);
    result.setDefaultCloseOperation(BaseDialog.HIDE_ON_CLOSE);
    icon = GUIHelper.getIcon(getClass());
    if (icon != null)
      result.setIconImage(icon.getImage());
    else
      result.setIconImage(GUIHelper.getIcon("flow.gif").getImage());
    if (panel instanceof MenuBarProvider)
      result.setJMenuBar(((MenuBarProvider) panel).getMenuBar());
    else if (this instanceof MenuBarProvider)
      result.setJMenuBar(((MenuBarProvider) this).getMenuBar());
    result.setLocation(ActorUtils.determineLocation(result, m_X, m_Y));
    result.pack();

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    Runnable	run;

    result = null;

    m_CurrentToken = m_InputToken;

    if (!isHeadless() && !isDisabled()) {
      if (evalCondition(m_CurrentToken))
	((Flow) getRoot()).pauseExecution();

      if (m_Panel == null) {
	m_Panel = newPanel();
	m_Frame = createFrame(m_Panel);
      }

      run = new Runnable() {
	public void run() {
	  if (!m_Frame.isVisible())
	    m_Frame.setVisible(true);
	  ((ControlPanel) m_Panel).breakpointReached();
	}
      };
      SwingUtilities.invokeLater(run);
    }

    m_OutputToken = m_CurrentToken;

    return result;
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.core.Unknown.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Unknown.class};
  }

  /**
   * Stops the execution. No message set.
   */
  public void stopExecution() {
    if ((m_Frame != null) && (m_Frame.isVisible()))
      m_Frame.setVisible(false);

    super.stopExecution();
  }

  /**
   * Cleans up after the execution has finished. Removes the control panel.
   */
  public void wrapUp() {
    m_CurrentToken = null;
    doCleanUpGUI();
    super.wrapUp();
  }

  /**
   * Removes all graphical components.
   */
  protected void cleanUpGUI() {
    if (m_Frame != null) {
      if (m_Panel instanceof CleanUpHandler)
	((CleanUpHandler) m_Panel).cleanUp();

      m_Frame.setVisible(false);
      m_Frame.dispose();

      m_Frame = null;
      m_Panel = null;
    }
  }

  /**
   * Queues the clean up of the GUI in the swing thread.
   *
   * @see		#cleanUpGUI()
   */
  protected void doCleanUpGUI() {
    Runnable	runnable;

    runnable = new Runnable() {
      public void run() {
	cleanUpGUI();
      }
    };
    SwingUtilities.invokeLater(runnable);
  }

  /**
   * Cleans up after the execution has finished. Also removes graphical
   * components.
   */
  public void cleanUp() {
    doCleanUpGUI();
    super.cleanUp();
  }
}
