/*
 * WhileLoop.java
 * Copyright (C) 2010-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;

import java.util.HashMap;

import adams.flow.core.AbstractActor;
import adams.flow.core.ActorHandlerInfo;
import adams.flow.core.InputConsumer;
import adams.flow.core.MutableActorHandler;
import adams.flow.core.Token;
import adams.flow.core.Unknown;
import adams.parser.BooleanExpression;
import adams.parser.GrammarSupplier;

/**
 <!-- globalinfo-start -->
 * Emulates a while-loop. The sub-actor gets executed as long as the expression evaluates to 'true'.<br/>
 * <br/>
 * The following grammar is used for evaluating the boolean expressions:<br/>
 * <br/>
 *  expr_list ::= expr_list expr_part | expr_part ;<br/>
 *  expr_part ::= boolexpr ;<br/>
 * <br/>
 *  boolexpr ::=    ( boolexpr )<br/>
 *                | true<br/>
 *                | false<br/>
 *                | numexpr &lt; numexpr<br/>
 *                | numexpr &lt;= numexpr<br/>
 *                | numexpr &gt; numexpr<br/>
 *                | numexpr &gt;= numexpr<br/>
 *                | numexpr = numexpr<br/>
 *                | numexpr != numexpr<br/>
 *                | str &lt; str<br/>
 *                | str &lt;= str<br/>
 *                | str &gt; str<br/>
 *                | str &gt;= str<br/>
 *                | str = str<br/>
 *                | str != str<br/>
 *                | ! boolexpr<br/>
 *                | boolexpr &amp; boolexpr<br/>
 *                | boolexpr | boolexpr<br/>
 *                ;<br/>
 * <br/>
 *  numexpr   ::=   ( numexpr )<br/>
 *                | NUMBER<br/>
 *                | -numexpr<br/>
 *                | constexpr<br/>
 *                | opexpr<br/>
 *                | varexpr<br/>
 *                | funcexpr<br/>
 *                ;<br/>
 * <br/>
 *  constexpr ::=   PI<br/>
 *                | E<br/>
 *                ;<br/>
 * <br/>
 *  opexpr    ::=   numexpr + numexpr<br/>
 *                | numexpr - numexpr<br/>
 *                | numexpr * numexpr<br/>
 *                | numexpr &#47; numexpr<br/>
 *                | numexpr ^ numexpr<br/>
 *                ;<br/>
 * <br/>
 *  varexpr  ::=  VARIABLE ;<br/>
 * <br/>
 *  funcexpr ::=    abs ( numexpr )<br/>
 *                | sqrt ( numexpr )<br/>
 *                | log ( numexpr )<br/>
 *                | exp ( numexpr )<br/>
 *                | sin ( numexpr )<br/>
 *                | cos ( numexpr )<br/>
 *                | tan ( numexpr )<br/>
 *                | rint ( numexpr )<br/>
 *                | floor ( numexpr )<br/>
 *                | pow ( numexpr , numexpr )<br/>
 *                | ceil ( numexpr )<br/>
 *                | ifelse ( boolexpr , numexpr (if true) , numexpr (if false) )<br/>
 *                | length ( str )<br/>
 *                ;<br/>
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: WhileLoop
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-expression &lt;java.lang.String&gt; (property: expression)
 * &nbsp;&nbsp;&nbsp;The expression to evaluate - only as long as it evaluates to 'true' the
 * &nbsp;&nbsp;&nbsp;loop actors get executed.
 * &nbsp;&nbsp;&nbsp;default: true
 * </pre>
 *
 * <pre>-loop &lt;adams.flow.core.AbstractActor&gt; [-loop ...] (property: actors)
 * &nbsp;&nbsp;&nbsp;The actors to execute in the loop.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3239 $
 */
public class WhileLoop
  extends AbstractControlActor
  implements InputConsumer, GrammarSupplier, MutableActorHandler {

  /** for serialization. */
  private static final long serialVersionUID = -2837014912083918343L;

  /** the expression to evaluate. */
  protected String m_Expression;

  /** the actors to execute. */
  protected Sequence m_Actors;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Emulates a while-loop. The sub-actor gets executed as long as the "
      + "expression evaluates to 'true'.\n"
      + "\n"
      + "The following grammar is used for evaluating the boolean expressions:\n\n"
      + getGrammar();
  }

  /**
   * Returns a string representation of the grammar.
   *
   * @return		the grammar, null if not available
   */
  public String getGrammar() {
    return new BooleanExpression().getGrammar();
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "expression", "expression",
	    "true");

    m_OptionManager.add(
	    "loop", "actors",
	    new AbstractActor[0]);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Actors = new Sequence();
  }

  /**
   * Sets the expression to evaluate.
   *
   * @param value	the expression
   */
  public void setExpression(String value) {
    m_Expression = value;
    reset();
  }

  /**
   * Returns the expression to evaluate.
   *
   * @return		the expression
   */
  public String getExpression() {
    return m_Expression;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String expressionTipText() {
    return
        "The expression to evaluate - only as long as it evaluates to 'true' "
      + "the loop actors get executed.";
  }

  /**
   * Sets the actors of the loop.
   *
   * @param value	the actors
   */
  public void setActors(AbstractActor[] value) {
    m_Actors.setActors(value);
    reset();
    updateParent();
  }

  /**
   * Returns the actors of the loop.
   *
   * @return		the actors
   */
  public AbstractActor[] getActors() {
    return m_Actors.getActors();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String actorsTipText() {
    return "The actors to execute in the loop.";
  }

  /**
   * Updates the parent of all actors in this group.
   */
  protected void updateParent() {
    m_Actors.setName(getName());
    m_Actors.setParent(null);
    m_Actors.setParent(getParent());
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("expression");

    if (variable != null)
      return variable;
    else if ((m_Expression != null) && (m_Expression.length() > 0))
      return m_Expression;
    else
      return null;
  }

  /**
   * Returns the size of the group.
   *
   * @return		always 1
   */
  public int size() {
    return m_Actors.size();
  }

  /**
   * Returns the actor at the given position.
   *
   * @param index	the position
   * @return		the actor
   */
  public AbstractActor get(int index) {
    return m_Actors.get(index);
  }

  /**
   * Sets the actor at the given position.
   *
   * @param index	the position
   * @param actor	the actor to set at this position
   */
  public void set(int index, AbstractActor actor) {
    m_Actors.set(index, actor);
    reset();
    updateParent();
  }

  /**
   * Returns the index of the actor.
   *
   * @param actor	the name of the actor to look for
   * @return		the index of -1 if not found
   */
  public int indexOf(String actor) {
    return m_Actors.indexOf(actor);
  }

  /**
   * Inserts the actor at the given position.
   *
   * @param index	the position
   * @param actor	the actor to insert
   */
  public void add(int index, AbstractActor actor) {
    m_Actors.add(index, actor);
    reset();
    updateParent();
  }

  /**
   * Removes the actor at the given position and returns the removed object.
   *
   * @param index	the position
   * @return		the removed actor
   */
  public AbstractActor remove(int index) {
    AbstractActor	result;

    result = m_Actors.remove(index);
    reset();

    return result;
  }

  /**
   * Removes all actors.
   */
  public void removeAll() {
    m_Actors.removeAll();
    reset();
  }

  /**
   * Returns some information about the actor handler, e.g., whether it can
   * contain standalones and the actor execution.
   *
   * @return		the info
   */
  public ActorHandlerInfo getActorHandlerInfo() {
    return m_Actors.getActorHandlerInfo();
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Unknown.class};
  }

  /**
   * Gets called in the setUp() method. Returns null if loop-actors are fine,
   * otherwise error message.
   *
   * @return		null if everything OK, otherwise error message
   */
  protected String setUpLoopActors() {
    return null;
  }

  /**
   * Performs the setUp of the sub-actors.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String setUpSubActors() {
    String	result;

    result = null;

    if (m_Actors.size() == 0)
      result = "No loop-actors provided!";

    if ((result == null) && (!getSkip())) {
      updateParent();
      result = setUpLoopActors();
      if (result == null)
	result = m_Actors.setUp();
    }

    return result;
  }

  /**
   * Initializes the sub-actors for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null) {
      if ((m_Expression == null) || (m_Expression.length() == 0))
	result = "No expression provided!";
    }

    return result;
  }

  /**
   * Does nothing.
   *
   * @param token	the token to accept and process
   */
  public void input(Token token) {
  }

  /**
   * Checks whether the loop should be executed.
   *
   * @return		true if the loop should be executed
   */
  protected boolean doLoop() {
    boolean	result;
    String	exp;
    HashMap	symbols;

    exp     = getVariables().expand(m_Expression);
    symbols = new HashMap();
    try {
      result = BooleanExpression.evaluate(exp, symbols);
    }
    catch (Exception e) {
      getSystemErr().println("Error evaluating boolean expression: " + exp);
      getSystemErr().printStackTrace(e);
      result = false;
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    String	msg;

    result = null;

    while (doLoop() && (result == null) && !isStopped()) {
      try {
	result = m_Actors.execute();
      }
      catch (Exception e) {
	msg    = m_Actors.getFullName() + " generated following exception: ";
	result = msg + e;
	getSystemErr().println(msg);
	getSystemErr().printStackTrace(e);
      }
    }

    if (result != null)
      m_Actors.handleError("execute", result);

    return result;
  }

  /**
   * Pauses the execution.
   */
  public void pauseExecution() {
    m_Actors.pauseExecution();
  }

  /**
   * Returns whether the object is currently paused.
   *
   * @return		true if object is paused
   */
  public boolean isPaused() {
    return m_Actors.isPaused();
  }

  /**
   * Resumes the execution.
   */
  public void resumeExecution() {
    m_Actors.resumeExecution();
  }

  /**
   * Stops the execution.
   */
  public void stopExecution() {
    m_Actors.stopExecution();

    super.stopExecution();
  }
}
