/*
 * FlowSetupWorker.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.setup;

import javax.swing.SwingWorker;

import adams.core.CleanUpHandler;
import adams.core.Pausable;
import adams.core.Stoppable;
import adams.db.DatabaseConnection;
import adams.event.FlowSetupStateEvent;
import adams.flow.FlowRunner;
import adams.flow.core.AbstractActor;

/**
 * A specialized SwingWorker class for executing FlowSetups.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3516 $
 */
public class FlowSetupWorker
  extends SwingWorker
  implements CleanUpHandler, Stoppable, Pausable {

  /** the underlying flowsetup. */
  protected FlowSetup m_Setup;

  /** for executing the flow. */
  protected FlowRunner m_FlowRunner;

  /** the last actor that was run. */
  protected AbstractActor m_LastActor;

  /**
   * Initializes the worker.
   *
   * @param setup	the underlying flowsetup
   */
  public FlowSetupWorker(FlowSetup setup) {
    super();

    m_Setup      = setup;
    m_FlowRunner = null;
    m_LastActor  = null;
  }

  /**
   * Returns the underlying flow setup.
   *
   * @return		the setup
   */
  public FlowSetup getSetup() {
    return m_Setup;
  }

  /**
   * Returns the actor that was executed.
   *
   * @return		the actor
   */
  public AbstractActor getLastActor() {
    return m_LastActor;
  }

  /**
   * Runs the flow in the background.
   *
   * @throws Exception	if something goes wrong
   * @return		the last error or null if none
   */
  protected Object doInBackground() throws Exception {
    m_LastActor  = null;
    m_FlowRunner = new FlowRunner();
    m_FlowRunner.setDatabaseConnection(DatabaseConnection.getSingleton());  // FIXME
    m_FlowRunner.setFile(m_Setup.getFile());
    m_FlowRunner.setHeadless(m_Setup.isHeadless());
    m_Setup.notifyFlowSetupStateChangeListeners(
	new FlowSetupStateEvent(m_Setup, FlowSetupStateEvent.Type.STARTED));
    String lastError = m_FlowRunner.execute();
    m_Setup.setLastError(lastError);
    return lastError;
  }

  /**
   * After executing the flow.
   */
  protected void done() {
    m_LastActor = m_FlowRunner.getLastActor();

    if (m_Setup.hasLastError())
      m_Setup.notifyFlowSetupStateChangeListeners(
	  new FlowSetupStateEvent(m_Setup, FlowSetupStateEvent.Type.ERROR));
    else
      m_Setup.notifyFlowSetupStateChangeListeners(
	  new FlowSetupStateEvent(m_Setup, FlowSetupStateEvent.Type.FINISHED));

    m_FlowRunner.setDatabaseConnection(null);
    m_FlowRunner.setFile(null);
    m_FlowRunner = null;
    m_Setup.finish();

    super.done();
  }

  /**
   * Pauses the execution.
   */
  public void pauseExecution() {
    if (m_FlowRunner != null)
      m_FlowRunner.pauseExecution();
  }

  /**
   * Returns whether the object is currently paused.
   *
   * @return		true if object is paused
   */
  public boolean isPaused() {
    if (m_FlowRunner != null)
      return m_FlowRunner.isPaused();
    else
      return false;
  }

  /**
   * Resumes the execution.
   */
  public void resumeExecution() {
    if (m_FlowRunner != null)
      m_FlowRunner.resumeExecution();
  }

  /**
   * Stops the flow execution.
   */
  public void stopExecution() {
    if (m_FlowRunner != null)
      m_FlowRunner.stopExecution();
  }

  /**
   * Cleans up data structures, frees up memory.
   */
  public void cleanUp() {
    m_LastActor  = null;
    m_Setup      = null;
    m_FlowRunner = null;
  }
}
