/*
 * Max.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import adams.data.statistics.StatUtils;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Returns the maximum value from a double/int array or the index of the maximum value.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Integer[]<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Double[]<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Integer<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Double<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 * &nbsp;&nbsp;&nbsp;If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Max
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-index (property: returnIndex)
 * &nbsp;&nbsp;&nbsp;If set to true, then the index of the maximum is returned instead of the
 * &nbsp;&nbsp;&nbsp;value.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2391 $
 */
public class Max
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = 2007764064808349672L;

  /** whether to return the index instead of the value. */
  protected boolean m_ReturnIndex;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Returns the maximum value from a double/int array or the index of the maximum value.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "index", "returnIndex",
	    false);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    if (m_ReturnIndex)
      return "Index";
    else
      return "Value";
  }

  /**
   * Sets whether to return the value or the index.
   *
   * @param value	true if to return the index, false to return value
   */
  public void setReturnIndex(boolean value) {
    m_ReturnIndex = value;
    reset();
  }

  /**
   * Returns whether to return the value or the index.
   *
   * @return		true the index is returned, false if the value is returned
   */
  public boolean getReturnIndex() {
    return m_ReturnIndex;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String returnIndexTipText() {
    return "If set to true, then the index of the maximum is returned instead of the value.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.Integer[].class, java.lang.Double[].class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Integer[].class, Double[].class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.Integer.class, java.lang.Double.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Integer.class, Double.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    Double[]	doubles;
    Integer[]	integers;

    result = null;

    try {
      if (m_InputToken.getPayload() instanceof Double[]) {
	doubles = (Double[]) m_InputToken.getPayload();
	if (m_ReturnIndex)
	  m_OutputToken = new Token(new Integer(StatUtils.maxIndex(doubles)));
	else
	  m_OutputToken = new Token((Double) (StatUtils.max(doubles)));
      }
      else {
	integers = (Integer[]) m_InputToken.getPayload();
	if (m_ReturnIndex)
	  m_OutputToken = new Token(new Integer(StatUtils.maxIndex(integers)));
	else
	  m_OutputToken = new Token((Integer) (StatUtils.max(integers)));
      }
    }
    catch (Exception e) {
      m_OutputToken = null;
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    return result;
  }
}
