/**
 * ActorExecutionClassProducer.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core.option;

import java.util.List;
import java.util.Vector;

import adams.env.Environment;
import adams.flow.core.AbstractActor;

/**
 * Generates a wrapper class for executing an actor.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2759 $
 */
public class ActorExecutionClassProducer
  extends AbstractActorJavaCodeProducer {

  /** for serialization. */
  private static final long serialVersionUID = -4862444496437534455L;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Generates a wrapper class for executing an actor.";
  }

  /**
   * Sets the simple name (without package) of the class to generate.
   *
   * @param value	the simple name
   */
  public void setSimpleName(String value) {
    super.setSimpleName(value);
  }

  /**
   * Returns the simple name to use.
   *
   * @return		the simple name
   */
  public String getSimpleName() {
    return super.getSimpleName();
  }

  /**
   * Sets the package name of the class to generate.
   *
   * @param value	the package name
   */
  public void setPackage(String value) {
    super.setPackage(value);
  }

  /**
   * Returns the package name to use.
   *
   * @return		the package name
   */
  public String getPackage() {
    return super.getPackage();
  }

  /**
   * Returns the indentation for code inside the try-catch-block.
   *
   * @return		the indentation string
   */
  protected String getIndentation() {
    return "    ";
  }

  /**
   * Returns other necessary imports.
   *
   * @return		the class names
   */
  protected List<String> getRequiredImports() {
    Vector<String>	result;

    result = new Vector<String>(super.getRequiredImports());

    result.add(AbstractActor.class.getName());

    return result;
  }

  /**
   * Adds the Javadoc for the class.
   */
  protected void addClassJavadoc() {
    m_OutputBuffer.append("/**\n");
    m_OutputBuffer.append(" * Wrapper class for executing an actor." + "\n");
    m_OutputBuffer.append(" *\n");
    m_OutputBuffer.append(" * @author " + System.getProperty("user.name") + "\n");
    m_OutputBuffer.append(" * @author " + getClass().getName() + " (code generator)" + "\n");
    m_OutputBuffer.append(" * @version $" + "Revision" + "$\n");
    m_OutputBuffer.append(" */\n");
  }

  /**
   * Adds a method that gets executed just prior to the actors setup and
   * execution.
   */
  protected void addPreExecuteMethod() {
    m_OutputBuffer.append("  /**\n");
    m_OutputBuffer.append("   * Hook method before the actor is executed.\n");
    m_OutputBuffer.append("   * <p/>\n");
    m_OutputBuffer.append("   * Default implementation does nothing.\n");
    m_OutputBuffer.append("   *\n");
    m_OutputBuffer.append("   * @param actor 		the actor that will get executed.\n");
    m_OutputBuffer.append("   * @throws Exception 	if an error occurs.\n");
    m_OutputBuffer.append("   */\n");
    m_OutputBuffer.append("  protected void preExecute(AbstractActor actor) throws Exception {\n");
    m_OutputBuffer.append("  }\n");
    m_OutputBuffer.append("\n");
  }

  /**
   * Adds a method that executes the actor.
   */
  protected void addDoExecuteMethod() {
    m_OutputBuffer.append("  /**\n");
    m_OutputBuffer.append("   * Performs the actual execution of the actor.\n");
    m_OutputBuffer.append("   *\n");
    m_OutputBuffer.append("   * @param actor 		the actor that will get executed.\n");
    m_OutputBuffer.append("   * @throws Exception 	if an error occurs.\n");
    m_OutputBuffer.append("   */\n");
    m_OutputBuffer.append("  protected void doExecute(AbstractActor actor) throws Exception {\n");
    m_OutputBuffer.append("    String result = actor.setUp();\n");
    m_OutputBuffer.append("    if (result != null)\n");
    m_OutputBuffer.append("      throw new Exception(\"Setting up of actor failed: \" + result);\n");
    m_OutputBuffer.append("    result = actor.execute();\n");
    m_OutputBuffer.append("    if (result != null)\n");
    m_OutputBuffer.append("      throw new Exception(\"Execution of actor failed: \" + result);\n");
    m_OutputBuffer.append("    actor.wrapUp();\n");
    m_OutputBuffer.append("    // the following call closes all graphical output (like dialogs and frames)\n");
    m_OutputBuffer.append("    // and frees up memory\n");
    m_OutputBuffer.append("    actor.cleanUp();\n");
    m_OutputBuffer.append("  }\n");
    m_OutputBuffer.append("\n");
  }

  /**
   * Adds a method that gets executed just after to the actors execution
   * and cleanUp.
   */
  protected void addPostExecuteMethod() {
    m_OutputBuffer.append("  /**\n");
    m_OutputBuffer.append("   * Hook method after the actor was executed.\n");
    m_OutputBuffer.append("   * <p/>\n");
    m_OutputBuffer.append("   * Default implementation does nothing.\n");
    m_OutputBuffer.append("   *\n");
    m_OutputBuffer.append("   * @param actor 		the actor that got executed.\n");
    m_OutputBuffer.append("   * @throws Exception 	if an error occurs.\n");
    m_OutputBuffer.append("   */\n");
    m_OutputBuffer.append("  protected void postExecute(AbstractActor actor) throws Exception {\n");
    m_OutputBuffer.append("  }\n");
    m_OutputBuffer.append("\n");
  }

  /**
   * Hook-method for adding additional methods necessary to make the class
   * compile.
   */
  protected void addAdditionalMethods() {
    // preExecute
    addPreExecuteMethod();
    // doExecute
    addDoExecuteMethod();
    // postExecute
    addPostExecuteMethod();
    // actual execute method
    m_OutputBuffer.append("  /**\n");
    m_OutputBuffer.append("   * Executes the actor.\n");
    m_OutputBuffer.append("   *\n");
    m_OutputBuffer.append("   * @throws Exception 	if an error occurs.\n");
    m_OutputBuffer.append("   */\n");
    m_OutputBuffer.append("  public void execute() throws Exception {\n");
    m_OutputBuffer.append("    AbstractActor actor = getActor();\n");
    m_OutputBuffer.append("    preExecute(actor);\n");
    m_OutputBuffer.append("    doExecute(actor);\n");
    m_OutputBuffer.append("    postExecute(actor);\n");
    m_OutputBuffer.append("  }\n");
    m_OutputBuffer.append("\n");
  }

  /**
   * Adds the method that encloses the generated code.
   */
  protected void addMethodStart() {
    // getActor - start
    m_OutputBuffer.append("  /**\n");
    m_OutputBuffer.append("   * Used to create an instance of a specific actor.\n");
    m_OutputBuffer.append("   *\n");
    m_OutputBuffer.append("   * @return a suitably configured <code>AbstractActor</code> value\n");
    m_OutputBuffer.append("   * @throws Exception if set up fails\n");
    m_OutputBuffer.append("   */\n");
    m_OutputBuffer.append("  public AbstractActor getActor() throws Exception {\n");
    m_OutputBuffer.append("    AbstractArgumentOption    argOption;\n");
    m_OutputBuffer.append("    \n");
    m_OutputBuffer.append("    " + m_Input.getClass().getName() + " " + getOuterVariableName() + " = new " + m_Input.getClass().getName() + "();\n");
    m_OutputBuffer.append("    \n");
  }

  /**
   * Closes the method with the generated code.
   */
  protected void addMethodEnd() {
    // getActor - end
    if (m_OutputBuffer.charAt(m_OutputBuffer.length() - 1) != '\n')
      m_OutputBuffer.append("    \n");
    m_OutputBuffer.append("    return " + getOuterVariableName() + ";\n");
    m_OutputBuffer.append("  }\n");
  }

  /**
   * Hook method for adding a main method.
   */
  protected void addMainMethod() {
    String	inst;

    m_OutputBuffer.append("\n");
    m_OutputBuffer.append("  /**\n");
    m_OutputBuffer.append("   * Executes the actor, when started from commandline.\n");
    m_OutputBuffer.append("   *\n");
    m_OutputBuffer.append("   * @param args ignored\n");
    m_OutputBuffer.append("   * @throws Exception if execution fails\n");
    m_OutputBuffer.append("   */\n");
    m_OutputBuffer.append("  public static void main(String[] args) throws Exception {\n");
    m_OutputBuffer.append("    Environment.setEnvironmentClass(" + Environment.getEnvironmentClass().getName() + ".class);\n");
    inst = m_SimpleName.toLowerCase() + "Inst";
    m_OutputBuffer.append("    " + m_SimpleName + " " + inst + " = new " + m_SimpleName + "();\n");
    m_OutputBuffer.append("    " + inst + ".execute();\n");
    m_OutputBuffer.append("  }\n");
  }
}
