/*
 * NormalDistribution.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.data.fit;

import adams.data.statistics.Percentile;
import adams.data.statistics.StatUtils;

/**
 <!-- globalinfo-start -->
 * Fits the normal distribution to the data.<br/>
 * <br/>
 * For derivatives, see:<br/>
 * http:&#47;&#47;www.numberempire.com&#47;derivatives.php
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 * &nbsp;&nbsp;&nbsp;If set to true, function may output additional info to the console.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2568 $
 */
public class NormalDistribution
  extends NonlinearFunction
  implements InitialParameterGuesser {

  /** for serialization. */
  private static final long serialVersionUID = 1670768051070643034L;

  /**
   * Returns a short description of the function.
   *
   * @return		a description of the function
   */
  public String globalInfo() {
    return
        "Fits the normal distribution to the data.\n"
      + "\n"
      + "For derivatives, see:\n"
      + "http://www.numberempire.com/derivatives.php";
  }

  /**
   * Calculates the partial derivatives based on the x value and the
   * coefficients.
   * For derivatives, see <a href="http://www.numberempire.com/derivatives.php" target="_blank">Number Empire</a>.
   *
   * @param x		the x value
   * @param a		the mean and the stdev
   * @return		the partial derivatives
   */
  public double[] calcDerivatives(double x, double[] a) {
    double[]	result;
    double	m;
    double	s;

    result = new double[2];

    if (a.length != result.length)
      throw new IllegalStateException(
	  "Number of coefficients differs from number of partial derivatives: "
	  + a.length + " != " + result.length);
    m = a[0];
    s = a[1];

    result[0] = (x-m)*Math.exp(-((x*x-2*m*x+m*m)/(2*s*s)))/(Math.sqrt(2)*Math.sqrt(Math.PI)*s*s*Math.abs(s));   // (x-m)*%e^-((x^2-2*m*x+m^2)/(2*s^2))/(sqrt(2)*sqrt(%pi)*s^2*abs(s))
    result[1] = (Math.sqrt(2)*x*x-Math.pow(2,(3/2))*m*x-Math.sqrt(2)*s*s+Math.sqrt(2)*m*m)*Math.exp(-((x*x-2*m*x+m*m)/(2*s*s)))/(2*Math.sqrt(Math.PI)*s*s*s*Math.abs(s));   // (sqrt(2)*x^2-2^(3/2)*m*x-sqrt(2)*s^2+sqrt(2)*m^2)*%e^-((x^2-2*m*x+m^2)/(2*s^2))/(2*sqrt(%pi)*s^3*abs(s))

    return result;
  }

  /**
   * Calculates the y value based on the x value and the coefficients.
   *
   * @param x		the x value
   * @param a		the mean and the stdev
   * @return		the y value
   */
  public double calcY(double x, double[] a) {
    double	m;
    double	s;

    m = a[0];
    s = a[1];

    return 1.0 / Math.sqrt(2*Math.PI*s*s) * Math.exp(-(x-m)*(x-m)/(2*s*s));
  }

  /**
   * Returns the guessed parameters based on the input data.
   * The guess for the mean is the median of the y values and the guess for
   * the stdev is based on the InterQuartileRange: stdev = (75% - 25%) / 1.349.
   *
   * @param x		the x values
   * @param y		the y values
   * @return		the guesses for the initial parameters; mean and stdev
   */
  public double[] guess(double[] x, double[] y) {
    double[]		result;
    int			i;
    Percentile<Double> 	perc;
    double		perc25;
    double		perc75;

    if (x.length != y.length)
      throw new IllegalArgumentException(
	  "Array lengths of x and y differ: " + x.length + " != " + y.length);

    result = new double[2];

    // mean guess (we use the median for this)
    result[0] = StatUtils.median(y);

    // stdev guess (based on InterQuartileRange)
    // IQR = 1.349 * stdev; stdev = IQR / 1.349
    // http://en.wikipedia.org/wiki/Interquartile_range#Interquartile_range_of_distributions
    perc = new Percentile<Double>();
    for (i = 0; i < y.length; i++)
      perc.add(y[i]);
    perc25    = perc.getPercentile(0.25);
    perc75    = perc.getPercentile(0.75);
    result[1] = (perc75 - perc25) / 1.349;

    return result;
  }
}
