/*
 * AbstractSimpleCSVReportReader.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.data.io.input;

import java.util.Vector;

import adams.core.io.CsvSpreadSheetReader;
import adams.core.io.SpreadSheet;
import adams.core.io.SpreadSheet.Row;
import adams.core.option.AbstractOption;
import adams.data.report.DataType;
import adams.data.report.Field;
import adams.data.report.Report;

/**
 * Abstract ancestor for reports to be written in CSV format.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2320 $
 * @param <T> the type of report to use
 */
public abstract class AbstractSimpleCSVReportReader<T extends Report>
  extends AbstractReportReader<T> {

  /** for serialization. */
  private static final long serialVersionUID = 8997935127278404969L;

  /** the file extensions. */
  public final static String FILE_EXTENSION = "csv";

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Reads a report file in CSV file format.";
  }

  /**
   * Returns a string describing the format (used in the file chooser).
   *
   * @return 			a description suitable for displaying in the
   * 				file chooser
   */
  public String getFormatDescription() {
    return "Simple report CSV file format";
  }

  /**
   * Returns the extension(s) of the format.
   *
   * @return 			the extension(s) (without the dot!)
   */
  public String[] getFormatExtensions() {
    return new String[]{FILE_EXTENSION};
  }

  /**
   * Tries to determine the parent ID for the current report.
   *
   * @param report	the report to determine the ID for
   * @return		the parent database ID, -1 if it cannot be determined
   */
  protected int determineParentID(Report report) {
    return report.getDatabaseID();
  }

  /**
   * Performs the actual reading.
   *
   * @return		the reports that were read
   */
  protected Vector<T> readData() {
    Vector<T>		result;
    SpreadSheet		sheet;
    Row			row;
    int			i;
    DataType		type;
    String		value;
    Field		field;
    String		fieldStr;
    String		colField;
    String		colType;
    String		colValue;

    result = new Vector<T>();
    result.add(newInstance());
    result.firstElement().setDebugLevel(getDebugLevel());
    sheet    = new CsvSpreadSheetReader().read(getInput().getAbsolutePath());
    colField = sheet.getHeaderRow().getCellKey(0);
    colType  = sheet.getHeaderRow().getCellKey(1);
    colValue = sheet.getHeaderRow().getCellKey(2);
    for (i = 0; i < sheet.getRowCount(); i++) {
      row      = sheet.getRow(i);
      fieldStr = row.getCell(colField).getContent();
      type     = (DataType) DataType.valueOf((AbstractOption) null, row.getCell(colType).getContent());
      value    = row.getCell(colValue).getContent();
      if (fieldStr.equals(Report.PROPERTY_PARENTID)) {
	result.firstElement().setDatabaseID(Integer.parseInt(value));
      }
      else {
	field = new Field(fieldStr, type);
	result.firstElement().setValue(field, value);
      }
    }

    return result;
  }
}
