/**
 * PlotContainerUpdater.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.control;

import adams.flow.container.SequencePlotterContainer;
import adams.flow.core.Compatibility;
import adams.flow.core.InputConsumer;
import adams.flow.core.OutputProducer;

/**
 <!-- globalinfo-start -->
 * Applies all sub-actors to process either the name, the X or the Y value of the plot container.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.container.SequencePlotterContainer<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.container.SequencePlotterContainer<br/>
 * <p/>
 * Container information:<br/>
 * - adams.flow.container.SequencePlotterContainer: PlotName, X, Y, IsMarker<br/>
 * - adams.flow.container.SequencePlotterContainer: PlotName, X, Y, IsMarker
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: PlotContainerUpdater
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-actor &lt;adams.flow.core.AbstractActor&gt; [-actor ...] (property: actors)
 * &nbsp;&nbsp;&nbsp;All the actors that define this sequence.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-container-value &lt;PLOT_NAME|X_VALUE|Y_VALUE&gt; (property: containerValue)
 * &nbsp;&nbsp;&nbsp;The type of value to update.
 * &nbsp;&nbsp;&nbsp;default: Y_VALUE
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3964 $
 */
public class PlotContainerUpdater
  extends AbstractContainerUpdater {

  /** for serialization. */
  private static final long serialVersionUID = 7140175689043000123L;

  /**
   * The values of the plot container that can be modified.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 3964 $
   */
  public enum PlotContainerValue {
    /** the name of the container. */
    PLOT_NAME,
    /** the X value. */
    X_VALUE,
    /** the Y value. */
    Y_VALUE
  }

  /** the value to modify. */
  protected PlotContainerValue m_ContainerValue;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Applies all sub-actors to process either the name, the X or the Y value of the plot container.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "container-value", "containerValue",
	    PlotContainerValue.Y_VALUE);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("containerValue");
    if (variable != null)
      result = variable;
    else
      result = "" + m_ContainerValue;

    return result;
  }

  /**
   * Sets the type of value to update.
   *
   * @param value	the type of value
   */
  public void setContainerValue(PlotContainerValue value) {
    switch (value) {
      case PLOT_NAME:
	setContainerValueName(SequencePlotterContainer.VALUE_PLOTNAME);
	break;
      case X_VALUE:
	setContainerValueName(SequencePlotterContainer.VALUE_X);
	break;
      case Y_VALUE:
	setContainerValueName(SequencePlotterContainer.VALUE_Y);
	break;
      default:
	throw new IllegalArgumentException("Unhandled container value: " + value);
    }
    m_ContainerValue = value;
    reset();
  }

  /**
   * Returns the type of value to update.
   *
   * @return		the type of value
   */
  public PlotContainerValue getContainerValue() {
    return m_ContainerValue;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String containerValueTipText() {
    return "The type of value to update.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		the Class of objects that can be processed
   */
  public Class[] accepts() {
    return new Class[]{SequencePlotterContainer.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		the Class of the generated tokens
   */
  public Class[] generates() {
    return new Class[]{SequencePlotterContainer.class};
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String		result;
    Compatibility	comp;
    Class[]		cls;

    result = super.setUp();

    if (result == null) {
      if (m_ContainerValue == PlotContainerValue.PLOT_NAME)
	cls = new Class[]{String.class};
      else
	cls = new Class[]{Double.class};
      comp = new Compatibility();
      if (!comp.isCompatible(((InputConsumer) firstActive()).accepts(), cls)) {
	result = "First active sub-actor does not accept " + cls[0].getName();
      }
      else {
	if (!comp.isCompatible(((OutputProducer) lastActive()).generates(), cls))
	  result = "Last active sub-actor does not generate " + cls[0].getName();
      }
    }

    return result;
  }
}
