/*
 * Sort.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.HashSet;

import adams.core.Utils;
import adams.flow.core.Token;
import adams.flow.core.Unknown;

/**
 <!-- globalinfo-start -->
 * Actor for sorting arrays.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown[]<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown[]<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Sort
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-reverse (property: reverse)
 * &nbsp;&nbsp;&nbsp;If set to true, then the sorting will be reversed.
 * </pre>
 *
 * <pre>-unique (property: unique)
 * &nbsp;&nbsp;&nbsp;If set to true, then only unique entries will be output.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3319 $
 */
public class Sort
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -1405106607250617855L;

  /** reverse sorting. */
  protected boolean m_Reverse;

  /** unqiue entries. */
  protected boolean m_Unique;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Actor for sorting arrays.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "reverse", "reverse",
	    false);

    m_OptionManager.add(
	    "unique", "unique",
	    false);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;

    result = "";
    if (m_Reverse)
      result += "order reversed";
    if (m_Unique) {
      if (result.length() > 0)
	result += ", ";
      result += "unique";
    }

    if (result.length() == 0)
      return null;
    else
      return result;
  }

  /**
   * Sets whether to reverse the sorting.
   *
   * @param value	if true then the sorting will be reversed
   */
  public void setReverse(boolean value) {
    m_Reverse = value;
    reset();
  }

  /**
   * Returns whether the sorting is reversed.
   *
   * @return		true if reversed
   */
  public boolean getReverse() {
    return m_Reverse;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String reverseTipText() {
    return "If set to true, then the sorting will be reversed.";
  }

  /**
   * Sets whether to allow only unique entries.
   *
   * @param value	if true then only unique entries are output
   */
  public void setUnique(boolean value) {
    m_Unique = value;
    reset();
  }

  /**
   * Returns whether only unique entries are output.
   *
   * @return		true if only unique entries are output
   */
  public boolean getUnique() {
    return m_Unique;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String uniqueTipText() {
    return "If set to true, then only unique entries will be output.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown[].class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Unknown[].class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.core.Unknown[].class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Unknown[].class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    Object[]	obj;
    Object[]	newObj;
    HashSet	entries;
    int		i;

    result = null;

    if (!m_InputToken.getPayload().getClass().isArray()) {
      getSystemErr().println("Input is not an array - ignored!");
      m_OutputToken = new Token(m_InputToken.getPayload());
    }
    else {
      obj = (Object[]) Utils.deepCopy(m_InputToken.getPayload());
      if (isDebugOn())
	debug("unsorted: " + Utils.arrayToString(obj));

      // no duplicates?
      if (m_Unique) {
	entries = new HashSet(Arrays.asList(obj));
	obj     = (Object[]) Array.newInstance(obj.getClass().getComponentType(), entries.size());
	i       = 0;
	for (Object o: entries) {
	  Array.set(obj, i, o);
	  i++;
	}
	if (isDebugOn())
	  debug("unique: " + Utils.arrayToString(obj));
      }

      // sort
      Arrays.sort(obj);
      if (!m_Reverse) {
	newObj = obj;
      }
      else {
	newObj = (Object[]) Array.newInstance(obj.getClass().getComponentType(), obj.length);
	for (i = 0; i < obj.length; i++)
	  newObj[i] = obj[obj.length - i - 1];
      }
      if (isDebugOn())
	debug("sorted: " + Utils.arrayToString(newObj));

      m_OutputToken = new Token(newObj);
    }

    return result;
  }
}
